/*
 * Copyright (c) 2019, 2022, Oracle and/or its affiliates.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided
 * with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written
 * permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.oracle.truffle.llvm.parser;

import com.oracle.truffle.llvm.parser.model.functions.FunctionSymbol;
import com.oracle.truffle.llvm.parser.model.symbols.globals.GlobalVariable;
import com.oracle.truffle.llvm.parser.model.target.TargetTriple;
import com.oracle.truffle.llvm.runtime.datalayout.DataLayout;

import java.util.List;

public final class LLVMParserResult {

    private final LLVMParserRuntime runtime;
    private final List<FunctionSymbol> definedFunctions;
    private final List<FunctionSymbol> externalFunctions;
    private final List<GlobalVariable> definedGlobals;
    private final List<GlobalVariable> externalGlobals;
    private final List<GlobalVariable> threadLocalGlobals;
    private final int threadLocalGlobalObjectCounter;
    private final DataLayout dataLayout;
    private final int symbolTableSize;
    private final TargetTriple targetTriple;

    LLVMParserResult(LLVMParserRuntime runtime,
                    List<FunctionSymbol> definedFunctions,
                    List<FunctionSymbol> externalFunctions,
                    List<GlobalVariable> definedGlobals,
                    List<GlobalVariable> externalGlobals,
                    List<GlobalVariable> threadLocalGlobals,
                    int threadLocalGlobalObjectCounter,
                    DataLayout dataLayout,
                    TargetTriple targetTriple,
                    int totalSize) {
        this.runtime = runtime;
        this.definedFunctions = definedFunctions;
        this.externalFunctions = externalFunctions;
        this.threadLocalGlobals = threadLocalGlobals;
        this.threadLocalGlobalObjectCounter = threadLocalGlobalObjectCounter;
        this.definedGlobals = definedGlobals;
        this.externalGlobals = externalGlobals;
        this.dataLayout = dataLayout;
        this.targetTriple = targetTriple;
        assert totalSize >= definedFunctions.size() + externalFunctions.size() + definedGlobals.size() + externalGlobals.size() + threadLocalGlobals.size();
        this.symbolTableSize = totalSize;
    }

    public LLVMParserRuntime getRuntime() {
        return runtime;
    }

    public List<FunctionSymbol> getDefinedFunctions() {
        return definedFunctions;
    }

    public List<FunctionSymbol> getExternalFunctions() {
        return externalFunctions;
    }

    public List<GlobalVariable> getDefinedGlobals() {
        return definedGlobals;
    }

    public List<GlobalVariable> getExternalGlobals() {
        return externalGlobals;
    }

    public List<GlobalVariable> getThreadLocalGlobals() {
        return threadLocalGlobals;
    }

    public int getThreadLocalGlobalObjectCounter() {
        return threadLocalGlobalObjectCounter;
    }

    public DataLayout getDataLayout() {
        return dataLayout;
    }

    @Override
    public String toString() {
        return "LLVMParserResult[" + runtime.getLibraryName() + "]";
    }

    public int getSymbolTableSize() {
        return symbolTableSize;
    }

    public TargetTriple getTargetTriple() {
        return targetTriple;
    }
}
