/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.graal.pointsto.reports;

import com.oracle.graal.pointsto.BigBang;
import com.oracle.graal.pointsto.meta.AnalysisMethod;
import com.oracle.graal.pointsto.meta.InvokeInfo;
import com.oracle.graal.pointsto.reports.AnalysisReportsOptions;
import com.oracle.graal.pointsto.reports.ReportUtils;
import com.oracle.graal.pointsto.util.AnalysisError;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import jdk.vm.ci.code.BytecodePosition;
import jdk.vm.ci.meta.JavaKind;
import jdk.vm.ci.meta.ResolvedJavaMethod;
import jdk.vm.ci.meta.ResolvedJavaType;

public final class CallTreePrinter {
    public static final Pattern CAMEL_CASE_PATTERN = Pattern.compile("\\b[a-zA-Z]|[A-Z]|\\.");
    private final BigBang bb;
    private final Map<AnalysisMethod, MethodNode> methodToNode;
    private static final String METHOD_FORMAT = "%H.%n(%P):%R";

    public static void print(BigBang bb, String reportsPath, String reportName) {
        CallTreePrinter printer = new CallTreePrinter(bb);
        printer.buildCallTree();
        AnalysisReportsOptions.CallTreeType optionValue = (AnalysisReportsOptions.CallTreeType)((Object)AnalysisReportsOptions.PrintAnalysisCallTreeType.getValue(bb.getOptions()));
        switch (optionValue) {
            case TXT: {
                ReportUtils.report("call tree", reportsPath, "call_tree_" + reportName, "txt", printer::printMethods);
                break;
            }
            case CSV: {
                CallTreePrinter.printCsvFiles(printer.methodToNode, reportsPath, reportName);
                break;
            }
            default: {
                throw AnalysisError.shouldNotReachHere("Unsupported CallTreeType " + optionValue + " used with PrintAnalysisCallTreeType option");
            }
        }
        ReportUtils.report("list of used methods", reportsPath, "used_methods_" + reportName, "txt", printer::printUsedMethods);
        ReportUtils.report("list of used classes", reportsPath, "used_classes_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, false));
        ReportUtils.report("list of used packages", reportsPath, "used_packages_" + reportName, "txt", writer -> printer.printClasses((PrintWriter)writer, true));
    }

    public CallTreePrinter(BigBang bb) {
        this.bb = bb;
        this.methodToNode = new LinkedHashMap<AnalysisMethod, MethodNode>();
    }

    public void buildCallTree() {
        ArrayList<ResolvedJavaMethod> roots = new ArrayList<ResolvedJavaMethod>();
        for (AnalysisMethod analysisMethod : this.bb.getUniverse().getMethods()) {
            if (analysisMethod.isDirectRootMethod() && analysisMethod.isImplementationInvoked()) {
                roots.add(analysisMethod);
            }
            if (!analysisMethod.isVirtualRootMethod()) continue;
            for (AnalysisMethod impl : analysisMethod.getImplementations()) {
                AnalysisError.guarantee(impl.isImplementationInvoked());
                roots.add(impl);
            }
        }
        roots.sort(ReportUtils.methodComparator);
        for (AnalysisMethod analysisMethod : roots) {
            this.methodToNode.put(analysisMethod, new MethodNode(analysisMethod, true));
        }
        ArrayDeque<MethodNode> workList = new ArrayDeque<MethodNode>(this.methodToNode.values());
        while (!workList.isEmpty()) {
            MethodNode methodNode = workList.removeFirst();
            ArrayList<InvokeInfo> invokeInfos = new ArrayList<InvokeInfo>();
            for (InvokeInfo invokeInfo : methodNode.method.getInvokes()) {
                invokeInfos.add(invokeInfo);
            }
            invokeInfos.sort(ReportUtils.invokeInfoComparator);
            for (InvokeInfo invokeInfo : invokeInfos) {
                this.processInvoke(invokeInfo, methodNode, workList);
            }
        }
    }

    private void processInvoke(InvokeInfo invokeInfo, MethodNode callerNode, Deque<MethodNode> workList) {
        InvokeNode invokeNode = new InvokeNode(invokeInfo.getTargetMethod(), invokeInfo.isDirectInvoke(), CallTreePrinter.sourceReference(invokeInfo.getPosition()));
        callerNode.addInvoke(invokeNode);
        invokeInfo.getCallees().stream().sorted(ReportUtils.methodComparator).forEach(callee -> {
            if (this.methodToNode.containsKey(callee)) {
                MethodNodeReference calleeNode = new MethodNodeReference(this.methodToNode.get(callee));
                invokeNode.addCallee(calleeNode);
            } else {
                MethodNode calleeNode = new MethodNode((AnalysisMethod)callee);
                invokeNode.addCallee(calleeNode);
                this.methodToNode.put((AnalysisMethod)callee, calleeNode);
                workList.add(calleeNode);
            }
        });
    }

    private static SourceReference[] sourceReference(BytecodePosition position) {
        ArrayList<SourceReference> sourceReference = new ArrayList<SourceReference>();
        for (BytecodePosition state = position; state != null; state = state.getCaller()) {
            sourceReference.add(new SourceReference(state.getBCI(), state.getMethod().asStackTraceElement(state.getBCI())));
        }
        return sourceReference.toArray(new SourceReference[sourceReference.size()]);
    }

    private void printMethods(PrintWriter out) {
        out.println("VM Entry Points");
        Iterator iterator = this.methodToNode.values().stream().filter(n -> n.isEntryPoint).iterator();
        while (iterator.hasNext()) {
            MethodNode node = (MethodNode)iterator.next();
            boolean lastEntryPoint = !iterator.hasNext();
            out.format("%s%s %s %n", lastEntryPoint ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "entry", node.format());
            CallTreePrinter.printCallTreeNode(out, lastEntryPoint ? "    " : "\u2502   ", node);
        }
        out.println();
    }

    private static void printCallTreeNode(PrintWriter out, String prefix, MethodNode node) {
        for (int invokeIdx = 0; invokeIdx < node.invokes.size(); ++invokeIdx) {
            boolean lastInvoke;
            InvokeNode invoke = node.invokes.get(invokeIdx);
            boolean bl = lastInvoke = invokeIdx == node.invokes.size() - 1;
            if (invoke.isDirectInvoke) {
                if (invoke.callees.size() <= 0) continue;
                Node calleeNode = invoke.callees.get(0);
                out.format("%s%s%s %s @bci=%s %n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "directly calls", calleeNode.format(), invoke.formatLocation());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   "), (MethodNode)calleeNode);
                continue;
            }
            out.format("%s%s%s %s @bci=%s%n", prefix, lastInvoke ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "virtually calls", invoke.formatTarget(), invoke.formatLocation());
            for (int calleeIdx = 0; calleeIdx < invoke.callees.size(); ++calleeIdx) {
                boolean lastCallee = calleeIdx == invoke.callees.size() - 1;
                Node calleeNode = invoke.callees.get(calleeIdx);
                out.format("%s%s%s %s %n", prefix + (lastInvoke ? "    " : "\u2502   "), lastCallee ? "\u2514\u2500\u2500 " : "\u251c\u2500\u2500 ", "is overridden by", calleeNode.format());
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.printCallTreeNode(out, prefix + (lastInvoke ? "    " : "\u2502   ") + (lastCallee ? "    " : "\u2502   "), (MethodNode)calleeNode);
            }
        }
    }

    private void printUsedMethods(PrintWriter out) {
        ArrayList<String> methodsList = new ArrayList<String>();
        for (ResolvedJavaMethod resolvedJavaMethod : this.methodToNode.keySet()) {
            methodsList.add(resolvedJavaMethod.format("%H.%n(%p):%r"));
        }
        methodsList.sort(null);
        for (String string : methodsList) {
            out.println(string);
        }
    }

    private void printClasses(PrintWriter out, boolean packageNameOnly) {
        ArrayList<String> classList = new ArrayList<String>(this.classesSet(packageNameOnly));
        classList.sort(null);
        for (String name : classList) {
            out.println(name);
        }
    }

    public Set<String> classesSet(boolean packageNameOnly) {
        HashSet<String> classSet = new HashSet<String>();
        for (ResolvedJavaMethod resolvedJavaMethod : this.methodToNode.keySet()) {
            String name = resolvedJavaMethod.getDeclaringClass().toJavaName(true);
            if (packageNameOnly && (name = CallTreePrinter.packagePrefix(name)).contains("$$Lambda$")) {
                name = CallTreePrinter.packagePrefix(name);
            }
            classSet.add(name);
        }
        return classSet;
    }

    private static String packagePrefix(String name) {
        int lastDot = name.lastIndexOf(46);
        if (lastDot == -1) {
            return name;
        }
        return name.substring(0, lastDot);
    }

    private static void printCsvFiles(Map<AnalysisMethod, MethodNode> methodToNode, String reportsPath, String reportName) {
        AtomicInteger virtualNodeId = new AtomicInteger(MethodNode.methodId);
        HashSet<Integer> entryPointIds = new HashSet<Integer>();
        HashSet<MethodNode> nonVirtualNodes = new HashSet<MethodNode>();
        HashMap<List<String>, Integer> virtualNodes = new HashMap<List<String>, Integer>();
        HashMap<Integer, Set<BciEndEdge>> directEdges = new HashMap<Integer, Set<BciEndEdge>>();
        HashMap<Integer, Set<BciEndEdge>> virtualEdges = new HashMap<Integer, Set<BciEndEdge>>();
        HashMap<Integer, Set<Integer>> overridenByEdges = new HashMap<Integer, Set<Integer>>();
        Iterator iterator = methodToNode.values().stream().filter(n -> n.isEntryPoint).iterator();
        while (iterator.hasNext()) {
            MethodNode node = (MethodNode)iterator.next();
            entryPointIds.add(node.id);
            CallTreePrinter.walkNodes(node, directEdges, virtualEdges, overridenByEdges, virtualNodes, nonVirtualNodes, virtualNodeId);
        }
        String msgPrefix = "call tree csv file for ";
        String timeStamp = ReportUtils.getTimeStampString();
        CallTreePrinter.toCsvFile(msgPrefix + "vm entry point", reportsPath, "call_tree_vm", reportName, timeStamp, CallTreePrinter::printVMEntryPoint);
        CallTreePrinter.toCsvFile(msgPrefix + "methods", reportsPath, "call_tree_methods", reportName, timeStamp, writer -> CallTreePrinter.printMethodNodes(methodToNode.values(), writer));
        CallTreePrinter.toCsvFile(msgPrefix + "virtual methods", reportsPath, "call_tree_virtual_methods", reportName, timeStamp, writer -> CallTreePrinter.printVirtualNodes(virtualNodes, writer));
        CallTreePrinter.toCsvFile(msgPrefix + "entry points", reportsPath, "call_tree_entry_points", reportName, timeStamp, writer -> CallTreePrinter.printEntryPointIds(entryPointIds, writer));
        CallTreePrinter.toCsvFile(msgPrefix + "direct edges", reportsPath, "call_tree_direct_edges", reportName, timeStamp, writer -> CallTreePrinter.printBciEdges(directEdges, writer));
        CallTreePrinter.toCsvFile(msgPrefix + "overriden by edges", reportsPath, "call_tree_override_by_edges", reportName, timeStamp, writer -> CallTreePrinter.printNonBciEdges(overridenByEdges, writer));
        CallTreePrinter.toCsvFile(msgPrefix + "virtual edges", reportsPath, "call_tree_virtual_edges", reportName, timeStamp, writer -> CallTreePrinter.printBciEdges(virtualEdges, writer));
    }

    private static void toCsvFile(String description, String reportsPath, String prefix, String reportName, String timeStamp, Consumer<PrintWriter> reporter) {
        String name = prefix + "_" + reportName;
        Path csvFile = ReportUtils.report(description, reportsPath, name, "csv", reporter, true, timeStamp);
        Path csvLink = Paths.get(reportsPath, new String[0]).resolve(prefix + ".csv");
        if (Files.exists(csvLink, LinkOption.NOFOLLOW_LINKS)) {
            try {
                Files.delete(csvLink);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        try {
            Files.createSymbolicLink(csvLink, csvFile.getFileName(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void printVMEntryPoint(PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id", "Name"));
        writer.println(CallTreePrinter.convertToCSV("0", "VM"));
    }

    private static void printMethodNodes(Collection<MethodNode> methods, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id", "Name", "Type", "Parameters", "Return", "Display", "Flags"));
        methods.stream().map(CallTreePrinter::methodNodeInfo).map(CallTreePrinter::convertToCSV).forEach(writer::println);
    }

    private static List<String> methodNodeInfo(MethodNode method) {
        return CallTreePrinter.resolvedJavaMethodInfo(method.id, method.method);
    }

    private static void walkNodes(MethodNode methodNode, Map<Integer, Set<BciEndEdge>> directEdges, Map<Integer, Set<BciEndEdge>> virtualEdges, Map<Integer, Set<Integer>> overridenByEdges, Map<List<String>, Integer> virtualNodes, Set<MethodNode> nonVirtualNodes, AtomicInteger virtualNodeId) {
        for (InvokeNode invoke : methodNode.invokes) {
            if (invoke.isDirectInvoke) {
                if (invoke.callees.size() <= 0) continue;
                Node calleeNode = invoke.callees.get(0);
                CallTreePrinter.addDirectEdge(methodNode.id, invoke, calleeNode, directEdges, nonVirtualNodes);
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.walkNodes((MethodNode)calleeNode, directEdges, virtualEdges, overridenByEdges, virtualNodes, nonVirtualNodes, virtualNodeId);
                continue;
            }
            int nodeId = CallTreePrinter.addVirtualNode(invoke, virtualNodes, virtualNodeId);
            CallTreePrinter.addVirtualMethodEdge(methodNode.id, invoke, nodeId, virtualEdges);
            for (Node calleeNode : invoke.callees) {
                CallTreePrinter.addOverridenByEdge(nodeId, calleeNode, overridenByEdges, nonVirtualNodes);
                if (!(calleeNode instanceof MethodNode)) continue;
                CallTreePrinter.walkNodes((MethodNode)calleeNode, directEdges, virtualEdges, overridenByEdges, virtualNodes, nonVirtualNodes, virtualNodeId);
            }
        }
    }

    private static void addDirectEdge(int nodeId, InvokeNode invoke, Node calleeNode, Map<Integer, Set<BciEndEdge>> edges, Set<MethodNode> nodes) {
        Set nodeEdges = edges.computeIfAbsent(nodeId, k -> new HashSet());
        MethodNode methodNode = calleeNode instanceof MethodNode ? (MethodNode)calleeNode : ((MethodNodeReference)calleeNode).methodNode;
        nodes.add(methodNode);
        nodeEdges.add(new BciEndEdge(methodNode.id, CallTreePrinter.bytecodeIndexes(invoke)));
    }

    private static List<Integer> bytecodeIndexes(InvokeNode node) {
        return Stream.of(node.sourceReferences).map(source -> source.bci).collect(Collectors.toList());
    }

    private static int addVirtualNode(InvokeNode node, Map<List<String>, Integer> virtualNodes, AtomicInteger virtualNodeId) {
        List<String> virtualMethodInfo = CallTreePrinter.virtualMethodInfo(node.targetMethod);
        return virtualNodes.computeIfAbsent(virtualMethodInfo, k -> virtualNodeId.getAndIncrement());
    }

    private static void addVirtualMethodEdge(int startId, InvokeNode invoke, int endId, Map<Integer, Set<BciEndEdge>> edges) {
        Set nodeEdges = edges.computeIfAbsent(startId, k -> new HashSet());
        nodeEdges.add(new BciEndEdge(endId, CallTreePrinter.bytecodeIndexes(invoke)));
    }

    private static void printVirtualNodes(Map<List<String>, Integer> virtualNodes, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id", "Name", "Type", "Parameters", "Return", "Display"));
        virtualNodes.entrySet().stream().map(CallTreePrinter::virtualMethodAndIdInfo).map(CallTreePrinter::convertToCSV).forEach(writer::println);
    }

    private static List<String> virtualMethodAndIdInfo(Map.Entry<List<String>, Integer> entry) {
        List<String> methodInfo = entry.getKey();
        ArrayList<String> result = new ArrayList<String>(methodInfo.size() + 1);
        result.add(String.valueOf(entry.getValue()));
        for (int i = 1; i < methodInfo.size(); ++i) {
            result.add(i, methodInfo.get(i));
        }
        return result;
    }

    private static void printEntryPointIds(Set<Integer> entryPoints, PrintWriter writer) {
        writer.println(CallTreePrinter.convertToCSV("Id"));
        entryPoints.forEach(writer::println);
    }

    private static void addOverridenByEdge(int nodeId, Node calleeNode, Map<Integer, Set<Integer>> edges, Set<MethodNode> nodes) {
        Set nodeEdges = edges.computeIfAbsent(nodeId, k -> new HashSet());
        MethodNode methodNode = calleeNode instanceof MethodNode ? (MethodNode)calleeNode : ((MethodNodeReference)calleeNode).methodNode;
        nodes.add(methodNode);
        nodeEdges.add(methodNode.id);
    }

    private static void printBciEdges(Map<Integer, Set<BciEndEdge>> edges, PrintWriter writer) {
        Set idEdges = edges.entrySet().stream().flatMap(entry -> ((Set)entry.getValue()).stream().map(endId -> new BciEdge((Integer)entry.getKey(), (BciEndEdge)endId))).collect(Collectors.toSet());
        writer.println(CallTreePrinter.convertToCSV("StartId", "EndId", "BytecodeIndexes"));
        idEdges.stream().map(edge -> CallTreePrinter.convertToCSV(String.valueOf(edge.startId), String.valueOf(edge.endEdge.id), CallTreePrinter.showBytecodeIndexes(edge.endEdge.bytecodeIndexes))).forEach(writer::println);
    }

    private static String showBytecodeIndexes(List<Integer> bytecodeIndexes) {
        return bytecodeIndexes.stream().map(String::valueOf).collect(Collectors.joining("->"));
    }

    private static void printNonBciEdges(Map<Integer, Set<Integer>> edges, PrintWriter writer) {
        Set idEdges = edges.entrySet().stream().flatMap(entry -> ((Set)entry.getValue()).stream().map(endId -> new NonBciEdge((Integer)entry.getKey(), (int)endId))).collect(Collectors.toSet());
        writer.println(CallTreePrinter.convertToCSV("StartId", "EndId"));
        idEdges.stream().map(edge -> CallTreePrinter.convertToCSV(String.valueOf(edge.startId), String.valueOf(edge.endId))).forEach(writer::println);
    }

    private static List<String> virtualMethodInfo(AnalysisMethod method) {
        return CallTreePrinter.resolvedJavaMethodInfo(null, method);
    }

    private static List<String> resolvedJavaMethodInfo(Integer id, ResolvedJavaMethod method) {
        String parameters = method.getSignature().getParameterCount(false) > 0 ? method.format("%P").replace(",", "") : "empty";
        return Arrays.asList(id == null ? null : Integer.toString(id), method.getName(), method.getDeclaringClass().toJavaName(true), parameters, method.getSignature().getReturnType(null).toJavaName(true), CallTreePrinter.display(method), CallTreePrinter.flags(method));
    }

    private static String display(ResolvedJavaMethod method) {
        ResolvedJavaType type = method.getDeclaringClass();
        String typeName = type.toJavaName(true);
        if (type.getJavaKind() == JavaKind.Object) {
            ArrayList<String> matchResults = new ArrayList<String>();
            Matcher matcher = CAMEL_CASE_PATTERN.matcher(typeName);
            while (matcher.find()) {
                matchResults.add(matcher.toMatchResult().group());
            }
            return String.join((CharSequence)"", matchResults) + "." + method.getName();
        }
        return typeName + "." + method.getName();
    }

    private static String flags(ResolvedJavaMethod method) {
        StringBuilder sb = new StringBuilder();
        if (method.isPublic()) {
            sb.append('p');
        } else if (method.isPrivate()) {
            sb.append('P');
        } else if (method.isProtected()) {
            sb.append('d');
        }
        if (method.isStatic()) {
            sb.append('s');
        }
        if (method.isFinal()) {
            sb.append('f');
        }
        if (method.isSynchronized()) {
            sb.append('S');
        }
        if (method.isBridge()) {
            sb.append('b');
        }
        if (method.isVarArgs()) {
            sb.append('v');
        }
        if (method.isNative()) {
            sb.append('n');
        }
        if (method.isAbstract()) {
            sb.append('a');
        }
        if (method.isSynthetic()) {
            sb.append('y');
        }
        return sb.toString();
    }

    private static String convertToCSV(String ... data) {
        return String.join((CharSequence)",", data);
    }

    private static String convertToCSV(List<String> data) {
        return String.join((CharSequence)",", data);
    }

    private static final class BciEndEdge {
        final int id;
        final List<Integer> bytecodeIndexes;

        private BciEndEdge(int id, List<Integer> bytecodeIndexes) {
            this.id = id;
            this.bytecodeIndexes = bytecodeIndexes;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BciEndEdge endEdge = (BciEndEdge)o;
            return this.id == endEdge.id && this.bytecodeIndexes.equals(endEdge.bytecodeIndexes);
        }

        public int hashCode() {
            return Objects.hash(this.id, this.bytecodeIndexes);
        }
    }

    private static final class BciEdge {
        final int startId;
        final BciEndEdge endEdge;

        private BciEdge(int startId, BciEndEdge endEdge) {
            this.startId = startId;
            this.endEdge = endEdge;
        }
    }

    private static final class NonBciEdge {
        final int startId;
        final int endId;

        private NonBciEdge(int startId, int endId) {
            this.startId = startId;
            this.endId = endId;
        }
    }

    static class SourceReference {
        static final SourceReference UNKNOWN_SOURCE_REFERENCE = new SourceReference(-1, null);
        final int bci;
        final StackTraceElement trace;

        SourceReference(int bci, StackTraceElement trace) {
            this.bci = bci;
            this.trace = trace;
        }
    }

    static class InvokeNode {
        private final AnalysisMethod targetMethod;
        private final List<Node> callees;
        private final boolean isDirectInvoke;
        private final SourceReference[] sourceReferences;

        InvokeNode(AnalysisMethod targetMethod, boolean isDirectInvoke, SourceReference[] sourceReferences) {
            this.targetMethod = targetMethod;
            this.isDirectInvoke = isDirectInvoke;
            this.sourceReferences = sourceReferences;
            this.callees = new ArrayList<Node>();
        }

        void addCallee(Node callee) {
            this.callees.add(callee);
        }

        String formatLocation() {
            return Arrays.stream(this.sourceReferences).map(s -> String.valueOf(s.bci)).collect(Collectors.joining("->"));
        }

        String formatTarget() {
            return this.targetMethod.format(CallTreePrinter.METHOD_FORMAT);
        }
    }

    static class MethodNode
    implements Node {
        static int methodId = 0;
        private final int id = methodId++;
        private final AnalysisMethod method;
        private final List<InvokeNode> invokes;
        private final boolean isEntryPoint;

        MethodNode(AnalysisMethod method) {
            this(method, false);
        }

        MethodNode(AnalysisMethod method, boolean isEntryPoint) {
            this.method = method;
            this.invokes = new ArrayList<InvokeNode>();
            this.isEntryPoint = isEntryPoint;
        }

        void addInvoke(InvokeNode invoke) {
            this.invokes.add(invoke);
        }

        @Override
        public String format() {
            return this.method.format(CallTreePrinter.METHOD_FORMAT) + " id=" + this.id;
        }
    }

    static class MethodNodeReference
    implements Node {
        private final MethodNode methodNode;

        MethodNodeReference(MethodNode methodNode) {
            this.methodNode = methodNode;
        }

        @Override
        public String format() {
            return this.methodNode.method.format(CallTreePrinter.METHOD_FORMAT) + " id-ref=" + this.methodNode.id;
        }
    }

    static interface Node {
        public String format();
    }
}

