/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.genscavenge;

import com.oracle.svm.core.MemoryWalker;
import com.oracle.svm.core.SubstrateOptions;
import com.oracle.svm.core.config.ConfigurationValues;
import com.oracle.svm.core.genscavenge.AlignedHeapChunk;
import com.oracle.svm.core.genscavenge.HeapChunk;
import com.oracle.svm.core.genscavenge.HeapImpl;
import com.oracle.svm.core.genscavenge.HeapOptions;
import com.oracle.svm.core.genscavenge.ImageHeapInfo;
import com.oracle.svm.core.genscavenge.ImageHeapWalker;
import com.oracle.svm.core.genscavenge.ObjectHeaderImpl;
import com.oracle.svm.core.genscavenge.OldGeneration;
import com.oracle.svm.core.genscavenge.Space;
import com.oracle.svm.core.genscavenge.UnalignedHeapChunk;
import com.oracle.svm.core.genscavenge.YoungGeneration;
import com.oracle.svm.core.genscavenge.remset.RememberedSet;
import com.oracle.svm.core.heap.ObjectReferenceVisitor;
import com.oracle.svm.core.heap.ObjectVisitor;
import com.oracle.svm.core.heap.ReferenceAccess;
import com.oracle.svm.core.heap.ReferenceInternals;
import com.oracle.svm.core.hub.DynamicHub;
import com.oracle.svm.core.hub.InteriorObjRefWalker;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.snippets.KnownIntrinsics;
import java.lang.ref.Reference;
import org.graalvm.compiler.word.Word;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.word.ComparableWord;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordBase;
import org.graalvm.word.WordFactory;

public final class HeapVerifier {
    private static final ObjectVerifier OBJECT_VERIFIER = new ObjectVerifier();
    private static final ImageHeapRegionVerifier IMAGE_HEAP_OBJECT_VERIFIER = new ImageHeapRegionVerifier();
    private static final ObjectReferenceVerifier REFERENCE_VERIFIER = new ObjectReferenceVerifier();

    @Platforms(value={Platform.HOSTED_ONLY.class})
    private HeapVerifier() {
    }

    public static boolean verify(Occasion occasion) {
        boolean success = true;
        success &= HeapVerifier.verifyImageHeapObjects();
        success &= HeapVerifier.verifyYoungGeneration(occasion);
        success &= HeapVerifier.verifyOldGeneration();
        return success &= HeapVerifier.verifyRememberedSets();
    }

    private static boolean verifyImageHeapObjects() {
        if (HeapImpl.usesImageHeapChunks()) {
            return HeapVerifier.verifyChunkedImageHeap();
        }
        return HeapVerifier.verifyNonChunkedImageHeap();
    }

    private static boolean verifyChunkedImageHeap() {
        boolean success = true;
        ImageHeapInfo info = HeapImpl.getImageHeapInfo();
        success &= HeapVerifier.verifyAlignedChunks(null, info.getFirstAlignedImageHeapChunk());
        return success &= HeapVerifier.verifyUnalignedChunks(null, info.getFirstUnalignedImageHeapChunk());
    }

    private static boolean verifyNonChunkedImageHeap() {
        IMAGE_HEAP_OBJECT_VERIFIER.initialize();
        ImageHeapWalker.walkRegions(HeapImpl.getImageHeapInfo(), IMAGE_HEAP_OBJECT_VERIFIER);
        return IMAGE_HEAP_OBJECT_VERIFIER.getResult();
    }

    private static boolean verifyYoungGeneration(Occasion occasion) {
        Space eden;
        boolean success = true;
        YoungGeneration youngGeneration = HeapImpl.getHeapImpl().getYoungGeneration();
        if (occasion == Occasion.AFTER_COLLECTION && !(eden = youngGeneration.getEden()).isEmpty()) {
            Log.log().string("Eden contains chunks after a collection: firstAlignedChunk: ").hex((WordBase)eden.getFirstAlignedHeapChunk()).string(", firstUnalignedChunk: ").hex((WordBase)eden.getFirstUnalignedHeapChunk()).newline();
            success = false;
        }
        success &= HeapVerifier.verifySpace(youngGeneration.getEden());
        for (int i = 0; i < youngGeneration.getMaxSurvivorSpaces(); ++i) {
            Space fromSpace = youngGeneration.getSurvivorFromSpaceAt(i);
            Space toSpace = youngGeneration.getSurvivorToSpaceAt(i);
            if (!toSpace.isEmpty()) {
                Log.log().string("Survivor to-space ").signed(i).string(" contains chunks: firstAlignedChunk: ").hex((WordBase)toSpace.getFirstAlignedHeapChunk()).string(", firstUnalignedChunk: ").hex((WordBase)toSpace.getFirstUnalignedHeapChunk()).newline();
                success = false;
            }
            success &= HeapVerifier.verifySpace(fromSpace);
            success &= HeapVerifier.verifySpace(toSpace);
        }
        return success;
    }

    private static boolean verifyOldGeneration() {
        boolean success = true;
        OldGeneration oldGeneration = HeapImpl.getHeapImpl().getOldGeneration();
        Space fromSpace = oldGeneration.getFromSpace();
        Space toSpace = oldGeneration.getToSpace();
        if (!toSpace.isEmpty()) {
            Log.log().string("Old generation to-space contains chunks: firstAlignedChunk: ").hex((WordBase)toSpace.getFirstAlignedHeapChunk()).string(", firstUnalignedChunk: ").hex((WordBase)toSpace.getFirstUnalignedHeapChunk()).newline();
            success = false;
        }
        success &= HeapVerifier.verifySpace(fromSpace);
        return success &= HeapVerifier.verifySpace(toSpace);
    }

    private static boolean verifyRememberedSets() {
        if (!SubstrateOptions.useRememberedSet() || !HeapOptions.VerifyRememberedSet.getValue().booleanValue()) {
            return true;
        }
        boolean success = true;
        RememberedSet rememberedSet = RememberedSet.get();
        if (HeapImpl.usesImageHeapChunks()) {
            ImageHeapInfo info = HeapImpl.getImageHeapInfo();
            success &= rememberedSet.verify(info.getFirstAlignedImageHeapChunk());
            success &= rememberedSet.verify(info.getFirstUnalignedImageHeapChunk());
        }
        OldGeneration oldGeneration = HeapImpl.getHeapImpl().getOldGeneration();
        Space toSpace = oldGeneration.getToSpace();
        success &= rememberedSet.verify(toSpace.getFirstAlignedHeapChunk());
        success &= rememberedSet.verify(toSpace.getFirstUnalignedHeapChunk());
        Space fromSpace = oldGeneration.getFromSpace();
        success &= rememberedSet.verify(fromSpace.getFirstAlignedHeapChunk());
        return success &= rememberedSet.verify(fromSpace.getFirstUnalignedHeapChunk());
    }

    private static boolean verifySpace(Space space) {
        boolean success = true;
        success &= HeapVerifier.verifyChunkList(space, "aligned", space.getFirstAlignedHeapChunk(), space.getLastAlignedHeapChunk());
        success &= HeapVerifier.verifyChunkList(space, "unaligned", space.getFirstUnalignedHeapChunk(), space.getLastUnalignedHeapChunk());
        success &= HeapVerifier.verifyAlignedChunks(space, space.getFirstAlignedHeapChunk());
        return success &= HeapVerifier.verifyUnalignedChunks(space, space.getFirstUnalignedHeapChunk());
    }

    private static boolean verifyChunkList(Space space, String kind, HeapChunk.Header<?> firstChunk, HeapChunk.Header<?> lastChunk) {
        boolean result = true;
        HeapChunk.Header<?> current = firstChunk;
        HeapChunk.Header<?> previous = (HeapChunk.Header<?>)WordFactory.nullPointer();
        while (current.isNonNull()) {
            Object previousOfCurrent = HeapChunk.getPrevious(current);
            if (previousOfCurrent.notEqual((ComparableWord)previous)) {
                Log.log().string("Verification failed for the doubly-linked list that holds ").string(kind).string(" chunks: space: ").string(space.getName()).string(", current: ").hex((WordBase)current).string(", current.previous: ").hex((WordBase)previousOfCurrent).string(", previous: ").hex((WordBase)previous).newline();
                result = false;
            }
            previous = current;
            current = HeapChunk.getNext(current);
        }
        if (previous.notEqual((ComparableWord)lastChunk)) {
            Log.log().string("Verification failed for the doubly-linked list that holds ").string(kind).string(" chunks: space: ").string(space.getName()).string(", previous: ").hex((WordBase)previous).string(", lastChunk: ").hex((WordBase)lastChunk).newline();
            result = false;
        }
        return result;
    }

    private static boolean verifyAlignedChunks(Space space, AlignedHeapChunk.AlignedHeader firstAlignedHeapChunk) {
        boolean success = true;
        AlignedHeapChunk.AlignedHeader aChunk = firstAlignedHeapChunk;
        while (aChunk.isNonNull()) {
            if (space != aChunk.getSpace()) {
                Log.log().string("Space ").string(space.getName()).string(" contains aligned chunk ").hex((WordBase)aChunk).string(" but the chunk does not reference the correct space: ").hex((WordBase)Word.objectToUntrackedPointer((Object)aChunk.getSpace())).newline();
                success = false;
            }
            OBJECT_VERIFIER.initialize(aChunk, (UnalignedHeapChunk.UnalignedHeader)WordFactory.nullPointer());
            AlignedHeapChunk.walkObjects(aChunk, OBJECT_VERIFIER);
            aChunk = HeapChunk.getNext(aChunk);
            success &= HeapVerifier.OBJECT_VERIFIER.result;
        }
        return success;
    }

    private static boolean verifyUnalignedChunks(Space space, UnalignedHeapChunk.UnalignedHeader firstUnalignedHeapChunk) {
        boolean success = true;
        UnalignedHeapChunk.UnalignedHeader uChunk = firstUnalignedHeapChunk;
        while (uChunk.isNonNull()) {
            if (space != uChunk.getSpace()) {
                Log.log().string("Space ").string(space.getName()).string(" contains unaligned chunk ").hex((WordBase)uChunk).string(" but the chunk does not reference the correct space: ").hex((WordBase)Word.objectToUntrackedPointer((Object)uChunk.getSpace())).newline();
                success = false;
            }
            OBJECT_VERIFIER.initialize((AlignedHeapChunk.AlignedHeader)WordFactory.nullPointer(), uChunk);
            UnalignedHeapChunk.walkObjects(uChunk, OBJECT_VERIFIER);
            uChunk = HeapChunk.getNext(uChunk);
            success &= HeapVerifier.OBJECT_VERIFIER.result;
        }
        return success;
    }

    private static boolean verifyObject(Object obj, AlignedHeapChunk.AlignedHeader aChunk, UnalignedHeapChunk.UnalignedHeader uChunk) {
        Word ptr = Word.objectToUntrackedPointer((Object)obj);
        if (ptr.isNull()) {
            Log.log().string("Encounter a null pointer while walking the heap objects.").newline();
            return false;
        }
        int objectAlignment = ConfigurationValues.getObjectLayout().getAlignment();
        if (ptr.unsignedRemainder(objectAlignment).notEqual(0)) {
            Log.log().string("Object ").hex((WordBase)ptr).string(" is not properly aligned to ").signed(objectAlignment).string(" bytes.").newline();
            return false;
        }
        UnsignedWord header = ObjectHeaderImpl.readHeaderFromPointer((Pointer)ptr);
        if (ObjectHeaderImpl.isProducedHeapChunkZapped(header) || ObjectHeaderImpl.isConsumedHeapChunkZapped(header)) {
            Log.log().string("Object ").hex((WordBase)ptr).string(" has a zapped header: ").hex((WordBase)header).newline();
            return false;
        }
        if (ObjectHeaderImpl.isForwardedHeader(header)) {
            Log.log().string("Object ").hex((WordBase)ptr).string(" has a forwarded header: ").hex((WordBase)header).newline();
            return false;
        }
        if (HeapImpl.usesImageHeapChunks() || !HeapImpl.getHeapImpl().isInImageHeap(obj)) {
            Space space;
            assert (aChunk.isNonNull() ^ uChunk.isNonNull());
            HeapChunk.Header<AlignedHeapChunk.AlignedHeader> expectedChunk = aChunk.isNonNull() ? aChunk : uChunk;
            HeapChunk.Header<?> chunk = HeapChunk.getEnclosingHeapChunk(obj);
            if (chunk.notEqual((ComparableWord)expectedChunk)) {
                Log.log().string("Object ").hex((WordBase)ptr).string(" should have ").hex((WordBase)expectedChunk).string(" as its enclosing chunk but getEnclosingHeapChunk returned ").hex((WordBase)chunk).newline();
                return false;
            }
            Pointer chunkStart = HeapChunk.asPointer(chunk);
            Pointer chunkTop = HeapChunk.getTopPointer(chunk);
            if (chunkStart.aboveOrEqual((UnsignedWord)ptr) || chunkTop.belowOrEqual((UnsignedWord)ptr)) {
                Log.log().string("Object ").hex((WordBase)ptr).string(" is not within the allocated part of the chunk: ").hex((WordBase)chunkStart).string(" - ").hex((WordBase)chunkTop).string("").newline();
                return false;
            }
            if (aChunk.isNonNull()) {
                if (!ObjectHeaderImpl.isAlignedHeader(header)) {
                    Log.log().string("Header of object ").hex((WordBase)ptr).string(" is not marked as aligned: ").hex((WordBase)header).newline();
                    return false;
                }
            } else {
                assert (uChunk.isNonNull());
                if (!ObjectHeaderImpl.isUnalignedHeader(header)) {
                    Log.log().string("Header of object ").hex((WordBase)ptr).string(" is not marked as unaligned: ").hex((WordBase)header).newline();
                    return false;
                }
            }
            if ((space = chunk.getSpace()) == null) {
                if (!HeapImpl.getHeapImpl().isInImageHeap(obj)) {
                    Log.log().string("Object ").hex((WordBase)ptr).string(" is not an image heap object even though the space of the parent chunk ").hex((WordBase)chunk).string(" is null.").newline();
                    return false;
                }
            } else if (space.isOldSpace() && SubstrateOptions.useRememberedSet() && !RememberedSet.get().hasRememberedSet(header)) {
                Log.log().string("Object ").hex((WordBase)ptr).string(" is in old generation chunk ").hex((WordBase)chunk).string(" but does not have a remembered set.").newline();
                return false;
            }
        }
        DynamicHub hub = KnownIntrinsics.readHub(obj);
        if (!HeapImpl.getHeapImpl().isInImageHeap(hub)) {
            Log.log().string("Object ").hex((WordBase)ptr).string(" references a hub that is not in the image heap: ").hex((WordBase)Word.objectToUntrackedPointer((Object)hub)).newline();
            return false;
        }
        return HeapVerifier.verifyReferences(obj);
    }

    private static boolean verifyReferences(Object obj) {
        if (!HeapOptions.VerifyReferences.getValue().booleanValue()) {
            return true;
        }
        REFERENCE_VERIFIER.initialize(obj);
        InteriorObjRefWalker.walkObject(obj, REFERENCE_VERIFIER);
        boolean success = REFERENCE_VERIFIER.result;
        DynamicHub hub = KnownIntrinsics.readHub(obj);
        if (hub.isReferenceInstanceClass()) {
            Reference ref = (Reference)obj;
            success &= HeapVerifier.verifyReferent(ref);
        }
        return success;
    }

    private static boolean verifyReferent(Reference<?> ref) {
        return HeapVerifier.verifyReference(ref, ReferenceInternals.getReferentFieldAddress(ref), ReferenceInternals.getReferentPointer(ref));
    }

    public static boolean verifyReference(Object parentObject, Pointer objRef, boolean compressed) {
        Word ptr = ReferenceAccess.singleton().readObjectAsUntrackedPointer(objRef, compressed);
        return HeapVerifier.verifyReference(parentObject, objRef, (Pointer)ptr);
    }

    private static boolean verifyReference(Object parentObject, Pointer reference, Pointer referencedObject) {
        if (referencedObject.isNull()) {
            return true;
        }
        if (!HeapVerifier.isInHeap(referencedObject)) {
            Log.log().string("Object reference at ").hex((WordBase)reference).string(" points outside the Java heap: ").hex((WordBase)referencedObject).string(". ");
            if (parentObject != null) {
                Log.log().string("The object that contains the invalid reference is of type ").string(parentObject.getClass().getName()).newline();
            } else {
                Log.log().string("The invalid reference is on the stack.").newline();
            }
            return false;
        }
        return true;
    }

    private static boolean isInHeap(Pointer ptr) {
        HeapImpl heap = HeapImpl.getHeapImpl();
        return heap.isInImageHeap(ptr) || HeapVerifier.isInYoungGen(ptr) || HeapVerifier.isInOldGen(ptr);
    }

    private static boolean isInYoungGen(Pointer ptr) {
        YoungGeneration youngGen = HeapImpl.getHeapImpl().getYoungGeneration();
        if (HeapVerifier.findPointerInSpace(youngGen.getEden(), ptr)) {
            return true;
        }
        for (int i = 0; i < youngGen.getMaxSurvivorSpaces(); ++i) {
            if (HeapVerifier.findPointerInSpace(youngGen.getSurvivorFromSpaceAt(i), ptr)) {
                return true;
            }
            if (!HeapVerifier.findPointerInSpace(youngGen.getSurvivorToSpaceAt(i), ptr)) continue;
            return true;
        }
        return false;
    }

    private static boolean isInOldGen(Pointer ptr) {
        OldGeneration oldGen = HeapImpl.getHeapImpl().getOldGeneration();
        return HeapVerifier.findPointerInSpace(oldGen.getFromSpace(), ptr) || HeapVerifier.findPointerInSpace(oldGen.getToSpace(), ptr);
    }

    private static boolean findPointerInSpace(Space space, Pointer p) {
        AlignedHeapChunk.AlignedHeader aChunk = space.getFirstAlignedHeapChunk();
        while (aChunk.isNonNull()) {
            Pointer start = AlignedHeapChunk.getObjectsStart(aChunk);
            if (start.belowOrEqual((UnsignedWord)p) && p.belowThan((UnsignedWord)HeapChunk.getTopPointer(aChunk))) {
                return true;
            }
            aChunk = HeapChunk.getNext(aChunk);
        }
        UnalignedHeapChunk.UnalignedHeader uChunk = space.getFirstUnalignedHeapChunk();
        while (uChunk.isNonNull()) {
            Pointer start = UnalignedHeapChunk.getObjectStart(uChunk);
            if (start.belowOrEqual((UnsignedWord)p) && p.belowThan((UnsignedWord)HeapChunk.getTopPointer(uChunk))) {
                return true;
            }
            uChunk = HeapChunk.getNext(uChunk);
        }
        return false;
    }

    public static enum Occasion {
        BEFORE_COLLECTION,
        AFTER_COLLECTION;

    }

    private static class ObjectReferenceVerifier
    implements ObjectReferenceVisitor {
        private Object parentObject;
        private boolean result;

        @Platforms(value={Platform.HOSTED_ONLY.class})
        ObjectReferenceVerifier() {
        }

        public void initialize(Object parentObject) {
            this.parentObject = parentObject;
            this.result = true;
        }

        @Override
        public boolean visitObjectReference(Pointer objRef, boolean compressed) {
            this.result &= HeapVerifier.verifyReference(this.parentObject, objRef, compressed);
            return true;
        }
    }

    private static class ImageHeapObjectVerifier
    extends ObjectVerifier {
        @Platforms(value={Platform.HOSTED_ONLY.class})
        ImageHeapObjectVerifier() {
        }

        @Override
        public boolean visitObject(Object object) {
            Word pointer = Word.objectToUntrackedPointer((Object)object);
            if (!HeapImpl.getHeapImpl().isInImageHeap(object)) {
                Log.log().string("Image heap object ").hex((WordBase)pointer).string(" is not considered as part of the image heap.").newline();
                this.result = false;
            }
            return super.visitObject(object);
        }
    }

    private static class ObjectVerifier
    implements ObjectVisitor {
        protected boolean result;
        private AlignedHeapChunk.AlignedHeader aChunk;
        private UnalignedHeapChunk.UnalignedHeader uChunk;

        @Platforms(value={Platform.HOSTED_ONLY.class})
        ObjectVerifier() {
        }

        void initialize(AlignedHeapChunk.AlignedHeader aChunk, UnalignedHeapChunk.UnalignedHeader uChunk) {
            this.result = true;
            this.aChunk = aChunk;
            this.uChunk = uChunk;
        }

        @Override
        public boolean visitObject(Object object) {
            this.result &= HeapVerifier.verifyObject(object, this.aChunk, this.uChunk);
            return true;
        }
    }

    private static class ImageHeapRegionVerifier
    implements MemoryWalker.ImageHeapRegionVisitor {
        private final ImageHeapObjectVerifier objectVerifier = new ImageHeapObjectVerifier();

        @Platforms(value={Platform.HOSTED_ONLY.class})
        ImageHeapRegionVerifier() {
        }

        public void initialize() {
            this.objectVerifier.initialize((AlignedHeapChunk.AlignedHeader)WordFactory.nullPointer(), (UnalignedHeapChunk.UnalignedHeader)WordFactory.nullPointer());
        }

        public boolean getResult() {
            return this.objectVerifier.result;
        }

        @Override
        public <T> boolean visitNativeImageHeapRegion(T region, MemoryWalker.NativeImageHeapRegionAccess<T> access) {
            access.visitObjects(region, this.objectVerifier);
            return true;
        }
    }
}

