/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.wasm.memory;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.dsl.Bind;
import com.oracle.truffle.api.dsl.Cached;
import com.oracle.truffle.api.interop.InteropLibrary;
import com.oracle.truffle.api.interop.InvalidArrayIndexException;
import com.oracle.truffle.api.interop.InvalidBufferOffsetException;
import com.oracle.truffle.api.interop.TruffleObject;
import com.oracle.truffle.api.interop.UnsupportedMessageException;
import com.oracle.truffle.api.interop.UnsupportedTypeException;
import com.oracle.truffle.api.library.CachedLibrary;
import com.oracle.truffle.api.library.ExportLibrary;
import com.oracle.truffle.api.library.ExportMessage;
import com.oracle.truffle.api.nodes.Node;
import com.oracle.truffle.api.profiles.InlinedBranchProfile;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.util.Locale;
import org.graalvm.wasm.Assert;
import org.graalvm.wasm.EmbedderDataHolder;
import org.graalvm.wasm.api.WebAssembly;
import org.graalvm.wasm.collection.ByteArrayList;
import org.graalvm.wasm.exception.Failure;
import org.graalvm.wasm.exception.WasmException;
import org.graalvm.wasm.memory.WasmMemoryLibrary;
import org.graalvm.wasm.nodes.WasmFunctionNode;

@ExportLibrary(value=InteropLibrary.class)
public abstract class WasmMemory
extends EmbedderDataHolder
implements TruffleObject {
    protected final long declaredMinSize;
    protected final long declaredMaxSize;
    protected long currentMinSize;
    protected final long maxAllowedSize;
    protected final boolean indexType64;
    protected final boolean shared;

    @CompilerDirectives.TruffleBoundary
    protected WasmMemory(long declaredMinSize, long declaredMaxSize, long initialSize, long maxAllowedSize, boolean indexType64, boolean shared) {
        Assert.assertUnsignedLongLessOrEqual(initialSize, maxAllowedSize, Failure.MEMORY_SIZE_LIMIT_EXCEEDED, "Initial memory size exceeds implementation limit");
        assert (Long.compareUnsigned(declaredMinSize, initialSize) <= 0);
        assert (Long.compareUnsigned(initialSize, maxAllowedSize) <= 0);
        assert (Long.compareUnsigned(maxAllowedSize, declaredMaxSize) <= 0);
        assert (indexType64 || Long.compareUnsigned(maxAllowedSize, 65536L) <= 0);
        assert (indexType64 || Long.compareUnsigned(declaredMaxSize, 65536L) <= 0);
        assert (!indexType64 || Long.compareUnsigned(maxAllowedSize, 976562500L) <= 0);
        assert (!indexType64 || Long.compareUnsigned(declaredMaxSize, 0x1000000000000L) <= 0);
        this.declaredMinSize = declaredMinSize;
        this.declaredMaxSize = declaredMaxSize;
        this.currentMinSize = declaredMinSize;
        this.maxAllowedSize = maxAllowedSize;
        this.indexType64 = indexType64;
        this.shared = shared;
    }

    public final long declaredMinSize() {
        return this.declaredMinSize;
    }

    public final long declaredMaxSize() {
        return this.declaredMaxSize;
    }

    public final long minSize() {
        return this.currentMinSize;
    }

    public final long maxAllowedSize() {
        return this.maxAllowedSize;
    }

    public final boolean hasIndexType64() {
        return this.indexType64;
    }

    public final boolean isShared() {
        return this.shared;
    }

    @CompilerDirectives.TruffleBoundary
    protected static final WasmException trapOutOfBounds(Node node, long address, long length, long byteSize) {
        String message = String.format(Locale.ROOT, "%d-byte memory access at address 0x%016X (%d) is out-of-bounds (memory size %d bytes).", length, address, address, byteSize);
        return WasmException.create(Failure.OUT_OF_BOUNDS_MEMORY_ACCESS, node, message);
    }

    @CompilerDirectives.TruffleBoundary
    protected static final WasmException trapUnalignedAtomic(Node node, long address, int length) {
        String message = String.format(Locale.ROOT, "%d-byte atomic memory access at address 0x%016X (%d) is unaligned.", length, address, address);
        return WasmException.create(Failure.UNALIGNED_ATOMIC, node, message);
    }

    @CompilerDirectives.TruffleBoundary
    protected static final WasmException trapNegativeLength(Node node, long length) {
        String message = String.format(Locale.ROOT, "memory access of negative length %d.", length);
        return WasmException.create(Failure.OUT_OF_BOUNDS_MEMORY_ACCESS, node, message);
    }

    @CompilerDirectives.TruffleBoundary
    protected static final WasmException trapUnsharedMemory(Node node) {
        String message = "Atomic wait operator can only be used on shared memories.";
        return WasmException.create(Failure.EXPECTED_SHARED_MEMORY, node, "Atomic wait operator can only be used on shared memories.");
    }

    @CompilerDirectives.TruffleBoundary
    protected static final WasmException trapOutOfBoundsBuffer(Node node, long offset, long length, long bufferSize) {
        String message = String.format(Locale.ROOT, "%d-byte buffer access at offset %d is out-of-bounds (buffer size %d bytes).", length, offset, bufferSize);
        return WasmException.create(Failure.OUT_OF_BOUNDS_MEMORY_ACCESS, node, message);
    }

    protected static final void validateAddress(Node node, long address, long length, long byteSize) {
        assert (length >= 0L);
        assert (byteSize >= 0L);
        if (address < 0L || address > byteSize - length) {
            CompilerDirectives.transferToInterpreterAndInvalidate();
            throw WasmMemory.trapOutOfBounds(node, address, length, byteSize);
        }
    }

    protected static final void validateAtomicAddress(Node node, long address, int length) {
        assert (length != 0 && (length - 1 & length) == 0) : "alignment length must be a power of two";
        if ((address & (long)(length - 1)) != 0L) {
            CompilerDirectives.transferToInterpreterAndInvalidate();
            throw WasmMemory.trapUnalignedAtomic(node, address, length);
        }
    }

    protected static final void validateLength(Node node, long length) {
        if (length < 0L) {
            CompilerDirectives.transferToInterpreterAndInvalidate();
            throw WasmMemory.trapNegativeLength(node, length);
        }
    }

    @CompilerDirectives.TruffleBoundary
    public String readString(int startOffset, WasmFunctionNode node) {
        byte currentByte;
        ByteArrayList bytes = new ByteArrayList();
        int offset = startOffset;
        while ((currentByte = (byte)WasmMemoryLibrary.getUncached().load_i32_8u(this, node, offset)) != 0) {
            bytes.add(currentByte);
            ++offset;
        }
        return new String(bytes.toArray(), StandardCharsets.UTF_8);
    }

    @CompilerDirectives.TruffleBoundary
    public final String readString(int startOffset, int length, Node node) {
        ByteArrayList bytes = new ByteArrayList();
        for (int i = 0; i < length; ++i) {
            bytes.add((byte)WasmMemoryLibrary.getUncached().load_i32_8u(this, node, startOffset + i));
        }
        return new String(bytes.toArray(), StandardCharsets.UTF_8);
    }

    @CompilerDirectives.TruffleBoundary
    public final int writeString(Node node, String string, int offset, int length) {
        int i;
        byte[] bytes = string.getBytes(StandardCharsets.UTF_8);
        for (i = 0; i < bytes.length && i < length; ++i) {
            WasmMemoryLibrary.getUncached().store_i32_8(this, node, offset + i, bytes[i]);
        }
        return i;
    }

    public final int writeString(Node node, String string, int offset) {
        return this.writeString(node, string, offset, Integer.MAX_VALUE);
    }

    @CompilerDirectives.TruffleBoundary
    public static int encodedStringLength(String string) {
        return string.getBytes(StandardCharsets.UTF_8).length;
    }

    long[] view(int address, int length) {
        long[] chunk = new long[length / 8];
        for (int p = address; p < address + length; p += 8) {
            chunk[(p - address) / 8] = WasmMemoryLibrary.getUncached().load_i64(this, null, p);
        }
        return chunk;
    }

    String viewByte(int address) {
        int value = WasmMemoryLibrary.getUncached().load_i32_8u(this, null, address);
        Object result = Integer.toHexString(value);
        if (((String)result).length() == 1) {
            result = "0" + (String)result;
        }
        return result;
    }

    public String hexView(int address, int length) {
        long[] chunk = this.view(address, length);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < chunk.length; ++i) {
            sb.append("0x").append(WasmMemory.hex((long)address + (long)i * 8L)).append(" | ");
            for (int j = 0; j < 8; ++j) {
                sb.append(this.viewByte(address + i * 8 + j)).append(" ");
            }
            sb.append("| ");
            sb.append(WasmMemory.batch(WasmMemory.hex(chunk[i]), 2)).append("\n");
        }
        return sb.toString();
    }

    private static String hex(long value) {
        return WasmMemory.pad(Long.toHexString(value), 16);
    }

    private static String batch(String s, int count) {
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < s.length(); ++i) {
            result.insert(0, s.charAt(i));
            if ((i + 1) % count != 0) continue;
            result.insert(0, " ");
        }
        return result.reverse().toString();
    }

    private static String pad(String s, int length) {
        StringBuilder padded = new StringBuilder(s);
        while (padded.length() < length) {
            padded.insert(0, "0");
        }
        return padded.toString();
    }

    @ExportMessage
    final boolean hasBufferElements() {
        return true;
    }

    @ExportMessage
    final long getBufferSize(@CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) {
        return wasmMemoryLibrary.byteSize(this);
    }

    private void checkOffset(Node node, WasmMemoryLibrary wasmMemoryLibrary, long byteOffset, int opLength, InlinedBranchProfile errorBranch) throws InvalidBufferOffsetException {
        if (opLength < 0 || byteOffset < 0L || this.getBufferSize(wasmMemoryLibrary) - (long)opLength < byteOffset) {
            errorBranch.enter(node);
            throw InvalidBufferOffsetException.create((long)byteOffset, (long)opLength);
        }
    }

    @ExportMessage
    final void readBuffer(long byteOffset, byte[] destination, int destinationOffset, int length, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, length, errorBranch);
        wasmMemoryLibrary.copyToBuffer(this, node, destination, byteOffset, destinationOffset, length);
    }

    @ExportMessage
    final byte readBufferByte(long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 1, errorBranch);
        return (byte)wasmMemoryLibrary.load_i32_8s(this, null, byteOffset);
    }

    @ExportMessage
    final short readBufferShort(ByteOrder order, long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 2, errorBranch);
        short result = (short)wasmMemoryLibrary.load_i32_16s(this, null, byteOffset);
        if (order == ByteOrder.BIG_ENDIAN) {
            result = Short.reverseBytes(result);
        }
        return result;
    }

    @ExportMessage
    final int readBufferInt(ByteOrder order, long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 4, errorBranch);
        int result = wasmMemoryLibrary.load_i32(this, null, byteOffset);
        if (order == ByteOrder.BIG_ENDIAN) {
            result = Integer.reverseBytes(result);
        }
        return result;
    }

    @ExportMessage
    final long readBufferLong(ByteOrder order, long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 8, errorBranch);
        long result = wasmMemoryLibrary.load_i64(this, null, byteOffset);
        if (order == ByteOrder.BIG_ENDIAN) {
            result = Long.reverseBytes(result);
        }
        return result;
    }

    @ExportMessage
    final float readBufferFloat(ByteOrder order, long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 4, errorBranch);
        float result = wasmMemoryLibrary.load_f32(this, null, byteOffset);
        if (order == ByteOrder.BIG_ENDIAN) {
            result = Float.intBitsToFloat(Integer.reverseBytes(Float.floatToRawIntBits(result)));
        }
        return result;
    }

    @ExportMessage
    final double readBufferDouble(ByteOrder order, long byteOffset, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 8, errorBranch);
        double result = wasmMemoryLibrary.load_f64(this, null, byteOffset);
        if (order == ByteOrder.BIG_ENDIAN) {
            result = Double.longBitsToDouble(Long.reverseBytes(Double.doubleToRawLongBits(result)));
        }
        return result;
    }

    @ExportMessage
    final boolean isBufferWritable() {
        return true;
    }

    @ExportMessage
    final void writeBufferByte(long byteOffset, byte value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 1, errorBranch);
        wasmMemoryLibrary.store_i32_8(this, null, byteOffset, value);
    }

    @ExportMessage
    final void writeBufferShort(ByteOrder order, long byteOffset, short value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 2, errorBranch);
        short actualValue = order == ByteOrder.LITTLE_ENDIAN ? value : Short.reverseBytes(value);
        wasmMemoryLibrary.store_i32_16(this, null, byteOffset, actualValue);
    }

    @ExportMessage
    final void writeBufferInt(ByteOrder order, long byteOffset, int value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 4, errorBranch);
        int actualValue = order == ByteOrder.LITTLE_ENDIAN ? value : Integer.reverseBytes(value);
        wasmMemoryLibrary.store_i32(this, null, byteOffset, actualValue);
    }

    @ExportMessage
    final void writeBufferLong(ByteOrder order, long byteOffset, long value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 8, errorBranch);
        long actualValue = order == ByteOrder.LITTLE_ENDIAN ? value : Long.reverseBytes(value);
        wasmMemoryLibrary.store_i64(this, null, byteOffset, actualValue);
    }

    @ExportMessage
    final void writeBufferFloat(ByteOrder order, long byteOffset, float value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 4, errorBranch);
        float actualValue = order == ByteOrder.LITTLE_ENDIAN ? value : Float.intBitsToFloat(Integer.reverseBytes(Float.floatToRawIntBits(value)));
        wasmMemoryLibrary.store_f32(this, null, byteOffset, actualValue);
    }

    @ExportMessage
    final void writeBufferDouble(ByteOrder order, long byteOffset, double value, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidBufferOffsetException {
        this.checkOffset(node, wasmMemoryLibrary, byteOffset, 8, errorBranch);
        double actualValue = order == ByteOrder.LITTLE_ENDIAN ? value : Double.longBitsToDouble(Long.reverseBytes(Double.doubleToRawLongBits(value)));
        wasmMemoryLibrary.store_f64(this, null, byteOffset, actualValue);
    }

    @ExportMessage
    boolean hasArrayElements() {
        return true;
    }

    @ExportMessage
    long getArraySize(@CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) {
        return wasmMemoryLibrary.byteSize(this);
    }

    @ExportMessage
    boolean isArrayElementReadable(long address, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) {
        return address >= 0L && address < this.getArraySize(wasmMemoryLibrary);
    }

    @ExportMessage
    final boolean isArrayElementModifiable(long address, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) {
        return this.isArrayElementReadable(address, wasmMemoryLibrary);
    }

    @ExportMessage
    final boolean isArrayElementInsertable(long address) {
        return false;
    }

    @ExportMessage
    public Object readArrayElement(long address, @Bind Node node, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidArrayIndexException {
        if (!this.isArrayElementReadable(address, wasmMemoryLibrary)) {
            errorBranch.enter(node);
            throw InvalidArrayIndexException.create((long)address);
        }
        return wasmMemoryLibrary.load_i32_8u(this, null, address);
    }

    @ExportMessage
    public void writeArrayElement(long address, Object value, @Bind Node node, @CachedLibrary(limit="3") InteropLibrary valueLib, @Cached.Shared(value="errorBranch") @Cached InlinedBranchProfile errorBranch, @CachedLibrary(value="this") WasmMemoryLibrary wasmMemoryLibrary) throws InvalidArrayIndexException, UnsupportedMessageException, UnsupportedTypeException {
        if (!this.isArrayElementModifiable(address, wasmMemoryLibrary)) {
            errorBranch.enter(node);
            throw InvalidArrayIndexException.create((long)address);
        }
        if (!valueLib.fitsInByte(value)) {
            errorBranch.enter(node);
            throw UnsupportedTypeException.create((Object[])new Object[]{value}, (String)"Only bytes can be stored into WebAssembly memory.");
        }
        byte rawValue = valueLib.asByte(value);
        wasmMemoryLibrary.store_i32_8(this, null, address, rawValue);
    }

    protected void invokeGrowCallback() {
        WebAssembly.invokeMemGrowCallback(this);
    }

    protected int invokeNotifyCallback(Node node, long address, int count) {
        return WebAssembly.invokeMemNotifyCallback(node, this, address, count);
    }

    protected int invokeWaitCallback(Node node, long address, long expected, long timeout, boolean is64) {
        return WebAssembly.invokeMemWaitCallback(node, this, address, expected, timeout, is64);
    }

    public final WasmMemory checkSize(WasmMemoryLibrary memoryLib, long initialSize) {
        if (memoryLib.byteSize(this) < initialSize * 65536L) {
            throw CompilerDirectives.shouldNotReachHere((String)"Memory size must not be less than initial size");
        }
        return this;
    }
}

