/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.shaded.elasticsearch7.org.elasticsearch.common;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.IsoFields;
import java.time.temporal.TemporalField;
import java.time.temporal.TemporalQueries;
import java.time.zone.ZoneOffsetTransition;
import java.time.zone.ZoneRules;
import java.util.List;
import java.util.Objects;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.ElasticsearchException;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.Version;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.common.io.stream.StreamInput;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.common.io.stream.StreamOutput;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.common.io.stream.Writeable;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.common.time.DateUtils;
import org.graylog.shaded.elasticsearch7.org.elasticsearch.common.unit.TimeValue;

public abstract class Rounding
implements Writeable {
    public abstract void innerWriteTo(StreamOutput var1) throws IOException;

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeByte(this.id());
        this.innerWriteTo(out);
    }

    public abstract byte id();

    public abstract long round(long var1);

    public abstract long nextRoundingValue(long var1);

    @Deprecated
    public abstract long offset();

    public abstract Rounding withoutOffset();

    public abstract boolean equals(Object var1);

    public abstract int hashCode();

    public static Builder builder(DateTimeUnit unit) {
        return new Builder(unit);
    }

    public static Builder builder(TimeValue interval) {
        return new Builder(interval);
    }

    public static Rounding read(StreamInput in) throws IOException {
        byte id = in.readByte();
        switch (id) {
            case 1: {
                return new TimeUnitRounding(in);
            }
            case 2: {
                return new TimeIntervalRounding(in);
            }
            case 3: {
                return new OffsetRounding(in);
            }
        }
        throw new ElasticsearchException("unknown rounding id [" + id + "]", new Object[0]);
    }

    public static class Builder {
        private final DateTimeUnit unit;
        private final long interval;
        private ZoneId timeZone = ZoneOffset.UTC;
        private long offset = 0L;

        public Builder(DateTimeUnit unit) {
            this.unit = unit;
            this.interval = -1L;
        }

        public Builder(TimeValue interval) {
            this.unit = null;
            if (interval.millis() < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval.millis();
        }

        public Builder timeZone(ZoneId timeZone) {
            if (timeZone == null) {
                throw new IllegalArgumentException("Setting null as timezone is not supported");
            }
            this.timeZone = timeZone;
            return this;
        }

        public Builder offset(long offset) {
            this.offset = offset;
            return this;
        }

        public Rounding build() {
            Rounding rounding = this.unit != null ? new TimeUnitRounding(this.unit, this.timeZone) : new TimeIntervalRounding(this.interval, this.timeZone);
            if (this.offset != 0L) {
                rounding = new OffsetRounding(rounding, this.offset);
            }
            return rounding;
        }
    }

    public static enum DateTimeUnit {
        WEEK_OF_WEEKYEAR(1, IsoFields.WEEK_OF_WEEK_BASED_YEAR){

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundWeekOfWeekYear(utcMillis);
            }
        }
        ,
        YEAR_OF_CENTURY(2, ChronoField.YEAR_OF_ERA){

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundYear(utcMillis);
            }
        }
        ,
        QUARTER_OF_YEAR(3, IsoFields.QUARTER_OF_YEAR){

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundQuarterOfYear(utcMillis);
            }
        }
        ,
        MONTH_OF_YEAR(4, ChronoField.MONTH_OF_YEAR){

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundMonthOfYear(utcMillis);
            }
        }
        ,
        DAY_OF_MONTH(5, ChronoField.DAY_OF_MONTH){
            final long unitMillis = ChronoField.DAY_OF_MONTH.getBaseUnit().getDuration().toMillis();

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundFloor(utcMillis, this.unitMillis);
            }
        }
        ,
        HOUR_OF_DAY(6, ChronoField.HOUR_OF_DAY){
            final long unitMillis = ChronoField.HOUR_OF_DAY.getBaseUnit().getDuration().toMillis();

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundFloor(utcMillis, this.unitMillis);
            }
        }
        ,
        MINUTES_OF_HOUR(7, ChronoField.MINUTE_OF_HOUR){
            final long unitMillis = ChronoField.MINUTE_OF_HOUR.getBaseUnit().getDuration().toMillis();

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundFloor(utcMillis, this.unitMillis);
            }
        }
        ,
        SECOND_OF_MINUTE(8, ChronoField.SECOND_OF_MINUTE){
            final long unitMillis = ChronoField.SECOND_OF_MINUTE.getBaseUnit().getDuration().toMillis();

            @Override
            long roundFloor(long utcMillis) {
                return DateUtils.roundFloor(utcMillis, this.unitMillis);
            }
        };

        private final byte id;
        private final TemporalField field;

        private DateTimeUnit(byte id, TemporalField field) {
            this.id = id;
            this.field = field;
        }

        abstract long roundFloor(long var1);

        public byte getId() {
            return this.id;
        }

        public TemporalField getField() {
            return this.field;
        }

        public static DateTimeUnit resolve(byte id) {
            switch (id) {
                case 1: {
                    return WEEK_OF_WEEKYEAR;
                }
                case 2: {
                    return YEAR_OF_CENTURY;
                }
                case 3: {
                    return QUARTER_OF_YEAR;
                }
                case 4: {
                    return MONTH_OF_YEAR;
                }
                case 5: {
                    return DAY_OF_MONTH;
                }
                case 6: {
                    return HOUR_OF_DAY;
                }
                case 7: {
                    return MINUTES_OF_HOUR;
                }
                case 8: {
                    return SECOND_OF_MINUTE;
                }
            }
            throw new ElasticsearchException("Unknown date time unit id [" + id + "]", new Object[0]);
        }
    }

    static class TimeUnitRounding
    extends Rounding {
        static final byte ID = 1;
        static final long TZ_OFFSET_NON_FIXED = -1L;
        private final DateTimeUnit unit;
        private final ZoneId timeZone;
        private final boolean unitRoundsToMidnight;
        private final long fixedOffsetMillis;

        TimeUnitRounding(DateTimeUnit unit, ZoneId timeZone) {
            this.unit = unit;
            this.timeZone = timeZone;
            this.unitRoundsToMidnight = this.unit.field.getBaseUnit().getDuration().toMillis() > 3600000L;
            this.fixedOffsetMillis = timeZone.getRules().isFixedOffset() ? (long)(timeZone.getRules().getOffset(Instant.EPOCH).getTotalSeconds() * 1000) : -1L;
        }

        TimeUnitRounding(StreamInput in) throws IOException {
            this(DateTimeUnit.resolve(in.readByte()), in.getVersion().onOrAfter(Version.V_7_0_0) ? in.readZoneId() : DateUtils.of(in.readString()));
        }

        @Override
        public void innerWriteTo(StreamOutput out) throws IOException {
            out.writeByte(this.unit.getId());
            if (out.getVersion().onOrAfter(Version.V_7_0_0)) {
                out.writeZoneId(this.timeZone);
            } else {
                out.writeString(DateUtils.zoneIdToDateTimeZone(this.timeZone).getID());
            }
        }

        @Override
        public byte id() {
            return 1;
        }

        private LocalDateTime truncateLocalDateTime(LocalDateTime localDateTime) {
            switch (this.unit) {
                case SECOND_OF_MINUTE: {
                    return localDateTime.withNano(0);
                }
                case MINUTES_OF_HOUR: {
                    return LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonthValue(), localDateTime.getDayOfMonth(), localDateTime.getHour(), localDateTime.getMinute(), 0, 0);
                }
                case HOUR_OF_DAY: {
                    return LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonth(), localDateTime.getDayOfMonth(), localDateTime.getHour(), 0, 0);
                }
                case DAY_OF_MONTH: {
                    LocalDate localDate = localDateTime.query(TemporalQueries.localDate());
                    return localDate.atStartOfDay();
                }
                case WEEK_OF_WEEKYEAR: {
                    return LocalDateTime.of(localDateTime.toLocalDate(), LocalTime.MIDNIGHT).with(ChronoField.DAY_OF_WEEK, 1L);
                }
                case MONTH_OF_YEAR: {
                    return LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonthValue(), 1, 0, 0);
                }
                case QUARTER_OF_YEAR: {
                    return LocalDateTime.of(localDateTime.getYear(), localDateTime.getMonth().firstMonthOfQuarter(), 1, 0, 0);
                }
                case YEAR_OF_CENTURY: {
                    return LocalDateTime.of(LocalDate.of(localDateTime.getYear(), 1, 1), LocalTime.MIDNIGHT);
                }
            }
            throw new IllegalArgumentException("NOT YET IMPLEMENTED for unit " + (Object)((Object)this.unit));
        }

        @Override
        public long round(long utcMillis) {
            if (this.fixedOffsetMillis != -1L) {
                long localMillis = utcMillis + this.fixedOffsetMillis;
                return this.unit.roundFloor(localMillis) - this.fixedOffsetMillis;
            }
            Instant instant = Instant.ofEpochMilli(utcMillis);
            if (this.unitRoundsToMidnight) {
                LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, this.timeZone);
                LocalDateTime localMidnight = this.truncateLocalDateTime(localDateTime);
                return this.firstTimeOnDay(localMidnight);
            }
            ZoneRules rules = this.timeZone.getRules();
            while (true) {
                Instant truncatedTime = this.truncateAsLocalTime(instant, rules);
                ZoneOffsetTransition previousTransition = rules.previousTransition(instant);
                if (previousTransition == null) {
                    return truncatedTime.toEpochMilli();
                }
                Instant previousTransitionInstant = previousTransition.getInstant();
                if (truncatedTime != null && previousTransitionInstant.compareTo(truncatedTime) < 1) {
                    return truncatedTime.toEpochMilli();
                }
                instant = previousTransitionInstant.minusNanos(1000000L);
            }
        }

        private long firstTimeOnDay(LocalDateTime localMidnight) {
            assert (localMidnight.toLocalTime().equals(LocalTime.of(0, 0, 0))) : "firstTimeOnDay should only be called at midnight";
            assert (this.unitRoundsToMidnight) : "firstTimeOnDay should only be called if unitRoundsToMidnight";
            List<ZoneOffset> currentOffsets = this.timeZone.getRules().getValidOffsets(localMidnight);
            if (!currentOffsets.isEmpty()) {
                ZoneOffset firstOffset = currentOffsets.get(0);
                OffsetDateTime offsetMidnight = localMidnight.atOffset(firstOffset);
                return offsetMidnight.toInstant().toEpochMilli();
            }
            ZoneOffsetTransition zoneOffsetTransition = this.timeZone.getRules().getTransition(localMidnight);
            return zoneOffsetTransition.getInstant().toEpochMilli();
        }

        private Instant truncateAsLocalTime(Instant instant, ZoneRules rules) {
            assert (!this.unitRoundsToMidnight) : "truncateAsLocalTime should not be called if unitRoundsToMidnight";
            LocalDateTime localDateTime = LocalDateTime.ofInstant(instant, this.timeZone);
            LocalDateTime truncatedLocalDateTime = this.truncateLocalDateTime(localDateTime);
            List<ZoneOffset> currentOffsets = rules.getValidOffsets(truncatedLocalDateTime);
            if (!currentOffsets.isEmpty()) {
                for (int offsetIndex = currentOffsets.size() - 1; offsetIndex >= 0; --offsetIndex) {
                    Instant result = truncatedLocalDateTime.atOffset(currentOffsets.get(offsetIndex)).toInstant();
                    if (result.isAfter(instant)) continue;
                    return result;
                }
                assert (false) : "rounded time not found for " + instant + " with " + this;
                return null;
            }
            return null;
        }

        private LocalDateTime nextRelevantMidnight(LocalDateTime localMidnight) {
            assert (localMidnight.toLocalTime().equals(LocalTime.MIDNIGHT)) : "nextRelevantMidnight should only be called at midnight";
            assert (this.unitRoundsToMidnight) : "firstTimeOnDay should only be called if unitRoundsToMidnight";
            switch (this.unit) {
                case DAY_OF_MONTH: {
                    return localMidnight.plus(1L, ChronoUnit.DAYS);
                }
                case WEEK_OF_WEEKYEAR: {
                    return localMidnight.plus(7L, ChronoUnit.DAYS);
                }
                case MONTH_OF_YEAR: {
                    return localMidnight.plus(1L, ChronoUnit.MONTHS);
                }
                case QUARTER_OF_YEAR: {
                    return localMidnight.plus(3L, ChronoUnit.MONTHS);
                }
                case YEAR_OF_CENTURY: {
                    return localMidnight.plus(1L, ChronoUnit.YEARS);
                }
            }
            throw new IllegalArgumentException("Unknown round-to-midnight unit: " + (Object)((Object)this.unit));
        }

        @Override
        public long nextRoundingValue(long utcMillis) {
            if (this.unitRoundsToMidnight) {
                LocalDateTime localDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(utcMillis), this.timeZone);
                LocalDateTime earlierLocalMidnight = this.truncateLocalDateTime(localDateTime);
                LocalDateTime localMidnight = this.nextRelevantMidnight(earlierLocalMidnight);
                return this.firstTimeOnDay(localMidnight);
            }
            long unitSize = this.unit.field.getBaseUnit().getDuration().toMillis();
            long roundedAfterOneIncrement = this.round(utcMillis + unitSize);
            if (utcMillis < roundedAfterOneIncrement) {
                return roundedAfterOneIncrement;
            }
            return this.round(utcMillis + 2L * unitSize);
        }

        @Override
        public long offset() {
            return 0L;
        }

        @Override
        public Rounding withoutOffset() {
            return this;
        }

        @Override
        public int hashCode() {
            return Objects.hash(new Object[]{this.unit, this.timeZone});
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            TimeUnitRounding other = (TimeUnitRounding)obj;
            return Objects.equals((Object)this.unit, (Object)other.unit) && Objects.equals(this.timeZone, other.timeZone);
        }

        public String toString() {
            return "Rounding[" + (Object)((Object)this.unit) + " in " + this.timeZone + "]";
        }
    }

    static class TimeIntervalRounding
    extends Rounding {
        static final byte ID = 2;
        private static final long TZ_OFFSET_NON_FIXED = -1L;
        private final long interval;
        private final ZoneId timeZone;
        private final long fixedOffsetMillis;

        TimeIntervalRounding(long interval, ZoneId timeZone) {
            if (interval < 1L) {
                throw new IllegalArgumentException("Zero or negative time interval not supported");
            }
            this.interval = interval;
            this.timeZone = timeZone;
            this.fixedOffsetMillis = timeZone.getRules().isFixedOffset() ? (long)(timeZone.getRules().getOffset(Instant.EPOCH).getTotalSeconds() * 1000) : -1L;
        }

        TimeIntervalRounding(StreamInput in) throws IOException {
            this(in.readVLong(), in.getVersion().onOrAfter(Version.V_7_0_0) ? in.readZoneId() : DateUtils.of(in.readString()));
        }

        @Override
        public void innerWriteTo(StreamOutput out) throws IOException {
            out.writeVLong(this.interval);
            if (out.getVersion().onOrAfter(Version.V_7_0_0)) {
                out.writeZoneId(this.timeZone);
            } else {
                out.writeString(DateUtils.zoneIdToDateTimeZone(this.timeZone).getID());
            }
        }

        @Override
        public byte id() {
            return 2;
        }

        @Override
        public long round(long utcMillis) {
            if (this.fixedOffsetMillis != -1L) {
                long localMillis = utcMillis + this.fixedOffsetMillis;
                return TimeIntervalRounding.roundKey(localMillis, this.interval) * this.interval - this.fixedOffsetMillis;
            }
            Instant utcInstant = Instant.ofEpochMilli(utcMillis);
            LocalDateTime rawLocalDateTime = LocalDateTime.ofInstant(utcInstant, this.timeZone);
            long localMillis = utcMillis + (long)(this.timeZone.getRules().getOffset(utcInstant).getTotalSeconds() * 1000);
            assert (localMillis == rawLocalDateTime.toInstant(ZoneOffset.UTC).toEpochMilli());
            long roundedMillis = TimeIntervalRounding.roundKey(localMillis, this.interval) * this.interval;
            LocalDateTime roundedLocalDateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(roundedMillis), ZoneOffset.UTC);
            List<ZoneOffset> currentOffsets = this.timeZone.getRules().getValidOffsets(roundedLocalDateTime);
            if (!currentOffsets.isEmpty()) {
                ZoneOffsetTransition previousTransition = this.timeZone.getRules().previousTransition(utcInstant.plusMillis(1L));
                for (int offsetIndex = currentOffsets.size() - 1; 0 <= offsetIndex; --offsetIndex) {
                    OffsetDateTime offsetTime = roundedLocalDateTime.atOffset(currentOffsets.get(offsetIndex));
                    Instant offsetInstant = offsetTime.toInstant();
                    if (previousTransition != null && offsetInstant.isBefore(previousTransition.getInstant())) {
                        return this.round(previousTransition.getInstant().toEpochMilli() - 1L);
                    }
                    if (utcInstant.isBefore(offsetTime.toInstant())) continue;
                    return offsetInstant.toEpochMilli();
                }
                OffsetDateTime offsetTime = roundedLocalDateTime.atOffset(currentOffsets.get(0));
                Instant offsetInstant = offsetTime.toInstant();
                assert (false) : this + " failed to round " + utcMillis + " down: " + offsetInstant + " is the earliest possible";
                return offsetInstant.toEpochMilli();
            }
            ZoneOffsetTransition zoneOffsetTransition = this.timeZone.getRules().getTransition(roundedLocalDateTime);
            return zoneOffsetTransition.getInstant().toEpochMilli();
        }

        private static long roundKey(long value, long interval) {
            if (value < 0L) {
                return (value - interval + 1L) / interval;
            }
            return value / interval;
        }

        @Override
        public long nextRoundingValue(long time) {
            int offsetSeconds = this.timeZone.getRules().getOffset(Instant.ofEpochMilli(time)).getTotalSeconds();
            long millis = time + this.interval + (long)(offsetSeconds * 1000);
            return ZonedDateTime.ofInstant(Instant.ofEpochMilli(millis), ZoneOffset.UTC).withZoneSameLocal(this.timeZone).toInstant().toEpochMilli();
        }

        @Override
        public long offset() {
            return 0L;
        }

        @Override
        public Rounding withoutOffset() {
            return this;
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.interval, this.timeZone);
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            TimeIntervalRounding other = (TimeIntervalRounding)obj;
            return Objects.equals(this.interval, other.interval) && Objects.equals(this.timeZone, other.timeZone);
        }

        public String toString() {
            return "Rounding[" + this.interval + " in " + this.timeZone + "]";
        }
    }

    static class OffsetRounding
    extends Rounding {
        static final byte ID = 3;
        private final Rounding delegate;
        private final long offset;

        OffsetRounding(Rounding delegate, long offset) {
            this.delegate = delegate;
            this.offset = offset;
        }

        OffsetRounding(StreamInput in) throws IOException {
            this.delegate = Rounding.read(in);
            this.offset = in.readZLong();
        }

        @Override
        public void innerWriteTo(StreamOutput out) throws IOException {
            if (out.getVersion().before(Version.V_7_6_0)) {
                throw new IllegalArgumentException("Offset rounding not supported before 7.6.0");
            }
            this.delegate.writeTo(out);
            out.writeZLong(this.offset);
        }

        @Override
        public byte id() {
            return 3;
        }

        @Override
        public long round(long value) {
            return this.delegate.round(value - this.offset) + this.offset;
        }

        @Override
        public long nextRoundingValue(long value) {
            return this.delegate.nextRoundingValue(value - this.offset) + this.offset;
        }

        @Override
        public long offset() {
            return this.offset;
        }

        @Override
        public Rounding withoutOffset() {
            return this.delegate;
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.delegate, this.offset);
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            OffsetRounding other = (OffsetRounding)obj;
            return this.delegate.equals(other.delegate) && this.offset == other.offset;
        }

        public String toString() {
            return this.delegate + " offset by " + this.offset;
        }
    }
}

