/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.elasticsearch.action.admin.indices.stats.IndexStats;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.indices.InvalidAliasNameException;
import org.graylog2.indexer.NoTargetIndexException;
import org.graylog2.indexer.SetIndexReadOnlyJob;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.ranges.CreateNewSingleIndexRangeJob;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJobConcurrencyException;
import org.graylog2.system.jobs.SystemJobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Deflector {
    private static final Logger LOG = LoggerFactory.getLogger(Deflector.class);
    public static final String DEFLECTOR_SUFFIX = "deflector";
    public static final String SEPARATOR = "_";
    public static final String RESTORED_ARCHIVE_SUFFIX = "_restored_archive";
    private final SystemJobManager systemJobManager;
    private final ActivityWriter activityWriter;
    private final CreateNewSingleIndexRangeJob.Factory createNewSingleIndexRangeJobFactory;
    private final String indexPrefix;
    private final String deflectorName;
    private final Indices indices;
    private final SetIndexReadOnlyJob.Factory indexReadOnlyJobFactory;
    private final Pattern deflectorIndexPattern;
    private final Pattern indexPattern;

    @Inject
    public Deflector(SystemJobManager systemJobManager, @Named(value="elasticsearch_index_prefix") String indexPrefix, ActivityWriter activityWriter, SetIndexReadOnlyJob.Factory indexReadOnlyJobFactory, CreateNewSingleIndexRangeJob.Factory createNewSingleIndexRangeJobFactory, Indices indices) {
        this.indexPrefix = indexPrefix;
        this.systemJobManager = systemJobManager;
        this.activityWriter = activityWriter;
        this.indexReadOnlyJobFactory = indexReadOnlyJobFactory;
        this.createNewSingleIndexRangeJobFactory = createNewSingleIndexRangeJobFactory;
        this.deflectorName = Deflector.buildName(indexPrefix);
        this.indices = indices;
        this.deflectorIndexPattern = Pattern.compile("^" + indexPrefix + SEPARATOR + "\\d+");
        this.indexPattern = Pattern.compile("^" + indexPrefix + SEPARATOR + "\\d+(?:" + RESTORED_ARCHIVE_SUFFIX + ")?");
    }

    public boolean isUp() {
        return this.indices.aliasExists(this.getName());
    }

    public void setUp() {
        if (this.isUp()) {
            LOG.info("Found deflector alias <{}>. Using it.", (Object)this.getName());
        } else {
            LOG.info("Did not find an deflector alias. Setting one up now.");
            try {
                String currentTarget = this.getNewestTargetName();
                LOG.info("Pointing to already existing index target <{}>", (Object)currentTarget);
                this.pointTo(currentTarget);
            }
            catch (NoTargetIndexException ex) {
                String msg = "There is no index target to point to. Creating one now.";
                LOG.info("There is no index target to point to. Creating one now.");
                this.activityWriter.write(new Activity("There is no index target to point to. Creating one now.", Deflector.class));
                this.cycle();
            }
            catch (InvalidAliasNameException e) {
                LOG.error("Seems like there already is an index called [{}]", (Object)this.getName());
            }
        }
    }

    public void cycle() {
        int oldTargetNumber;
        LOG.info("Cycling deflector to next index now.");
        try {
            oldTargetNumber = this.getNewestTargetNumber();
        }
        catch (NoTargetIndexException ex) {
            oldTargetNumber = -1;
        }
        int newTargetNumber = oldTargetNumber + 1;
        String newTarget = Deflector.buildIndexName(this.indexPrefix, newTargetNumber);
        String oldTarget = Deflector.buildIndexName(this.indexPrefix, oldTargetNumber);
        if (oldTargetNumber == -1) {
            LOG.info("Cycling from <none> to <{}>", (Object)newTarget);
        } else {
            LOG.info("Cycling from <{}> to <{}>", (Object)oldTarget, (Object)newTarget);
        }
        LOG.info("Creating index target <{}>...", (Object)newTarget);
        if (!this.indices.create(newTarget)) {
            LOG.error("Could not properly create new target <{}>", (Object)newTarget);
        }
        LOG.info("Waiting for index allocation of <{}>", (Object)newTarget);
        ClusterHealthStatus healthStatus = this.indices.waitForRecovery(newTarget);
        LOG.debug("Health status of index <{}>: {}", (Object)newTarget, (Object)healthStatus);
        LOG.info("Done!");
        LOG.info("Pointing deflector to new target index....");
        Activity activity = new Activity(Deflector.class);
        if (oldTargetNumber == -1) {
            this.pointTo(newTarget);
            activity.setMessage("Cycled deflector from <none> to <" + newTarget + ">");
        } else {
            this.pointTo(newTarget, oldTarget);
            this.addSingleIndexRanges(oldTarget);
            SetIndexReadOnlyJob makeReadOnlyJob = this.indexReadOnlyJobFactory.create(oldTarget);
            try {
                this.systemJobManager.submitWithDelay(makeReadOnlyJob, 30L, TimeUnit.SECONDS);
            }
            catch (SystemJobConcurrencyException e) {
                LOG.error("Cannot set index <" + oldTarget + "> to read only. It won't be optimized.", (Throwable)e);
            }
            activity.setMessage("Cycled deflector from <" + oldTarget + "> to <" + newTarget + ">");
        }
        this.addSingleIndexRanges(newTarget);
        LOG.info("Done!");
        this.activityWriter.write(activity);
    }

    public int getNewestTargetNumber() throws NoTargetIndexException {
        Map<String, IndexStats> indices = this.indices.getAll();
        if (indices.isEmpty()) {
            throw new NoTargetIndexException();
        }
        ArrayList indexNumbers = Lists.newArrayListWithExpectedSize((int)indices.size());
        for (String indexName : indices.keySet()) {
            if (!this.isGraylogDeflectorIndex(indexName)) continue;
            try {
                indexNumbers.add(Deflector.extractIndexNumber(indexName));
            }
            catch (NumberFormatException ex) {
                LOG.debug("Couldn't extract index number from index name " + indexName, (Throwable)ex);
            }
        }
        if (indexNumbers.isEmpty()) {
            throw new NoTargetIndexException();
        }
        return (Integer)Collections.max(indexNumbers);
    }

    public String[] getAllGraylogIndexNames() {
        Map<String, IndexStats> indices = this.indices.getAll();
        ArrayList result = Lists.newArrayListWithExpectedSize((int)indices.size());
        for (String indexName : indices.keySet()) {
            if (!this.isGraylogIndex(indexName)) continue;
            result.add(indexName);
        }
        return result.toArray(new String[result.size()]);
    }

    public Map<String, IndexStats> getAllGraylogDeflectorIndices() {
        ImmutableMap.Builder result = ImmutableMap.builder();
        for (Map.Entry<String, IndexStats> e : this.indices.getAll().entrySet()) {
            String name = e.getKey();
            if (!this.isGraylogDeflectorIndex(name)) continue;
            result.put((Object)name, (Object)e.getValue());
        }
        return result.build();
    }

    public String getNewestTargetName() throws NoTargetIndexException {
        return Deflector.buildIndexName(this.indexPrefix, this.getNewestTargetNumber());
    }

    public static String buildIndexName(String prefix, int number) {
        return prefix + SEPARATOR + number;
    }

    public static String buildName(String prefix) {
        return prefix + SEPARATOR + DEFLECTOR_SUFFIX;
    }

    public static int extractIndexNumber(String indexName) throws NumberFormatException {
        String[] parts = indexName.split(SEPARATOR);
        try {
            return Integer.parseInt(parts[parts.length - 1]);
        }
        catch (Exception e) {
            LOG.debug("Could not extract index number from index <" + indexName + ">.", (Throwable)e);
            throw new NumberFormatException();
        }
    }

    public void pointTo(String newIndex, String oldIndex) {
        this.indices.cycleAlias(this.getName(), newIndex, oldIndex);
    }

    public void pointTo(String newIndex) {
        this.indices.cycleAlias(this.getName(), newIndex);
    }

    private void addSingleIndexRanges(String indexName) {
        try {
            this.systemJobManager.submit(this.createNewSingleIndexRangeJobFactory.create(this, indexName));
        }
        catch (SystemJobConcurrencyException e) {
            String msg = "Could not calculate index ranges for index " + indexName + " after cycling deflector: Maximum concurrency of job is reached.";
            this.activityWriter.write(new Activity(msg, Deflector.class));
            LOG.error(msg, (Throwable)e);
        }
    }

    @Nullable
    public String getCurrentActualTargetIndex() {
        return this.indices.aliasTarget(this.getName());
    }

    public String getName() {
        return this.deflectorName;
    }

    public String getDeflectorWildcard() {
        return this.indexPrefix + SEPARATOR + "*";
    }

    public boolean isDeflectorAlias(String indexName) {
        return this.getName().equals(indexName);
    }

    public boolean isGraylogDeflectorIndex(String indexName) {
        return !Strings.isNullOrEmpty((String)indexName) && !this.isDeflectorAlias(indexName) && this.deflectorIndexPattern.matcher(indexName).matches();
    }

    public boolean isGraylogIndex(String indexName) {
        return !Strings.isNullOrEmpty((String)indexName) && !this.isDeflectorAlias(indexName) && this.indexPattern.matcher(indexName).matches();
    }
}

