/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.outputs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.inject.assistedinject.Assisted;
import java.io.File;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.Nullable;
import javax.inject.Inject;
import org.graylog2.gelfclient.GelfConfiguration;
import org.graylog2.gelfclient.GelfMessage;
import org.graylog2.gelfclient.GelfMessageBuilder;
import org.graylog2.gelfclient.GelfMessageLevel;
import org.graylog2.gelfclient.GelfTransports;
import org.graylog2.gelfclient.transport.GelfTransport;
import org.graylog2.plugin.Message;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.configuration.Configuration;
import org.graylog2.plugin.configuration.ConfigurationRequest;
import org.graylog2.plugin.configuration.fields.BooleanField;
import org.graylog2.plugin.configuration.fields.ConfigurationField;
import org.graylog2.plugin.configuration.fields.DropdownField;
import org.graylog2.plugin.configuration.fields.NumberField;
import org.graylog2.plugin.configuration.fields.TextField;
import org.graylog2.plugin.outputs.MessageOutput;
import org.graylog2.plugin.outputs.MessageOutputConfigurationException;
import org.graylog2.plugin.streams.Stream;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GelfOutput
implements MessageOutput {
    private static final Logger LOG = LoggerFactory.getLogger(GelfOutput.class);
    private static final String CK_PROTOCOL = "protocol";
    private static final String CK_HOSTNAME = "hostname";
    private static final String CK_PORT = "port";
    private static final String CK_CONNECT_TIMEOUT = "connect_timeout";
    private static final String CK_RECONNECT_DELAY = "reconnect_delay";
    private static final String CK_TCP_NO_DELAY = "tcp_no_delay";
    private static final String CK_TCP_KEEP_ALIVE = "tcp_keep_alive";
    private static final String CK_TLS_VERIFICATION_ENABLED = "tls_verification_enabled";
    private static final String CK_TLS_TRUST_CERT_CHAIN = "tls_trust_cert_chain";
    private final AtomicBoolean isRunning = new AtomicBoolean(false);
    private final GelfTransport transport;

    @Inject
    public GelfOutput(@Assisted Configuration configuration) throws MessageOutputConfigurationException {
        this(GelfOutput.buildTransport(configuration));
    }

    @VisibleForTesting
    GelfOutput(GelfTransport gelfTransport) {
        this.transport = (GelfTransport)Preconditions.checkNotNull((Object)gelfTransport);
        this.isRunning.set(true);
    }

    @Override
    public void stop() {
        LOG.debug("Stopping {}", (Object)this.transport.getClass().getName());
        try {
            this.transport.stop();
        }
        catch (Exception e) {
            LOG.error("Error stopping " + this.transport.getClass().getName(), (Throwable)e);
        }
        this.isRunning.set(false);
    }

    @Override
    public boolean isRunning() {
        return this.isRunning.get();
    }

    protected static GelfTransport buildTransport(Configuration configuration) throws MessageOutputConfigurationException {
        File tlsTrustCertChainFile;
        boolean tlsEnabled;
        GelfTransports transport;
        String protocol = configuration.getString(CK_PROTOCOL);
        String hostname = configuration.getString(CK_HOSTNAME);
        int port = configuration.getInt(CK_PORT);
        int connectTimeout = configuration.getInt(CK_CONNECT_TIMEOUT, 1000);
        int reconnectDelay = configuration.getInt(CK_RECONNECT_DELAY, 500);
        boolean tcpKeepAlive = configuration.getBoolean(CK_TCP_KEEP_ALIVE, false);
        boolean tcpNoDelay = configuration.getBoolean(CK_TCP_NO_DELAY, false);
        boolean tlsVerificationEnabled = configuration.getBoolean(CK_TLS_VERIFICATION_ENABLED, false);
        String tlsTrustCertChain = configuration.getString(CK_TLS_TRUST_CERT_CHAIN);
        if (Strings.isNullOrEmpty((String)protocol) || Strings.isNullOrEmpty((String)hostname) || !configuration.intIsSet(CK_PORT)) {
            throw new MessageOutputConfigurationException("Protocol and/or hostname missing!");
        }
        switch (protocol.toUpperCase(Locale.ENGLISH)) {
            case "UDP": {
                transport = GelfTransports.UDP;
                tlsEnabled = false;
                break;
            }
            case "TCP": {
                transport = GelfTransports.TCP;
                tlsEnabled = false;
                break;
            }
            case "TCP+TLS": {
                transport = GelfTransports.TCP;
                tlsEnabled = true;
                break;
            }
            default: {
                throw new MessageOutputConfigurationException("Unknown protocol " + protocol);
            }
        }
        if (tlsEnabled && !Strings.isNullOrEmpty((String)tlsTrustCertChain)) {
            tlsTrustCertChainFile = new File(tlsTrustCertChain);
            if (!tlsTrustCertChainFile.isFile() && !tlsTrustCertChainFile.canRead()) {
                throw new MessageOutputConfigurationException("TLS trust certificate chain file cannot be read!");
            }
        } else {
            tlsTrustCertChainFile = null;
        }
        GelfConfiguration gelfConfiguration = new GelfConfiguration(hostname, port).transport(transport).connectTimeout(connectTimeout).reconnectDelay(reconnectDelay).tcpKeepAlive(tcpKeepAlive).tcpNoDelay(tcpNoDelay);
        if (tlsEnabled) {
            gelfConfiguration.enableTls();
            if (tlsVerificationEnabled) {
                gelfConfiguration.enableTlsCertVerification();
            } else {
                gelfConfiguration.disableTlsCertVerification();
            }
            if (tlsTrustCertChainFile != null) {
                gelfConfiguration.tlsTrustCertChainFile(tlsTrustCertChainFile);
            }
        }
        LOG.debug("Initializing GELF sender and connecting to {}://{}:{}", new Object[]{protocol, hostname, port});
        try {
            return GelfTransports.create((GelfConfiguration)gelfConfiguration);
        }
        catch (Exception e) {
            String error = "Error initializing " + GelfOutput.class;
            LOG.error(error, (Throwable)e);
            throw new MessageOutputConfigurationException(error);
        }
    }

    @Override
    public void write(Message message) throws Exception {
        this.transport.send(this.toGELFMessage(message));
    }

    @Override
    public void write(List<Message> messages) throws Exception {
        for (Message message : messages) {
            this.write(message);
        }
    }

    @Nullable
    private GelfMessageLevel extractLevel(Object rawLevel) {
        Object level;
        if (rawLevel instanceof Number) {
            int numericLevel = ((Number)rawLevel).intValue();
            level = this.extractLevel(numericLevel);
        } else if (rawLevel instanceof String) {
            Integer numericLevel;
            try {
                numericLevel = Integer.parseInt((String)rawLevel);
            }
            catch (NumberFormatException e) {
                LOG.debug("Invalid message level " + rawLevel, (Throwable)e);
                numericLevel = null;
            }
            level = numericLevel == null ? null : this.extractLevel(numericLevel);
        } else {
            LOG.debug("Invalid message level {}", rawLevel);
            level = null;
        }
        return level;
    }

    @Nullable
    private GelfMessageLevel extractLevel(int numericLevel) {
        GelfMessageLevel level;
        try {
            level = GelfMessageLevel.fromNumericLevel((int)numericLevel);
        }
        catch (IllegalArgumentException e) {
            LOG.debug("Invalid numeric message level " + numericLevel, (Throwable)e);
            level = null;
        }
        return level;
    }

    protected GelfMessage toGELFMessage(Message message) {
        Object fieldTimeStamp = message.getField("timestamp");
        DateTime timestamp = fieldTimeStamp instanceof DateTime ? (DateTime)fieldTimeStamp : Tools.nowUTC();
        GelfMessageLevel messageLevel = this.extractLevel(message.getField("level"));
        String fullMessage = (String)message.getField("full_message");
        String facility = (String)message.getField("facility");
        String forwarder = GelfOutput.class.getCanonicalName();
        GelfMessageBuilder builder = new GelfMessageBuilder(message.getMessage(), message.getSource()).timestamp((double)timestamp.getMillis() / 1000.0).additionalField("_forwarder", (Object)forwarder).additionalFields(message.getFields());
        if (messageLevel != null) {
            builder.level(messageLevel);
        }
        if (fullMessage != null) {
            builder.fullMessage(fullMessage);
        }
        if (facility != null) {
            builder.additionalField("_facility", (Object)facility);
        }
        return builder.build();
    }

    public static class Descriptor
    extends MessageOutput.Descriptor {
        public Descriptor() {
            super("GELF Output", false, "", "An output sending GELF over TCP or UDP");
        }
    }

    public static class Config
    extends MessageOutput.Config {
        @Override
        public ConfigurationRequest getRequestedConfiguration() {
            ImmutableMap protocols = ImmutableMap.of((Object)"TCP", (Object)"TCP", (Object)"TCP+TLS", (Object)"TCP+TLS", (Object)"UDP", (Object)"UDP");
            ConfigurationRequest configurationRequest = new ConfigurationRequest();
            configurationRequest.addField(new TextField(GelfOutput.CK_HOSTNAME, "Destination host", "", "This is the hostname of the destination", ConfigurationField.Optional.NOT_OPTIONAL));
            configurationRequest.addField(new NumberField(GelfOutput.CK_PORT, "Destination port", 12201, "This is the port of the destination", ConfigurationField.Optional.NOT_OPTIONAL, NumberField.Attribute.IS_PORT_NUMBER));
            configurationRequest.addField(new DropdownField(GelfOutput.CK_PROTOCOL, "Protocol", "TCP", (Map<String, String>)protocols, "The protocol used to connect", ConfigurationField.Optional.NOT_OPTIONAL));
            configurationRequest.addField(new NumberField(GelfOutput.CK_CONNECT_TIMEOUT, "TCP Connect Timeout", 1000, "Connection timeout for TCP connections in milliseconds", ConfigurationField.Optional.OPTIONAL, NumberField.Attribute.ONLY_POSITIVE));
            configurationRequest.addField(new NumberField(GelfOutput.CK_RECONNECT_DELAY, "TCP Reconnect Delay", 500, "Time to wait between reconnects in milliseconds", ConfigurationField.Optional.OPTIONAL, NumberField.Attribute.ONLY_POSITIVE));
            configurationRequest.addField(new BooleanField(GelfOutput.CK_TCP_NO_DELAY, "TCP No Delay", false, "Whether to use Nagle's algorithm for TCP connections"));
            configurationRequest.addField(new BooleanField(GelfOutput.CK_TCP_KEEP_ALIVE, "TCP Keep Alive", false, "Whether to send TCP keep alive packets"));
            configurationRequest.addField(new BooleanField(GelfOutput.CK_TLS_VERIFICATION_ENABLED, "TLS verification", false, "Whether to verify peers when using TLS"));
            configurationRequest.addField(new TextField(GelfOutput.CK_TLS_TRUST_CERT_CHAIN, "TLS Trust Certificate Chain", "", "Local file which contains the trust certificate chain", ConfigurationField.Optional.OPTIONAL));
            return configurationRequest;
        }
    }

    public static interface Factory
    extends MessageOutput.Factory<GelfOutput> {
        @Override
        public GelfOutput create(Stream var1, Configuration var2);

        @Override
        public Config getConfig();

        @Override
        public Descriptor getDescriptor();
    }
}

