/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.security.ldap;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.SimpleTimeLimiter;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.common.util.concurrent.UncheckedTimeoutException;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.apache.directory.api.ldap.model.cursor.CursorException;
import org.apache.directory.api.ldap.model.cursor.EntryCursor;
import org.apache.directory.api.ldap.model.entry.Attribute;
import org.apache.directory.api.ldap.model.entry.Entry;
import org.apache.directory.api.ldap.model.entry.Value;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.exception.LdapInvalidDnException;
import org.apache.directory.api.ldap.model.message.BindRequest;
import org.apache.directory.api.ldap.model.message.BindRequestImpl;
import org.apache.directory.api.ldap.model.message.BindResponse;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.api.ldap.model.message.SearchScope;
import org.apache.directory.api.ldap.model.name.Dn;
import org.apache.directory.ldap.client.api.LdapConnectionConfig;
import org.apache.directory.ldap.client.api.LdapNetworkConnection;
import org.graylog2.shared.security.ldap.LdapEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapConnector {
    private static final Logger LOG = LoggerFactory.getLogger(LdapConnector.class);
    private static final String ATTRIBUTE_UNIQUE_MEMBER = "uniqueMember";
    private static final String ATTRIBUTE_MEMBER = "member";
    private static final String ATTRIBUTE_MEMBER_UID = "memberUid";
    private final int connectionTimeout;

    @Inject
    public LdapConnector(@Named(value="ldap_connection_timeout") int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public LdapNetworkConnection connect(LdapConnectionConfig config) throws LdapException {
        final LdapNetworkConnection connection = new LdapNetworkConnection(config);
        connection.setTimeOut((long)this.connectionTimeout);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Connecting to LDAP server {}:{}, binding with user {}", new Object[]{config.getLdapHost(), config.getLdapPort(), config.getName()});
        }
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("ldap-connector-%d").build();
        SimpleTimeLimiter timeLimiter = new SimpleTimeLimiter(Executors.newSingleThreadExecutor(threadFactory));
        Callable timeLimitedConnection = (Callable)timeLimiter.newProxy((Object)new Callable<Boolean>(){

            @Override
            public Boolean call() throws Exception {
                return connection.connect();
            }
        }, Callable.class, (long)this.connectionTimeout, TimeUnit.MILLISECONDS);
        try {
            Boolean connected = (Boolean)timeLimitedConnection.call();
            if (!connected.booleanValue()) {
                return null;
            }
        }
        catch (UncheckedTimeoutException e) {
            LOG.error("Timed out connecting to LDAP server", (Throwable)e);
            throw new LdapException("Could not connect to LDAP server", e.getCause());
        }
        catch (LdapException e) {
            throw e;
        }
        catch (Exception e) {
            throw new LdapException("Unexpected error connecting to LDAP", (Throwable)e);
        }
        connection.bind();
        return connection;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public LdapEntry search(LdapNetworkConnection connection, String searchBase, String searchPattern, String displayNameAttribute, String principal, boolean activeDirectory, String groupSearchBase, String groupIdAttribute, String groupSearchPattern) throws LdapException, CursorException {
        LdapEntry ldapEntry = new LdapEntry();
        HashSet groupDns = Sets.newHashSet();
        String filter = MessageFormat.format(searchPattern, this.sanitizePrincipal(principal));
        if (LOG.isTraceEnabled()) {
            LOG.trace("Search {} for {}, starting at {}", new Object[]{activeDirectory ? "ActiveDirectory" : "LDAP", filter, searchBase});
        }
        try (EntryCursor entryCursor = null;){
            entryCursor = connection.search(searchBase, filter, SearchScope.SUBTREE, new String[]{groupIdAttribute, displayNameAttribute, "dn", "uid", "userPrincipalName", "mail", "rfc822Mailbox", "memberOf", "isMemberOf"});
            Iterator it = entryCursor.iterator();
            if (it.hasNext()) {
                Entry e = (Entry)it.next();
                ldapEntry.setDn(e.getDn().getName());
                if (!activeDirectory) {
                    ldapEntry.setBindPrincipal(e.getDn().getName());
                }
                for (Attribute attribute : e.getAttributes()) {
                    if (activeDirectory && "userPrincipalName".equalsIgnoreCase(attribute.getId())) {
                        ldapEntry.setBindPrincipal(attribute.getString());
                    }
                    if (attribute.isHumanReadable()) {
                        ldapEntry.put(attribute.getId(), Joiner.on((String)", ").join(attribute.iterator()));
                    }
                    if (!"memberOf".equalsIgnoreCase(attribute.getId()) && !"isMemberOf".equalsIgnoreCase(attribute.getId())) continue;
                    for (Value group : attribute) {
                        groupDns.add(group.getString());
                    }
                }
            } else {
                LOG.trace("No LDAP entry found for filter {}", (Object)filter);
                LdapEntry e = null;
                return e;
            }
            if (!(groupDns.isEmpty() || Strings.isNullOrEmpty((String)groupSearchBase) || Strings.isNullOrEmpty((String)groupIdAttribute))) {
                try {
                    for (String groupDn : groupDns) {
                        LOG.trace("Looking up group {}", (Object)groupDn);
                        try {
                            Entry group = connection.lookup(groupDn, new String[]{groupIdAttribute});
                            if (group != null) {
                                Attribute groupId = group.get(groupIdAttribute);
                                LOG.trace("Resolved {} to group {}", (Object)groupDn, (Object)groupId);
                                if (groupId == null) continue;
                                String string = groupId.getString();
                                ldapEntry.addGroups(Collections.singleton(string));
                                continue;
                            }
                            LOG.debug("Unable to lookup group: {}", (Object)groupDn);
                        }
                        catch (LdapException e) {
                            LOG.warn("Error while looking up group " + groupDn, (Throwable)e);
                        }
                    }
                }
                catch (Exception e) {
                    LOG.error("Unexpected error during LDAP group resolution", (Throwable)e);
                }
            }
            if (ldapEntry.getGroups().isEmpty() && !Strings.isNullOrEmpty((String)groupSearchBase) && !Strings.isNullOrEmpty((String)groupIdAttribute) && !Strings.isNullOrEmpty((String)groupSearchPattern)) {
                ldapEntry.addGroups(this.findGroups(connection, groupSearchBase, groupSearchPattern, groupIdAttribute, ldapEntry));
                LOG.trace("LDAP search found entry for DN {} with search filter {}: {}", new Object[]{ldapEntry.getDn(), filter, ldapEntry});
            } else if (groupDns.isEmpty()) {
                LOG.info("LDAP group search base, id attribute or object class missing, not iterating over LDAP groups.");
            }
            LdapEntry ldapEntry2 = ldapEntry;
            return ldapEntry2;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<String> findGroups(LdapNetworkConnection connection, String groupSearchBase, String groupSearchPattern, String groupIdAttribute, @Nullable LdapEntry ldapEntry) {
        HashSet groups = Sets.newHashSet();
        try (EntryCursor groupSearch = null;){
            groupSearch = connection.search(groupSearchBase, groupSearchPattern, SearchScope.SUBTREE, new String[]{"objectClass", ATTRIBUTE_UNIQUE_MEMBER, ATTRIBUTE_MEMBER, ATTRIBUTE_MEMBER_UID, groupIdAttribute});
            LOG.trace("LDAP search for groups: {} starting at {}", (Object)groupSearchPattern, (Object)groupSearchBase);
            for (Entry e : groupSearch) {
                Attribute members;
                String memberAttribute;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Group Entry: {}", (Object)e.toString("  "));
                }
                if (!e.containsAttribute(new String[]{groupIdAttribute})) {
                    LOG.warn("Unknown group id attribute {}, skipping group entry {}", (Object)groupIdAttribute, (Object)e);
                    continue;
                }
                String groupId = e.get(groupIdAttribute).getString();
                if (ldapEntry == null) {
                    groups.add(groupId);
                    continue;
                }
                if (e.hasObjectClass(new String[]{"groupOfUniqueNames"})) {
                    memberAttribute = ATTRIBUTE_UNIQUE_MEMBER;
                } else if (e.hasObjectClass(new String[]{"groupOfNames"}) || e.hasObjectClass(new String[]{"group"})) {
                    memberAttribute = ATTRIBUTE_MEMBER;
                } else if (e.hasObjectClass(new String[]{"posixGroup"})) {
                    memberAttribute = ATTRIBUTE_MEMBER_UID;
                } else {
                    memberAttribute = e.containsAttribute(new String[]{ATTRIBUTE_UNIQUE_MEMBER}) ? ATTRIBUTE_UNIQUE_MEMBER : (e.containsAttribute(new String[]{ATTRIBUTE_MEMBER_UID}) ? ATTRIBUTE_MEMBER_UID : ATTRIBUTE_MEMBER);
                    LOG.warn("Unable to auto-detect the LDAP group object class, assuming '{}' is the correct attribute.", (Object)memberAttribute);
                }
                if ((members = e.get(memberAttribute)) == null) continue;
                String dn = this.normalizedDn(ldapEntry.getDn());
                String uid = ldapEntry.get("uid");
                for (Value member : members) {
                    LOG.trace("DN {} == {} member?", (Object)dn, (Object)member.getString());
                    if (dn.equalsIgnoreCase(this.normalizedDn(member.getString()))) {
                        groups.add(groupId);
                        continue;
                    }
                    if (Strings.isNullOrEmpty((String)uid) || !uid.equalsIgnoreCase(member.getString())) continue;
                    LOG.trace("UID {} == {} member?", (Object)uid, (Object)member.getString());
                    groups.add(groupId);
                }
            }
        }
        return groups;
    }

    @Nullable
    private String normalizedDn(String dn) {
        if (Strings.isNullOrEmpty((String)dn)) {
            return dn;
        }
        try {
            return new Dn(new String[]{dn}).getNormName();
        }
        catch (LdapInvalidDnException e) {
            LOG.warn("Invalid DN", (Throwable)e);
            return dn;
        }
    }

    public Set<String> listGroups(LdapNetworkConnection connection, String groupSearchBase, String groupSearchPattern, String groupIdAttribute) {
        return this.findGroups(connection, groupSearchBase, groupSearchPattern, groupIdAttribute, null);
    }

    private String sanitizePrincipal(String input) {
        String s = "";
        for (int i = 0; i < input.length(); ++i) {
            byte[] utf8bytes;
            char c = input.charAt(i);
            if (c == '*') {
                s = s + "\\2a";
                continue;
            }
            if (c == '(') {
                s = s + "\\28";
                continue;
            }
            if (c == ')') {
                s = s + "\\29";
                continue;
            }
            if (c == '\\') {
                s = s + "\\5c";
                continue;
            }
            if (c == '\u0000') {
                s = s + "\\00";
                continue;
            }
            if (c <= '\u007f') {
                s = s + String.valueOf(c);
                continue;
            }
            if (c < '\u0080') continue;
            for (byte b : utf8bytes = String.valueOf(c).getBytes(StandardCharsets.UTF_8)) {
                s = s + String.format(Locale.ENGLISH, "\\%02x", b);
            }
        }
        return s;
    }

    public boolean authenticate(LdapNetworkConnection connection, String principal, String credentials) throws LdapException {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)principal) ? 1 : 0) != 0, (Object)"Binding with empty principal is forbidden.");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)credentials) ? 1 : 0) != 0, (Object)"Binding with empty credentials is forbidden.");
        BindRequestImpl bindRequest = new BindRequestImpl();
        bindRequest.setName(principal);
        bindRequest.setCredentials(credentials);
        LOG.trace("Re-binding with DN {} using password", (Object)principal);
        BindResponse bind = connection.bind((BindRequest)bindRequest);
        if (!bind.getLdapResult().getResultCode().equals((Object)ResultCodeEnum.SUCCESS)) {
            LOG.trace("Re-binding DN {} failed", (Object)principal);
            throw new RuntimeException(bind.toString());
        }
        LOG.trace("Binding DN {} did not throw, connection authenticated: {}", (Object)principal, (Object)connection.isAuthenticated());
        return connection.isAuthenticated();
    }
}

