/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSortedSet;
import com.google.inject.assistedinject.Assisted;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.inject.Inject;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.indices.InvalidAliasNameException;
import org.graylog2.audit.AuditActor;
import org.graylog2.audit.AuditEventSender;
import org.graylog2.indexer.IndexSet;
import org.graylog2.indexer.NoTargetIndexException;
import org.graylog2.indexer.indexset.IndexSetConfig;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.indices.TooManyAliasesException;
import org.graylog2.indexer.indices.jobs.SetIndexReadOnlyAndCalculateRangeJob;
import org.graylog2.indexer.ranges.IndexRange;
import org.graylog2.indexer.ranges.IndexRangeService;
import org.graylog2.plugin.system.NodeId;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJobConcurrencyException;
import org.graylog2.system.jobs.SystemJobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MongoIndexSet
implements IndexSet {
    private static final Logger LOG = LoggerFactory.getLogger(MongoIndexSet.class);
    public static final String SEPARATOR = "_";
    public static final String DEFLECTOR_SUFFIX = "deflector";
    public static final String RESTORED_ARCHIVE_SUFFIX = "_restored_archive";
    private final IndexSetConfig config;
    private final Indices indices;
    private final Pattern indexPattern;
    private final Pattern deflectorIndexPattern;
    private final String indexWildcard;
    private final IndexRangeService indexRangeService;
    private final AuditEventSender auditEventSender;
    private final NodeId nodeId;
    private final SystemJobManager systemJobManager;
    private final SetIndexReadOnlyAndCalculateRangeJob.Factory jobFactory;
    private final ActivityWriter activityWriter;

    @Inject
    public MongoIndexSet(@Assisted IndexSetConfig config, Indices indices, NodeId nodeId, IndexRangeService indexRangeService, AuditEventSender auditEventSender, SystemJobManager systemJobManager, SetIndexReadOnlyAndCalculateRangeJob.Factory jobFactory, ActivityWriter activityWriter) {
        this.config = Objects.requireNonNull(config);
        this.indices = Objects.requireNonNull(indices);
        this.nodeId = Objects.requireNonNull(nodeId);
        this.indexRangeService = Objects.requireNonNull(indexRangeService);
        this.auditEventSender = Objects.requireNonNull(auditEventSender);
        this.systemJobManager = Objects.requireNonNull(systemJobManager);
        this.jobFactory = Objects.requireNonNull(jobFactory);
        this.activityWriter = Objects.requireNonNull(activityWriter);
        if (Strings.isNullOrEmpty((String)config.indexMatchPattern())) {
            this.indexPattern = Pattern.compile("^" + config.indexPrefix() + SEPARATOR + "\\d+(?:" + RESTORED_ARCHIVE_SUFFIX + ")?");
            this.deflectorIndexPattern = Pattern.compile("^" + config.indexPrefix() + SEPARATOR + "\\d+");
        } else {
            this.indexPattern = Pattern.compile("^" + config.indexMatchPattern() + SEPARATOR + "\\d+(?:" + RESTORED_ARCHIVE_SUFFIX + ")?");
            this.deflectorIndexPattern = Pattern.compile("^" + config.indexMatchPattern() + SEPARATOR + "\\d+");
        }
        this.indexWildcard = Strings.isNullOrEmpty((String)config.indexWildcard()) ? config.indexPrefix() + SEPARATOR + "*" : config.indexWildcard();
    }

    @Override
    public String[] getManagedIndices() {
        Set<String> indexNames = this.indices.getIndexNamesAndAliases(this.getIndexWildcard()).keySet();
        List<String> result = indexNames.stream().filter(this::isManagedIndex).collect(Collectors.toList());
        return result.toArray(new String[result.size()]);
    }

    @Override
    public String getWriteIndexAlias() {
        return this.config.indexPrefix() + SEPARATOR + DEFLECTOR_SUFFIX;
    }

    @Override
    public String getIndexWildcard() {
        return this.indexWildcard;
    }

    @Override
    public String getNewestIndex() throws NoTargetIndexException {
        return this.buildIndexName(this.getNewestIndexNumber());
    }

    @VisibleForTesting
    int getNewestIndexNumber() throws NoTargetIndexException {
        Set<String> indexNames = this.indices.getIndexNamesAndAliases(this.getIndexWildcard()).keySet();
        if (indexNames.isEmpty()) {
            throw new NoTargetIndexException();
        }
        int highestIndexNumber = -1;
        for (String indexName : indexNames) {
            if (!this.isGraylogDeflectorIndex(indexName)) continue;
            int currentHighest = highestIndexNumber;
            highestIndexNumber = this.extractIndexNumber(indexName).map(indexNumber -> Math.max(indexNumber, currentHighest)).orElse(highestIndexNumber);
        }
        if (highestIndexNumber == -1) {
            throw new NoTargetIndexException();
        }
        return highestIndexNumber;
    }

    @Override
    public Optional<Integer> extractIndexNumber(String indexName) {
        int beginIndex = this.config.indexPrefix().length() + 1;
        if (indexName.length() < beginIndex) {
            return Optional.empty();
        }
        String suffix = indexName.substring(beginIndex);
        try {
            return Optional.of(Integer.parseInt(suffix));
        }
        catch (NumberFormatException e) {
            return Optional.empty();
        }
    }

    @VisibleForTesting
    String buildIndexName(int number) {
        return this.config.indexPrefix() + SEPARATOR + number;
    }

    @VisibleForTesting
    boolean isGraylogDeflectorIndex(String indexName) {
        return !Strings.isNullOrEmpty((String)indexName) && !this.isWriteIndexAlias(indexName) && this.deflectorIndexPattern.matcher(indexName).matches();
    }

    @Override
    @Nullable
    public String getActiveWriteIndex() throws TooManyAliasesException {
        return this.indices.aliasTarget(this.getWriteIndexAlias());
    }

    @Override
    public Map<String, Set<String>> getAllIndexAliases() {
        Map<String, Set<String>> indexNamesAndAliases = this.indices.getIndexNamesAndAliases(this.getIndexWildcard());
        return indexNamesAndAliases.entrySet().stream().filter(e -> this.isGraylogDeflectorIndex((String)e.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    @Override
    public String getIndexPrefix() {
        return this.config.indexPrefix();
    }

    @Override
    public boolean isUp() {
        return this.indices.aliasExists(this.getWriteIndexAlias());
    }

    @Override
    public boolean isWriteIndexAlias(String index) {
        return this.getWriteIndexAlias().equals(index);
    }

    @Override
    public boolean isManagedIndex(String index) {
        return !Strings.isNullOrEmpty((String)index) && !this.isWriteIndexAlias(index) && this.indexPattern.matcher(index).matches();
    }

    @Override
    public void setUp() {
        if (!this.getConfig().isWritable()) {
            LOG.debug("Not setting up non-writable index set <{}> ({})", (Object)this.getConfig().id(), (Object)this.getConfig().title());
            return;
        }
        if (this.isUp()) {
            LOG.info("Found deflector alias <{}>. Using it.", (Object)this.getWriteIndexAlias());
        } else {
            LOG.info("Did not find an deflector alias. Setting one up now.");
            try {
                String currentTarget = this.getNewestIndex();
                LOG.info("Pointing to already existing index target <{}>", (Object)currentTarget);
                this.pointTo(currentTarget);
            }
            catch (NoTargetIndexException ex) {
                String msg = "There is no index target to point to. Creating one now.";
                LOG.info("There is no index target to point to. Creating one now.");
                this.activityWriter.write(new Activity("There is no index target to point to. Creating one now.", IndexSet.class));
                this.cycle();
            }
            catch (InvalidAliasNameException e) {
                LOG.error("Seems like there already is an index called <{}>", (Object)this.getWriteIndexAlias());
            }
        }
    }

    @Override
    public void cycle() {
        int oldTargetNumber;
        if (!this.getConfig().isWritable()) {
            LOG.debug("Not cycling non-writable index set <{}> ({})", (Object)this.getConfig().id(), (Object)this.getConfig().title());
            return;
        }
        try {
            oldTargetNumber = this.getNewestIndexNumber();
        }
        catch (NoTargetIndexException ex) {
            oldTargetNumber = -1;
        }
        int newTargetNumber = oldTargetNumber + 1;
        String newTarget = this.buildIndexName(newTargetNumber);
        String oldTarget = this.buildIndexName(oldTargetNumber);
        if (oldTargetNumber == -1) {
            LOG.info("Cycling from <none> to <{}>.", (Object)newTarget);
        } else {
            LOG.info("Cycling from <{}> to <{}>.", (Object)oldTarget, (Object)newTarget);
        }
        LOG.info("Creating target index <{}>.", (Object)newTarget);
        if (!this.indices.create(newTarget, this)) {
            throw new RuntimeException("Could not create new target index <" + newTarget + ">.");
        }
        LOG.info("Waiting for allocation of index <{}>.", (Object)newTarget);
        ClusterHealthStatus healthStatus = this.indices.waitForRecovery(newTarget);
        LOG.debug("Health status of index <{}>: {}", (Object)newTarget, (Object)healthStatus);
        this.addDeflectorIndexRange(newTarget);
        LOG.info("Index <{}> has been successfully allocated.", (Object)newTarget);
        String indexAlias = this.getWriteIndexAlias();
        LOG.info("Pointing index alias <{}> to new index <{}>.", (Object)indexAlias, (Object)newTarget);
        Activity activity = new Activity(IndexSet.class);
        if (oldTargetNumber == -1) {
            this.pointTo(newTarget);
            activity.setMessage("Cycled index alias <" + indexAlias + "> from <none> to <" + newTarget + ">.");
        } else {
            LOG.debug("Switching over index alias <{}>.", (Object)indexAlias);
            this.pointTo(newTarget, oldTarget);
            this.setIndexReadOnlyAndCalculateRange(oldTarget);
            activity.setMessage("Cycled index alias <" + indexAlias + "> from <" + oldTarget + "> to <" + newTarget + ">.");
        }
        LOG.info("Successfully pointed index alias <{}> to index <{}>.", (Object)indexAlias, (Object)newTarget);
        this.activityWriter.write(activity);
        this.auditEventSender.success(AuditActor.system(this.nodeId), "server:es_write_index:update", (Map<String, Object>)ImmutableMap.of((Object)"indexName", (Object)newTarget));
    }

    private void setIndexReadOnlyAndCalculateRange(String indexName) {
        SetIndexReadOnlyAndCalculateRangeJob setIndexReadOnlyAndCalculateRangeJob = this.jobFactory.create(indexName);
        try {
            this.systemJobManager.submitWithDelay(setIndexReadOnlyAndCalculateRangeJob, 30L, TimeUnit.SECONDS);
        }
        catch (SystemJobConcurrencyException e) {
            LOG.error("Cannot set index <" + indexName + "> to read only and calculate its range. It won't be optimized.", (Throwable)e);
        }
    }

    private void addDeflectorIndexRange(String indexName) {
        IndexRange deflectorRange = this.indexRangeService.createUnknownRange(indexName);
        this.indexRangeService.save(deflectorRange);
    }

    @Override
    public void cleanupAliases(Set<String> indexNames) {
        ImmutableSortedSet sortedSet = ImmutableSortedSet.orderedBy((Comparator)new IndexSet.IndexNameComparator(this)).addAll(indexNames).build();
        this.indices.removeAliases(this.getWriteIndexAlias(), sortedSet.headSet((String)sortedSet.last()));
    }

    @Override
    public void pointTo(String newIndexName, String oldIndexName) {
        this.indices.cycleAlias(this.getWriteIndexAlias(), newIndexName, oldIndexName);
    }

    private void pointTo(String indexName) {
        this.indices.cycleAlias(this.getWriteIndexAlias(), indexName);
    }

    @Override
    public IndexSetConfig getConfig() {
        return this.config;
    }

    @Override
    public int compareTo(IndexSet o) {
        return ComparisonChain.start().compare((Comparable)this.getConfig(), (Comparable)o.getConfig()).result();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        MongoIndexSet that = (MongoIndexSet)o;
        return Objects.equals(this.config, that.config);
    }

    public int hashCode() {
        return this.config.hashCode();
    }

    public String toString() {
        return "MongoIndexSet{config=" + this.config + '}';
    }

    public static interface Factory {
        public MongoIndexSet create(IndexSetConfig var1);
    }
}

