/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.streams;

import com.codahale.metrics.Timer;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.SimpleTimeLimiter;
import com.google.common.util.concurrent.TimeLimiter;
import com.google.common.util.concurrent.UncheckedTimeoutException;
import com.google.inject.assistedinject.Assisted;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Provider;
import org.graylog2.plugin.Message;
import org.graylog2.plugin.streams.DefaultStream;
import org.graylog2.plugin.streams.Stream;
import org.graylog2.plugin.streams.StreamRule;
import org.graylog2.plugin.streams.StreamRuleType;
import org.graylog2.streams.InvalidStreamRuleTypeException;
import org.graylog2.streams.StreamFaultManager;
import org.graylog2.streams.StreamListFingerprint;
import org.graylog2.streams.StreamMetrics;
import org.graylog2.streams.StreamRuleMatcherFactory;
import org.graylog2.streams.matchers.StreamRuleMatcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamRouterEngine {
    private static final Logger LOG = LoggerFactory.getLogger(StreamRouterEngine.class);
    private final EnumSet<StreamRuleType> ruleTypesNotNeedingFieldPresence = EnumSet.of(StreamRuleType.PRESENCE, StreamRuleType.EXACT, StreamRuleType.REGEX, StreamRuleType.ALWAYS_MATCH);
    private final List<Stream> streams;
    private final StreamFaultManager streamFaultManager;
    private final StreamMetrics streamMetrics;
    private final TimeLimiter timeLimiter;
    private final long streamProcessingTimeout;
    private final String fingerprint;
    private final Provider<Stream> defaultStreamProvider;
    private final List<Rule> rulesList;

    @Inject
    public StreamRouterEngine(@Assisted List<Stream> streams, @Assisted ExecutorService executorService, StreamFaultManager streamFaultManager, StreamMetrics streamMetrics, @DefaultStream Provider<Stream> defaultStreamProvider) {
        this.streams = streams;
        this.streamFaultManager = streamFaultManager;
        this.streamMetrics = streamMetrics;
        this.timeLimiter = SimpleTimeLimiter.create((ExecutorService)executorService);
        this.streamProcessingTimeout = streamFaultManager.getStreamProcessingTimeout();
        this.fingerprint = new StreamListFingerprint(streams).getFingerprint();
        this.defaultStreamProvider = defaultStreamProvider;
        ArrayList alwaysMatchRules = Lists.newArrayList();
        ArrayList presenceRules = Lists.newArrayList();
        ArrayList exactRules = Lists.newArrayList();
        ArrayList greaterRules = Lists.newArrayList();
        ArrayList smallerRules = Lists.newArrayList();
        ArrayList regexRules = Lists.newArrayList();
        ArrayList containsRules = Lists.newArrayList();
        for (Stream stream : streams) {
            for (StreamRule streamRule : stream.getStreamRules()) {
                Rule rule;
                try {
                    rule = new Rule(stream, streamRule, stream.getMatchingType());
                }
                catch (InvalidStreamRuleTypeException e) {
                    LOG.warn("Invalid stream rule type. Skipping matching for this rule. " + e.getMessage(), (Throwable)e);
                    continue;
                }
                switch (streamRule.getType()) {
                    case ALWAYS_MATCH: {
                        alwaysMatchRules.add(rule);
                        break;
                    }
                    case PRESENCE: {
                        presenceRules.add(rule);
                        break;
                    }
                    case EXACT: {
                        exactRules.add(rule);
                        break;
                    }
                    case GREATER: {
                        greaterRules.add(rule);
                        break;
                    }
                    case SMALLER: {
                        smallerRules.add(rule);
                        break;
                    }
                    case REGEX: {
                        regexRules.add(rule);
                        break;
                    }
                    case CONTAINS: {
                        containsRules.add(rule);
                    }
                }
            }
        }
        int size = alwaysMatchRules.size() + presenceRules.size() + exactRules.size() + greaterRules.size() + smallerRules.size() + containsRules.size() + regexRules.size();
        this.rulesList = Lists.newArrayListWithCapacity((int)size);
        this.rulesList.addAll(alwaysMatchRules);
        this.rulesList.addAll(presenceRules);
        this.rulesList.addAll(exactRules);
        this.rulesList.addAll(greaterRules);
        this.rulesList.addAll(smallerRules);
        this.rulesList.addAll(containsRules);
        this.rulesList.addAll(regexRules);
    }

    public List<Stream> getStreams() {
        return this.streams;
    }

    public String getFingerprint() {
        return this.fingerprint;
    }

    public List<Stream> match(Message message) {
        HashSet result = Sets.newHashSet();
        HashSet blackList = Sets.newHashSet();
        for (Rule rule : this.rulesList) {
            if (blackList.contains(rule.getStream())) continue;
            StreamRule streamRule = rule.getStreamRule();
            StreamRuleType streamRuleType = streamRule.getType();
            Stream.MatchingType matchingType = rule.getMatchingType();
            if (!this.ruleTypesNotNeedingFieldPresence.contains((Object)streamRuleType) && !message.hasField(streamRule.getField())) {
                if (matchingType != Stream.MatchingType.AND) continue;
                result.remove(rule.getStream());
                blackList.add(rule.getStream());
                continue;
            }
            Stream stream = streamRuleType != StreamRuleType.REGEX ? rule.match(message) : rule.matchWithTimeOut(message, this.streamProcessingTimeout, TimeUnit.MILLISECONDS);
            if (stream == null) {
                if (matchingType != Stream.MatchingType.AND) continue;
                result.remove(rule.getStream());
                blackList.add(rule.getStream());
                continue;
            }
            result.add(stream);
            if (matchingType != Stream.MatchingType.OR) continue;
            blackList.add(rule.getStream());
        }
        Stream defaultStream = (Stream)this.defaultStreamProvider.get();
        boolean alreadyRemovedDefaultStream = false;
        for (Stream stream : result) {
            this.streamMetrics.markIncomingMeter(stream.getId());
            if (!stream.getRemoveMatchesFromDefaultStream()) continue;
            if (alreadyRemovedDefaultStream || message.removeStream(defaultStream)) {
                alreadyRemovedDefaultStream = true;
                if (!LOG.isTraceEnabled()) continue;
                LOG.trace("Successfully removed default stream <{}> from message <{}>", (Object)defaultStream.getId(), (Object)message.getId());
                continue;
            }
            if (!LOG.isWarnEnabled()) continue;
            LOG.warn("Couldn't remove default stream <{}> from message <{}>", (Object)defaultStream.getId(), (Object)message.getId());
        }
        if (!alreadyRemovedDefaultStream) {
            this.streamMetrics.markIncomingMeter(defaultStream.getId());
        }
        return ImmutableList.copyOf((Collection)result);
    }

    public List<StreamTestMatch> testMatch(Message message) {
        ArrayList matches = Lists.newArrayList();
        for (Stream stream : this.streams) {
            StreamTestMatch match = new StreamTestMatch(stream);
            for (StreamRule streamRule : stream.getStreamRules()) {
                try {
                    Rule rule = new Rule(stream, streamRule, stream.getMatchingType());
                    match.addRule(rule);
                }
                catch (InvalidStreamRuleTypeException e) {
                    LOG.warn("Invalid stream rule type. Skipping matching for this rule. " + e.getMessage(), (Throwable)e);
                }
            }
            match.matchMessage(message);
            matches.add(match);
        }
        return matches;
    }

    public static class StreamTestMatch {
        private final Stream stream;
        private final List<Rule> rules = Lists.newArrayList();
        private final Stream.MatchingType matchingType;
        private final Map<StreamRule, Boolean> matches = Maps.newHashMap();

        public StreamTestMatch(Stream stream) {
            this.stream = stream;
            this.matchingType = stream.getMatchingType();
        }

        public void addRule(Rule rule) {
            this.rules.add(rule);
        }

        public void matchMessage(Message message) {
            for (Rule rule : this.rules) {
                Stream match = rule.match(message);
                this.matches.put(rule.getStreamRule(), match != null && match.equals(this.stream));
            }
        }

        public boolean isMatched() {
            switch (this.matchingType) {
                case OR: {
                    return this.matches.values().contains(true);
                }
            }
            return this.matches.size() > 0 && !this.matches.values().contains(false);
        }

        public Stream getStream() {
            return this.stream;
        }

        public Map<StreamRule, Boolean> getMatches() {
            return this.matches;
        }
    }

    private class Rule {
        private final Stream stream;
        private final StreamRule rule;
        private final StreamRuleMatcher matcher;
        private final Stream.MatchingType matchingType;

        public Rule(Stream stream, StreamRule rule, Stream.MatchingType matchingType) throws InvalidStreamRuleTypeException {
            this.stream = stream;
            this.rule = rule;
            this.matchingType = matchingType;
            this.matcher = StreamRuleMatcherFactory.build(rule.getType());
        }

        public Stream.MatchingType getMatchingType() {
            return this.matchingType;
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Nullable
        public Stream match(Message message) {
            try (Timer.Context ignored = StreamRouterEngine.this.streamMetrics.getExecutionTimer(this.rule.getId()).time();){
                if (this.matcher.match(message, this.rule)) {
                    Stream stream2 = this.stream;
                    return stream2;
                }
                Stream stream = null;
                return stream;
            }
            catch (Exception e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Error matching stream rule <" + (Object)((Object)this.rule.getType()) + "/" + this.rule.getValue() + ">: " + e.getMessage(), (Throwable)e);
                }
                StreamRouterEngine.this.streamMetrics.markExceptionMeter(this.rule.getStreamId());
                return null;
            }
        }

        @Nullable
        private Stream matchWithTimeOut(final Message message, long timeout, TimeUnit unit) {
            Stream matchedStream = null;
            try (Timer.Context ignored = StreamRouterEngine.this.streamMetrics.getExecutionTimer(this.rule.getId()).time();){
                matchedStream = (Stream)StreamRouterEngine.this.timeLimiter.callWithTimeout((Callable)new Callable<Stream>(){

                    @Override
                    @Nullable
                    public Stream call() throws Exception {
                        return Rule.this.match(message);
                    }
                }, timeout, unit);
            }
            catch (UncheckedTimeoutException e) {
                StreamRouterEngine.this.streamFaultManager.registerFailure(this.stream);
            }
            catch (Exception e) {
                LOG.warn("Unexpected error during stream matching", (Throwable)e);
                StreamRouterEngine.this.streamMetrics.markExceptionMeter(this.rule.getStreamId());
            }
            return matchedStream;
        }

        public StreamRule getStreamRule() {
            return this.rule;
        }

        public Stream getStream() {
            return this.stream;
        }
    }

    public static interface Factory {
        public StreamRouterEngine create(List<Stream> var1, ExecutorService var2);
    }
}

