/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.plugins.sidecar.rest.resources;

import com.google.common.collect.ImmutableMap;
import com.google.common.hash.Hashing;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.ws.rs.BadRequestException;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.CacheControl;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.EntityTag;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import org.apache.shiro.authz.annotation.RequiresAuthentication;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.graylog.plugins.sidecar.rest.models.Collector;
import org.graylog.plugins.sidecar.rest.models.CollectorSummary;
import org.graylog.plugins.sidecar.rest.responses.CollectorListResponse;
import org.graylog.plugins.sidecar.rest.responses.CollectorSummaryResponse;
import org.graylog.plugins.sidecar.services.CollectorService;
import org.graylog.plugins.sidecar.services.ConfigurationService;
import org.graylog.plugins.sidecar.services.EtagService;
import org.graylog2.audit.jersey.AuditEvent;
import org.graylog2.audit.jersey.NoAuditEvent;
import org.graylog2.database.PaginatedList;
import org.graylog2.plugin.rest.PluginRestResource;
import org.graylog2.plugin.rest.ValidationResult;
import org.graylog2.search.SearchQuery;
import org.graylog2.search.SearchQueryField;
import org.graylog2.search.SearchQueryParser;
import org.graylog2.shared.rest.resources.RestResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Api(value="Sidecar/Collectors", description="Manage collectors")
@Path(value="/sidecar/collectors")
@Consumes(value={"application/json"})
@Produces(value={"application/json"})
@RequiresAuthentication
public class CollectorResource
extends RestResource
implements PluginRestResource {
    private static final Logger LOG = LoggerFactory.getLogger(CollectorResource.class);
    private static final Pattern VALID_COLLECTOR_NAME_PATTERN = Pattern.compile("^[A-Za-z0-9_.-]+$");
    private static final Pattern VALID_PATH_PATTERN = Pattern.compile("^[^;*?\"<>|&]+$");
    private static final List<String> VALID_LINUX_SERVICE_TYPES = Arrays.asList("exec");
    private static final List<String> VALID_WINDOWS_SERVICE_TYPES = Arrays.asList("exec", "svc");
    private static final List<String> VALID_OPERATING_SYSTEMS = Arrays.asList("linux", "windows");
    private final CollectorService collectorService;
    private final ConfigurationService configurationService;
    private final EtagService etagService;
    private final SearchQueryParser searchQueryParser;
    private static final ImmutableMap<String, SearchQueryField> SEARCH_FIELD_MAPPING = ImmutableMap.builder().put((Object)"id", (Object)SearchQueryField.create("id")).put((Object)"name", (Object)SearchQueryField.create("name")).put((Object)"operating_system", (Object)SearchQueryField.create("node_operating_system")).build();

    @Inject
    public CollectorResource(CollectorService collectorService, ConfigurationService configurationService, EtagService etagService) {
        this.collectorService = collectorService;
        this.configurationService = configurationService;
        this.etagService = etagService;
        this.searchQueryParser = new SearchQueryParser("name", (Map<String, SearchQueryField>)SEARCH_FIELD_MAPPING);
    }

    @GET
    @Path(value="/{id}")
    @RequiresPermissions(value={"sidecars:read"})
    @Produces(value={"application/json"})
    @ApiOperation(value="Show collector details")
    public Collector getCollector(@ApiParam(name="id", required=true) @PathParam(value="id") String id) {
        Collector collector = this.collectorService.find(id);
        if (collector == null) {
            throw new NotFoundException("Cound not find collector <" + id + ">.");
        }
        return collector;
    }

    @GET
    @RequiresPermissions(value={"sidecars:read"})
    @Produces(value={"application/json"})
    @ApiOperation(value="List all collectors")
    public Response listCollectors(@Context HttpHeaders httpHeaders) {
        EntityTag etag;
        String ifNoneMatch = httpHeaders.getHeaderString("If-None-Match");
        Boolean etagCached = false;
        Response.ResponseBuilder builder = Response.noContent();
        if (ifNoneMatch != null && this.etagService.isPresent((etag = new EntityTag(ifNoneMatch.replaceAll("\"", ""))).toString())) {
            etagCached = true;
            builder = Response.notModified();
            builder.tag(etag);
        }
        if (!etagCached.booleanValue()) {
            List<Collector> result = this.collectorService.all();
            CollectorListResponse collectorListResponse = CollectorListResponse.create(result.size(), result);
            String etagString = this.collectorsToEtag(collectorListResponse);
            EntityTag collectorsEtag = new EntityTag(etagString);
            builder = Response.ok((Object)collectorListResponse);
            builder.tag(collectorsEtag);
            this.etagService.put(collectorsEtag.toString());
        }
        CacheControl cacheControl = new CacheControl();
        cacheControl.setNoTransform(true);
        cacheControl.setPrivate(true);
        builder.cacheControl(cacheControl);
        return builder.build();
    }

    @GET
    @Path(value="/summary")
    @RequiresPermissions(value={"sidecars:read"})
    @Produces(value={"application/json"})
    @ApiOperation(value="List a summary of all collectors")
    public CollectorSummaryResponse listSummary(@ApiParam(name="page") @QueryParam(value="page") @DefaultValue(value="1") int page, @ApiParam(name="per_page") @QueryParam(value="per_page") @DefaultValue(value="50") int perPage, @ApiParam(name="query") @QueryParam(value="query") @DefaultValue(value="") String query, @ApiParam(name="sort", value="The field to sort the result on", required=true, allowableValues="name,id,collector_id") @DefaultValue(value="name") @QueryParam(value="sort") String sort, @ApiParam(name="order", value="The sort direction", allowableValues="asc, desc") @DefaultValue(value="asc") @QueryParam(value="order") String order) {
        SearchQuery searchQuery = this.searchQueryParser.parse(query);
        PaginatedList<Collector> collectors = this.collectorService.findPaginated(searchQuery, page, perPage, sort, order);
        long total = this.collectorService.count();
        List<CollectorSummary> summaries = collectors.stream().map(CollectorSummary::create).collect(Collectors.toList());
        return CollectorSummaryResponse.create(query, collectors.pagination(), total, sort, order, summaries);
    }

    @POST
    @RequiresPermissions(value={"sidecars:create"})
    @Produces(value={"application/json"})
    @ApiOperation(value="Create a new collector")
    @AuditEvent(type="sidecar:collector:create")
    public Response createCollector(@ApiParam(name="JSON body", required=true) @Valid @NotNull Collector request) throws BadRequestException {
        Collector collector = this.collectorService.fromRequest(request);
        ValidationResult validationResult = this.validate(collector);
        if (validationResult.failed()) {
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)validationResult).build();
        }
        this.etagService.invalidateAll();
        return Response.ok().entity((Object)this.collectorService.save(collector)).build();
    }

    @PUT
    @Path(value="/{id}")
    @RequiresPermissions(value={"sidecars:update"})
    @Produces(value={"application/json"})
    @ApiOperation(value="Update a collector")
    @AuditEvent(type="sidecar:collector:update")
    public Response updateCollector(@ApiParam(name="id", required=true) @PathParam(value="id") String id, @ApiParam(name="JSON body", required=true) @Valid @NotNull Collector request) throws BadRequestException {
        Collector collector = this.collectorService.fromRequest(id, request);
        ValidationResult validationResult = this.validate(collector);
        if (validationResult.failed()) {
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)validationResult).build();
        }
        this.etagService.invalidateAll();
        return Response.ok().entity((Object)this.collectorService.save(collector)).build();
    }

    @POST
    @Path(value="/{id}/{name}")
    @RequiresPermissions(value={"sidecars:create"})
    @ApiOperation(value="Copy a collector")
    @AuditEvent(type="sidecar:collector:clone")
    public Response copyCollector(@ApiParam(name="id", required=true) @PathParam(value="id") String id, @ApiParam(name="name", required=true) @PathParam(value="name") String name) throws NotFoundException, BadRequestException {
        Collector collector = this.collectorService.copy(id, name);
        ValidationResult validationResult = this.validate(collector);
        if (validationResult.failed()) {
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)validationResult).build();
        }
        this.etagService.invalidateAll();
        this.collectorService.save(collector);
        return Response.accepted().build();
    }

    @DELETE
    @Path(value="/{id}")
    @RequiresPermissions(value={"sidecars:delete"})
    @Produces(value={"application/json"})
    @ApiOperation(value="Delete a collector")
    @AuditEvent(type="sidecar:collector:delete")
    public Response deleteCollector(@ApiParam(name="id", required=true) @PathParam(value="id") String id) {
        long configurationsForCollector = this.configurationService.all().stream().filter(configuration -> configuration.collectorId().equals(id)).count();
        if (configurationsForCollector > 0L) {
            throw new BadRequestException("Collector still in use, cannot delete.");
        }
        int deleted = this.collectorService.delete(id);
        if (deleted == 0) {
            return Response.notModified().build();
        }
        this.etagService.invalidateAll();
        return Response.accepted().build();
    }

    @POST
    @Path(value="/validate")
    @NoAuditEvent(value="Validation only")
    @RequiresPermissions(value={"configurations:read"})
    @Produces(value={"application/json"})
    @ApiOperation(value="Validates collector parameters")
    public ValidationResult validateCollector(@Valid @ApiParam(value="collector") Collector toValidate) {
        return this.validate(toValidate);
    }

    private ValidationResult validate(Collector toValidate) {
        Collector collector;
        Optional<Collector> collectorOptional;
        ValidationResult validation = new ValidationResult();
        if (toValidate.name().isEmpty()) {
            validation.addError("name", "Collector name cannot be empty.");
        } else if (!this.validateCollectorName(toValidate.name())) {
            validation.addError("name", "Collector name can only contain the following characters: A-Z,a-z,0-9,_,-,.");
        }
        if (toValidate.executablePath().isEmpty()) {
            validation.addError("executable_path", "Collector binary path cannot be empty.");
        } else if (!this.validatePath(toValidate.executablePath())) {
            validation.addError("executable_path", "Collector binary path cannot contain the following characters: ; * ? \" < > | &");
        }
        if (toValidate.nodeOperatingSystem() != null) {
            if (!this.validateOperatingSystem(toValidate.nodeOperatingSystem())) {
                validation.addError("node_operating_system", "Operating system can only be 'linux' or 'windows'.");
            }
            if (!this.validateServiceType(toValidate.serviceType(), toValidate.nodeOperatingSystem())) {
                validation.addError("service_type", "Linux collectors only support 'Foreground execution' while Windows collectors additionally support 'Windows service'.");
            }
            collectorOptional = Optional.ofNullable(this.collectorService.findByNameAndOs(toValidate.name(), toValidate.nodeOperatingSystem()));
        } else {
            collectorOptional = Optional.ofNullable(this.collectorService.findByName(toValidate.name()));
        }
        if (collectorOptional.isPresent() && !(collector = collectorOptional.get()).id().equals(toValidate.id())) {
            validation.addError("name", "Collector \"" + toValidate.name() + "\" already exists for the \"" + collector.nodeOperatingSystem() + "\" operating system.");
        }
        return validation;
    }

    private boolean validateCollectorName(String name) {
        return VALID_COLLECTOR_NAME_PATTERN.matcher(name).matches();
    }

    private boolean validateServiceType(String type, String operatingSystem) {
        switch (operatingSystem) {
            case "linux": {
                if (!VALID_LINUX_SERVICE_TYPES.contains(type)) break;
                return true;
            }
            case "windows": {
                if (!VALID_WINDOWS_SERVICE_TYPES.contains(type)) break;
                return true;
            }
        }
        return false;
    }

    private boolean validateOperatingSystem(String operatingSystem) {
        return VALID_OPERATING_SYSTEMS.contains(operatingSystem);
    }

    private boolean validatePath(String path) {
        return VALID_PATH_PATTERN.matcher(path).matches();
    }

    private String collectorsToEtag(CollectorListResponse collectors) {
        return Hashing.md5().hashInt(collectors.hashCode()).toString();
    }
}

