/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.events.processor;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.inject.Inject;
import org.graylog.events.processor.DBEventDefinitionService;
import org.graylog.events.processor.EventDefinitionDto;
import org.graylog.events.processor.EventProcessorExecutionJob;
import org.graylog.events.processor.EventProcessorSchedulerConfig;
import org.graylog.scheduler.DBJobDefinitionService;
import org.graylog.scheduler.DBJobTriggerService;
import org.graylog.scheduler.JobDefinitionDto;
import org.graylog.scheduler.JobTriggerDto;
import org.graylog.scheduler.clock.JobSchedulerClock;
import org.graylog2.plugin.database.users.User;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventDefinitionHandler {
    private static final Logger LOG = LoggerFactory.getLogger(EventDefinitionHandler.class);
    private final DBEventDefinitionService eventDefinitionService;
    private final DBJobDefinitionService jobDefinitionService;
    private final DBJobTriggerService jobTriggerService;
    private final JobSchedulerClock clock;

    @Inject
    public EventDefinitionHandler(DBEventDefinitionService eventDefinitionService, DBJobDefinitionService jobDefinitionService, DBJobTriggerService jobTriggerService, JobSchedulerClock clock) {
        this.eventDefinitionService = eventDefinitionService;
        this.jobDefinitionService = jobDefinitionService;
        this.jobTriggerService = jobTriggerService;
        this.clock = clock;
    }

    public EventDefinitionDto create(EventDefinitionDto unsavedEventDefinition, Optional<User> user) {
        EventDefinitionDto eventDefinition = this.createEventDefinition(unsavedEventDefinition, user);
        try {
            this.createJobDefinitionAndTriggerIfScheduledType(eventDefinition);
        }
        catch (Exception e) {
            LOG.error("Removing event definition <{}/{}> because of an error creating the job definition", new Object[]{eventDefinition.id(), eventDefinition.title(), e});
            this.eventDefinitionService.delete(eventDefinition.id());
            throw e;
        }
        return eventDefinition;
    }

    public EventDefinitionDto createWithoutSchedule(EventDefinitionDto unsavedEventDefinition, Optional<User> user) {
        return this.createEventDefinition(unsavedEventDefinition, user);
    }

    public EventDefinitionDto update(EventDefinitionDto updatedEventDefinition, boolean schedule) {
        Optional oldEventDefinition = this.eventDefinitionService.get(updatedEventDefinition.id());
        EventDefinitionDto eventDefinition = this.updateEventDefinition(updatedEventDefinition);
        try {
            if (schedule) {
                if (this.getJobDefinition(eventDefinition).isPresent()) {
                    this.updateJobDefinitionAndTriggerIfScheduledType(eventDefinition);
                } else {
                    this.createJobDefinitionAndTriggerIfScheduledType(eventDefinition);
                }
            } else {
                this.unschedule(eventDefinition.id());
            }
        }
        catch (Exception e) {
            LOG.error("Reverting to old event definition <{}/{}> because of an error updating the job definition", new Object[]{eventDefinition.id(), eventDefinition.title(), e});
            oldEventDefinition.ifPresent(this.eventDefinitionService::save);
            throw e;
        }
        return eventDefinition;
    }

    public boolean delete(String eventDefinitionId) {
        Optional optionalEventDefinition = this.eventDefinitionService.get(eventDefinitionId);
        if (!optionalEventDefinition.isPresent()) {
            return false;
        }
        EventDefinitionDto eventDefinition = (EventDefinitionDto)optionalEventDefinition.get();
        this.getJobDefinition(eventDefinition).ifPresent(jobDefinition -> this.deleteJobDefinitionAndTrigger((JobDefinitionDto)jobDefinition, eventDefinition));
        LOG.debug("Deleting event definition <{}/{}>", (Object)eventDefinition.id(), (Object)eventDefinition.title());
        return this.eventDefinitionService.delete(eventDefinitionId) > 0;
    }

    public void schedule(String eventDefinitionId) {
        EventDefinitionDto eventDefinition = this.getEventDefinitionOrThrowIAE(eventDefinitionId);
        this.createJobDefinitionAndTriggerIfScheduledType(eventDefinition);
    }

    public void unschedule(String eventDefinitionId) {
        EventDefinitionDto eventDefinition = this.getEventDefinitionOrThrowIAE(eventDefinitionId);
        this.getJobDefinition(eventDefinition).ifPresent(jobDefinition -> this.deleteJobDefinitionAndTrigger((JobDefinitionDto)jobDefinition, eventDefinition));
    }

    private EventDefinitionDto createEventDefinition(EventDefinitionDto unsavedEventDefinition, Optional<User> user) {
        EventDefinitionDto eventDefinition;
        if (user.isPresent()) {
            eventDefinition = this.eventDefinitionService.saveWithOwnership(unsavedEventDefinition, user.get());
            LOG.debug("Created event definition <{}/{}> with user <{}>", new Object[]{eventDefinition.id(), eventDefinition.title(), user.get()});
        } else {
            eventDefinition = this.eventDefinitionService.save(unsavedEventDefinition);
            LOG.debug("Created event definition <{}/{}> without user", (Object)eventDefinition.id(), (Object)eventDefinition.title());
        }
        return eventDefinition;
    }

    private EventDefinitionDto getEventDefinitionOrThrowIAE(String eventDefinitionId) {
        return (EventDefinitionDto)this.eventDefinitionService.get(eventDefinitionId).orElseThrow(() -> new IllegalArgumentException("Event definition <" + eventDefinitionId + "> doesn't exist"));
    }

    private EventDefinitionDto updateEventDefinition(EventDefinitionDto updatedEventDefinition) {
        EventDefinitionDto eventDefinition = this.eventDefinitionService.save(updatedEventDefinition);
        LOG.debug("Updated event definition <{}/{}>", (Object)eventDefinition.id(), (Object)eventDefinition.title());
        return eventDefinition;
    }

    private JobDefinitionDto newJobDefinition(EventDefinitionDto eventDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        return JobDefinitionDto.builder().title(eventDefinition.title()).description(eventDefinition.description()).config(schedulerConfig.jobDefinitionConfig()).build();
    }

    private JobDefinitionDto createJobDefinition(EventDefinitionDto eventDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        JobDefinitionDto jobDefinition = this.jobDefinitionService.save(this.newJobDefinition(eventDefinition, schedulerConfig));
        LOG.debug("Created scheduler job definition <{}/{}> for event definition <{}/{}>", new Object[]{jobDefinition.id(), jobDefinition.title(), eventDefinition.id(), eventDefinition.title()});
        return jobDefinition;
    }

    private void createJobDefinitionAndTrigger(EventDefinitionDto eventDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        JobDefinitionDto jobDefinition = this.createJobDefinition(eventDefinition, schedulerConfig);
        try {
            this.createJobTrigger(eventDefinition, jobDefinition, schedulerConfig);
        }
        catch (Exception e) {
            LOG.error("Removing job definition <{}/{}> because of an error creating the job trigger", new Object[]{jobDefinition.id(), jobDefinition.title(), e});
            this.jobDefinitionService.delete(jobDefinition.id());
            throw e;
        }
    }

    private void createJobDefinitionAndTriggerIfScheduledType(EventDefinitionDto eventDefinition) {
        this.getJobSchedulerConfig(eventDefinition).ifPresent(schedulerConfig -> this.createJobDefinitionAndTrigger(eventDefinition, (EventProcessorSchedulerConfig)schedulerConfig));
    }

    private Optional<JobDefinitionDto> getJobDefinition(EventDefinitionDto eventDefinition) {
        return this.jobDefinitionService.getByConfigField("event_definition_id", eventDefinition.id());
    }

    private JobDefinitionDto getJobDefinitionOrThrowISE(EventDefinitionDto eventDefinition) {
        return this.getJobDefinition(eventDefinition).orElseThrow(() -> new IllegalStateException("Couldn't find job definition for event definition <" + eventDefinition.id() + ">"));
    }

    private JobDefinitionDto updateJobDefinition(EventDefinitionDto eventDefinition, JobDefinitionDto oldJobDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        JobDefinitionDto unsavedJobDefinition = oldJobDefinition.toBuilder().title(eventDefinition.title()).description(eventDefinition.description()).config(schedulerConfig.jobDefinitionConfig()).build();
        JobDefinitionDto jobDefinition = this.jobDefinitionService.save(unsavedJobDefinition);
        LOG.debug("Updated scheduler job definition <{}/{}> for event definition <{}/{}>", new Object[]{jobDefinition.id(), jobDefinition.title(), eventDefinition.id(), eventDefinition.title()});
        return jobDefinition;
    }

    private void updateJobDefinitionAndTrigger(EventDefinitionDto eventDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        JobDefinitionDto oldJobDefinition = this.getJobDefinitionOrThrowISE(eventDefinition);
        JobDefinitionDto jobDefinition = this.updateJobDefinition(eventDefinition, oldJobDefinition, schedulerConfig);
        try {
            this.updateJobTrigger(eventDefinition, jobDefinition, oldJobDefinition, schedulerConfig);
        }
        catch (Exception e) {
            LOG.error("Reverting to old job definition <{}/{}> because of an error updating the job trigger", new Object[]{jobDefinition.id(), jobDefinition.title(), e});
            this.jobDefinitionService.save(oldJobDefinition);
            throw e;
        }
    }

    private void updateJobDefinitionAndTriggerIfScheduledType(EventDefinitionDto eventDefinition) {
        this.getJobSchedulerConfig(eventDefinition).ifPresent(schedulerConfig -> this.updateJobDefinitionAndTrigger(eventDefinition, (EventProcessorSchedulerConfig)schedulerConfig));
    }

    private void deleteJobDefinition(JobDefinitionDto jobDefinition, EventDefinitionDto eventDefinition) {
        LOG.debug("Deleting job definition <{}/{}> for event definition <{}/{}>", new Object[]{jobDefinition.id(), jobDefinition.title(), eventDefinition.id(), eventDefinition.title()});
        this.jobDefinitionService.delete(jobDefinition.id());
    }

    private void deleteJobDefinitionAndTrigger(JobDefinitionDto jobDefinition, EventDefinitionDto eventDefinition) {
        this.deleteJobTrigger(jobDefinition, eventDefinition);
        this.deleteJobDefinition(jobDefinition, eventDefinition);
    }

    private JobTriggerDto newJobTrigger(JobDefinitionDto jobDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        return JobTriggerDto.builderWithClock(this.clock).jobDefinitionId(Objects.requireNonNull(jobDefinition.id(), "Job definition ID cannot be null")).nextTime(this.clock.nowUTC()).schedule(schedulerConfig.schedule()).build();
    }

    private void createJobTrigger(EventDefinitionDto dto, JobDefinitionDto jobDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        JobTriggerDto jobTrigger = this.jobTriggerService.create(this.newJobTrigger(jobDefinition, schedulerConfig));
        LOG.debug("Created job trigger <{}> for job definition <{}/{}> and event definition <{}/{}>", new Object[]{jobTrigger.id(), jobDefinition.id(), jobDefinition.title(), dto.id(), dto.title()});
    }

    private Optional<JobTriggerDto> getJobTrigger(JobDefinitionDto jobDefinition) {
        List<JobTriggerDto> jobTriggers = this.jobTriggerService.getForJob(jobDefinition.id());
        if (jobTriggers.isEmpty()) {
            return Optional.empty();
        }
        return Optional.ofNullable(jobTriggers.get(0));
    }

    private void updateJobTrigger(EventDefinitionDto eventDefinition, JobDefinitionDto jobDefinition, JobDefinitionDto oldJobDefinition, EventProcessorSchedulerConfig schedulerConfig) {
        Optional<JobTriggerDto> optionalOldJobTrigger = this.getJobTrigger(jobDefinition);
        if (!optionalOldJobTrigger.isPresent()) {
            return;
        }
        JobTriggerDto oldJobTrigger = optionalOldJobTrigger.get();
        JobTriggerDto.Builder unsavedJobTriggerBuilder = oldJobTrigger.toBuilder().jobDefinitionId(Objects.requireNonNull(jobDefinition.id(), "Job definition ID cannot be null")).schedule(schedulerConfig.schedule()).nextTime(this.clock.nowUTC());
        EventProcessorExecutionJob.Config oldConfig = (EventProcessorExecutionJob.Config)oldJobDefinition.config();
        EventProcessorExecutionJob.Config config = (EventProcessorExecutionJob.Config)jobDefinition.config();
        if (!config.hasEqualSchedule(oldConfig)) {
            unsavedJobTriggerBuilder.data(null);
            unsavedJobTriggerBuilder.nextTime(config.parameters().timerange().getTo());
        }
        JobTriggerDto jobTrigger = unsavedJobTriggerBuilder.build();
        this.jobTriggerService.update(jobTrigger);
        LOG.debug("Updated scheduler job trigger <{}> for job definition <{}/{}> and event definition <{}/{}>", new Object[]{jobTrigger.id(), jobDefinition.id(), jobDefinition.title(), eventDefinition.id(), eventDefinition.title()});
    }

    private void deleteJobTrigger(JobDefinitionDto jobDefinition, EventDefinitionDto eventDefinition) {
        Optional<JobTriggerDto> optionalJobTrigger = this.getJobTrigger(jobDefinition);
        if (!optionalJobTrigger.isPresent()) {
            return;
        }
        JobTriggerDto jobTrigger = optionalJobTrigger.get();
        LOG.debug("Deleting scheduler job trigger <{}> for job definition <{}/{}> and event definition <{}/{}>", new Object[]{jobTrigger.id(), jobDefinition.id(), jobDefinition.title(), eventDefinition.id(), eventDefinition.title()});
        this.jobTriggerService.delete(jobTrigger.id());
    }

    private Optional<EventProcessorSchedulerConfig> getJobSchedulerConfig(EventDefinitionDto eventDefinition) {
        return eventDefinition.config().toJobSchedulerConfig(eventDefinition, this.clock);
    }
}

