/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.plugins.map.geoip;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.AbstractIdleService;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.graylog.plugins.map.config.DatabaseType;
import org.graylog.plugins.map.config.DatabaseVendorType;
import org.graylog.plugins.map.config.GeoIpResolverConfig;
import org.graylog.plugins.map.geoip.GeoIpDbFileChangedEvent;
import org.graylog.plugins.map.geoip.GeoIpVendorResolverService;
import org.graylog2.cluster.ClusterConfigChangedEvent;
import org.graylog2.plugin.cluster.ClusterConfigService;
import org.graylog2.plugin.utilities.FileInfo;
import org.graylog2.plugin.validate.ConfigValidationException;
import org.graylog2.rest.resources.system.GeoIpResolverConfigValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public final class GeoIpDbFileChangeMonitorService
extends AbstractIdleService {
    private static final Logger LOG = LoggerFactory.getLogger((String)GeoIpDbFileChangeMonitorService.class.getSimpleName());
    private ScheduledFuture<?> refreshTask;
    private final ScheduledExecutorService scheduler;
    private final GeoIpResolverConfigValidator geoIpResolverConfigValidator;
    private final ClusterConfigService clusterConfigService;
    private final EventBus eventBus;
    private Duration dbRefreshInterval = Duration.ZERO;
    private FileInfo cityDbFileInfo = FileInfo.empty();
    private FileInfo asnDbFileInfo = FileInfo.empty();
    private GeoIpResolverConfig config;

    @Inject
    public GeoIpDbFileChangeMonitorService(@Named(value="daemonScheduler") ScheduledExecutorService scheduler, EventBus eventBus, ClusterConfigService clusterConfigService, GeoIpVendorResolverService geoIpVendorResolverService) {
        this.scheduler = Objects.requireNonNull(scheduler);
        this.eventBus = Objects.requireNonNull(eventBus);
        this.clusterConfigService = Objects.requireNonNull(clusterConfigService);
        this.geoIpResolverConfigValidator = new GeoIpResolverConfigValidator(geoIpVendorResolverService);
    }

    @Subscribe
    public void onClusterConfigChanged(ClusterConfigChangedEvent event) {
        if (GeoIpResolverConfig.class.getCanonicalName().equals(event.type())) {
            this.scheduler.schedule(this::updateConfiguration, 0L, TimeUnit.SECONDS);
        }
    }

    protected void startUp() throws Exception {
        this.eventBus.register((Object)this);
        this.updateConfiguration();
    }

    protected void shutDown() throws Exception {
        this.eventBus.unregister((Object)this);
    }

    private void refreshDatabases() {
        LOG.debug("Starting GeoIP database refresh");
        Map<DatabaseType, FileInfo.Change> changes = this.checkForChanges();
        if (changes.isEmpty()) {
            LOG.debug("GeoIP Database files have not changed--will not refresh");
        } else {
            GeoIpDbFileChangedEvent event = GeoIpDbFileChangedEvent.create();
            this.eventBus.post((Object)event);
        }
    }

    private Map<DatabaseType, FileInfo.Change> checkForChanges() {
        FileInfo.Change cityDbChange = this.cityDbFileInfo.checkForChange();
        FileInfo.Change asnDbChange = this.asnDbFileInfo.checkForChange();
        if (this.config == null) {
            this.config = this.getCurrentConfig();
        }
        DatabaseVendorType vendorType = this.config.databaseVendorType();
        EnumMap<DatabaseType, FileInfo.Change> changes = new EnumMap<DatabaseType, FileInfo.Change>(DatabaseType.class);
        if (cityDbChange.isChanged()) {
            changes.put(vendorType.getCityDbType(), cityDbChange);
            this.cityDbFileInfo = cityDbChange.fileInfo();
        }
        if (asnDbChange.isChanged()) {
            changes.put(vendorType.getAsnDbType(), asnDbChange);
            this.asnDbFileInfo = asnDbChange.fileInfo();
        }
        return changes;
    }

    private void updateConfiguration() {
        try {
            this.config = this.getCurrentConfig();
            this.geoIpResolverConfigValidator.validate(this.config);
            if (this.config.enabled()) {
                this.reScheduleRefreshIfNeeded();
                this.cityDbFileInfo = this.getDbFileInfo(this.config.cityDbPath());
                this.asnDbFileInfo = this.getDbFileInfo(this.config.asnDbPath());
            } else {
                LOG.debug("GeoIP Processor is disabled.  Will not schedule GeoIP database file change monitor");
                this.cancelScheduledRefreshTask();
                this.dbRefreshInterval = Duration.ZERO;
            }
        }
        catch (IllegalArgumentException | IllegalStateException | ConfigValidationException e) {
            LOG.error("Error validating GeoIP Database files. {}", (Object)e.getMessage(), (Object)e);
        }
    }

    private void cancelScheduledRefreshTask() {
        if (this.refreshTask != null) {
            boolean canceled = this.refreshTask.cancel(true);
            if (canceled) {
                LOG.debug("The GeoIP database file change monitor was running.  It has been cancelled");
                this.refreshTask = null;
            } else {
                LOG.warn("The GeoIP database file change monitor was running and failed to stop it");
            }
        }
    }

    private void reScheduleRefreshIfNeeded() {
        if (!this.dbRefreshInterval.equals(this.config.refreshIntervalAsDuration())) {
            boolean reschedule;
            boolean bl = reschedule = this.refreshTask == null || this.refreshTask.cancel(true);
            if (reschedule) {
                this.dbRefreshInterval = this.config.refreshIntervalAsDuration();
                this.scheduleDbRefresh();
            } else {
                LOG.warn("Failed to Cancel existing GeoIp Database Refresh Task.  Will not update refresh interval.");
            }
        }
    }

    private FileInfo getDbFileInfo(String path) {
        try {
            return FileInfo.forPath(Paths.get(path, new String[0]));
        }
        catch (Exception e) {
            return FileInfo.empty();
        }
    }

    private void scheduleDbRefresh() {
        try {
            long millis = this.dbRefreshInterval.toMillis();
            this.refreshTask = this.scheduler.scheduleAtFixedRate(this::refreshDatabases, millis, millis, TimeUnit.MILLISECONDS);
            LOG.debug("Scheduled GeoIP database refresh every '{}' Milliseconds", (Object)millis);
        }
        catch (Exception e) {
            LOG.error("Error scheduling GeoIP database refresh job. {}", (Object)e.getMessage(), (Object)e);
        }
    }

    private GeoIpResolverConfig getCurrentConfig() {
        return this.clusterConfigService.getOrDefault(GeoIpResolverConfig.class, GeoIpResolverConfig.defaultConfig());
    }
}

