/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer.rotation.strategies;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.graylog2.audit.AuditEventSender;
import org.graylog2.configuration.ElasticsearchConfiguration;
import org.graylog2.indexer.IndexSet;
import org.graylog2.indexer.indexset.IndexSetConfig;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.rotation.strategies.AbstractRotationStrategy;
import org.graylog2.indexer.rotation.strategies.TimeBasedRotationStrategyConfig;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.indexer.rotation.RotationStrategyConfig;
import org.graylog2.plugin.system.NodeId;
import org.joda.time.DateTime;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadablePeriod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class TimeBasedRotationStrategy
extends AbstractRotationStrategy {
    private static final Logger log = LoggerFactory.getLogger(TimeBasedRotationStrategy.class);
    public static final String NAME = "time";
    public static final String OVERRIDE_HINT = "(elasticsearch_max_write_index_age overrides configured period)";
    private final Indices indices;
    private Map<String, DateTime> anchor = new ConcurrentHashMap<String, DateTime>();

    @Inject
    public TimeBasedRotationStrategy(Indices indices, NodeId nodeId, AuditEventSender auditEventSender, ElasticsearchConfiguration elasticsearchConfiguration) {
        super(auditEventSender, nodeId, elasticsearchConfiguration);
        this.indices = Objects.requireNonNull(indices, "indices must not be null");
    }

    @Override
    public Class<? extends RotationStrategyConfig> configurationClass() {
        return TimeBasedRotationStrategyConfig.class;
    }

    @Override
    public RotationStrategyConfig defaultConfiguration() {
        return TimeBasedRotationStrategyConfig.builder().maxRotationPeriod(this.elasticsearchConfiguration.getMaxWriteIndexAge()).build();
    }

    public void reset() {
        this.anchor.clear();
    }

    static DateTime determineRotationPeriodAnchor(@Nullable DateTime lastAnchor, Period period) {
        Period normalized = period.normalizedStandard();
        int years = normalized.getYears();
        int months = normalized.getMonths();
        int weeks = normalized.getWeeks();
        int days = normalized.getDays();
        int hours = normalized.getHours();
        int minutes = normalized.getMinutes();
        int seconds = normalized.getSeconds();
        if (years == 0 && months == 0 && weeks == 0 && days == 0 && hours == 0 && minutes == 0 && seconds == 0) {
            throw new IllegalArgumentException("Invalid rotation period specified");
        }
        DateTimeFieldType largestStrideType = null;
        if (seconds > 0) {
            largestStrideType = DateTimeFieldType.secondOfMinute();
        }
        if (minutes > 0) {
            largestStrideType = DateTimeFieldType.minuteOfHour();
        }
        if (hours > 0) {
            largestStrideType = DateTimeFieldType.hourOfDay();
        }
        if (days > 0) {
            largestStrideType = DateTimeFieldType.dayOfMonth();
        }
        if (weeks > 0) {
            largestStrideType = DateTimeFieldType.weekOfWeekyear();
        }
        if (months > 0) {
            largestStrideType = DateTimeFieldType.monthOfYear();
        }
        if (years > 0) {
            largestStrideType = DateTimeFieldType.year();
        }
        if (largestStrideType == null) {
            throw new IllegalArgumentException("Could not determine rotation stride length.");
        }
        DateTime anchorTime = TimeBasedRotationStrategy.anchorTimeFrom(lastAnchor);
        DateTimeField field = largestStrideType.getField(anchorTime.getChronology());
        int periodValue = normalized.get(largestStrideType.getDurationType());
        long fieldValue = field.roundFloor(anchorTime.getMillis());
        int fieldValueInUnit = field.get(fieldValue);
        if (periodValue == 0) {
            log.warn("Determining stride length failed because of a 0 period. Defaulting back to 1 period to avoid crashing, but this is a bug!");
            periodValue = 1;
        }
        long difference = fieldValueInUnit % periodValue;
        long newValue = field.add(fieldValue, -1L * difference);
        return new DateTime(newValue, DateTimeZone.UTC);
    }

    private static DateTime anchorTimeFrom(@Nullable DateTime lastAnchor) {
        if (lastAnchor != null && !lastAnchor.getZone().equals((Object)DateTimeZone.UTC)) {
            return lastAnchor.withZone(DateTimeZone.UTC);
        }
        return (DateTime)MoreObjects.firstNonNull((Object)lastAnchor, (Object)Tools.nowUTC());
    }

    private static boolean isLonger(Period p1, Period p2) {
        Instant now = Instant.now();
        Duration d1 = p1.toDurationTo((ReadableInstant)now);
        Duration d2 = p2.toDurationTo((ReadableInstant)now);
        return d1.isLongerThan((ReadableDuration)d2);
    }

    @Override
    @Nullable
    protected AbstractRotationStrategy.Result shouldRotate(String index, IndexSet indexSet) {
        IndexSetConfig indexSetConfig = Objects.requireNonNull(indexSet.getConfig(), "Index set configuration must not be null");
        String indexSetId = indexSetConfig.id();
        Preconditions.checkState((!Strings.isNullOrEmpty((String)index) ? 1 : 0) != 0, (Object)"Index name must not be null or empty");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)indexSetId) ? 1 : 0) != 0, (Object)"Index set ID must not be null or empty");
        Preconditions.checkState((boolean)(indexSetConfig.rotationStrategy() instanceof TimeBasedRotationStrategyConfig), (String)"Invalid rotation strategy config <%s> for index set <%s>", (Object)indexSetConfig.rotationStrategy().getClass().getCanonicalName(), (Object)indexSet);
        TimeBasedRotationStrategyConfig config = (TimeBasedRotationStrategyConfig)indexSetConfig.rotationStrategy();
        Pair<Period, Boolean> normalizedRotationPeriod = this.getNormalizedRotationPeriod(config);
        Period normalizedPeriod = (Period)normalizedRotationPeriod.getLeft();
        boolean overriding = (Boolean)normalizedRotationPeriod.getRight();
        if (!this.anchor.containsKey(indexSetId)) {
            this.indices.indexCreationDate(index).ifPresent(creationDate -> {
                DateTime currentAnchor = TimeBasedRotationStrategy.determineRotationPeriodAnchor(creationDate, normalizedPeriod);
                this.anchor.put(indexSetId, currentAnchor);
            });
            if (!this.anchor.containsKey(indexSetId)) {
                return new SimpleResult(true, "No known previous rotation time, forcing index rotation now.");
            }
        }
        DateTime now = Tools.nowUTC();
        DateTime currentAnchor = this.anchor.get(indexSetId);
        DateTime nextRotation = currentAnchor.plus((ReadablePeriod)normalizedPeriod);
        if (nextRotation.isAfter((ReadableInstant)now)) {
            String message = new MessageFormat("Next rotation at {0} {1}", Locale.ENGLISH).format(new Object[]{nextRotation, overriding ? OVERRIDE_HINT : ""});
            return new SimpleResult(false, message);
        }
        DateTime nextAnchor = this.calculateNextAnchor(currentAnchor, normalizedPeriod, now);
        this.anchor.put(indexSetId, nextAnchor);
        if (!config.rotateEmptyIndexSet() && this.isEmptyIndexSet(indexSet)) {
            log.debug("Index set {} contains no messages, skipping rotation!", (Object)indexSet);
            String message = new MessageFormat("Index set contains no messages, skipping rotation! Next rotation at {0} {1}", Locale.ENGLISH).format(new Object[]{nextAnchor, overriding ? OVERRIDE_HINT : ""});
            return new SimpleResult(false, message);
        }
        String message = new MessageFormat("Rotation period {0} elapsed, next rotation at {1} {2}", Locale.ENGLISH).format(new Object[]{now, nextAnchor, overriding ? OVERRIDE_HINT : ""});
        return new SimpleResult(true, message);
    }

    private boolean isEmptyIndexSet(IndexSet indexSet) {
        Set<String> allIndices = this.indices.getIndices(indexSet, new String[0]);
        for (String index : allIndices) {
            if (this.indices.numberOfMessages(index) <= 0L) continue;
            return false;
        }
        return true;
    }

    private Pair<Period, Boolean> getNormalizedRotationPeriod(TimeBasedRotationStrategyConfig config) {
        Period rotationPeriod = config.rotationPeriod();
        Period maxPeriod = this.elasticsearchConfiguration.getMaxWriteIndexAge();
        boolean overriding = false;
        if (maxPeriod != null && TimeBasedRotationStrategy.isLonger(rotationPeriod, maxPeriod)) {
            log.debug("Max rotation limit {} overrides configured period {}", (Object)maxPeriod, (Object)rotationPeriod);
            rotationPeriod = maxPeriod;
            overriding = true;
        }
        Period normalizedPeriod = rotationPeriod.normalizedStandard();
        return new ImmutablePair((Object)normalizedPeriod, (Object)overriding);
    }

    private DateTime calculateNextAnchor(DateTime currentAnchor, Period normalizedPeriod, DateTime now) {
        DateTime tmpAnchor;
        int multiplicator = 0;
        while ((tmpAnchor = currentAnchor.withPeriodAdded((ReadablePeriod)normalizedPeriod, ++multiplicator)).isBefore((ReadableInstant)now)) {
        }
        return currentAnchor.withPeriodAdded((ReadablePeriod)normalizedPeriod, multiplicator - 1);
    }

    @Override
    public String getStrategyName() {
        return NAME;
    }

    static class SimpleResult
    implements AbstractRotationStrategy.Result {
        private final String message;
        private final boolean rotate;

        SimpleResult(boolean rotate, String message) {
            this.message = message;
            this.rotate = rotate;
            log.debug("{} because of: {}", (Object)(rotate ? "Rotating" : "Not rotating"), (Object)message);
        }

        @Override
        public String getDescription() {
            return this.message;
        }

        @Override
        public boolean shouldRotate() {
            return this.rotate;
        }
    }
}

