/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.security.certutil;

import com.google.common.annotations.VisibleForTesting;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.graylog.scheduler.DBJobTriggerService;
import org.graylog.scheduler.JobTriggerDto;
import org.graylog.scheduler.clock.JobSchedulerClock;
import org.graylog.security.certutil.CaService;
import org.graylog.security.certutil.CertRenewalService;
import org.graylog.security.certutil.ca.exceptions.KeyStoreStorageException;
import org.graylog.security.certutil.keystore.storage.KeystoreMongoStorage;
import org.graylog.security.certutil.keystore.storage.location.KeystoreMongoCollections;
import org.graylog.security.certutil.keystore.storage.location.KeystoreMongoLocation;
import org.graylog2.cluster.Node;
import org.graylog2.cluster.NodeService;
import org.graylog2.cluster.preflight.DataNodeProvisioningConfig;
import org.graylog2.cluster.preflight.DataNodeProvisioningService;
import org.graylog2.notifications.Notification;
import org.graylog2.notifications.NotificationService;
import org.graylog2.plugin.certificates.RenewalPolicy;
import org.graylog2.plugin.cluster.ClusterConfigService;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class CertRenewalServiceImpl
implements CertRenewalService {
    private static final Logger LOG = LoggerFactory.getLogger(CertRenewalServiceImpl.class);
    private final ClusterConfigService clusterConfigService;
    private final KeystoreMongoStorage keystoreMongoStorage;
    private final NodeService nodeService;
    private final DataNodeProvisioningService dataNodeProvisioningService;
    private final NotificationService notificationService;
    private final DBJobTriggerService jobTriggerService;
    private final JobSchedulerClock clock;
    private final CaService caService;
    private final char[] passwordSecret;
    private long CERT_RENEWAL_THRESHOLD_PERCENTAGE = 10L;

    @Inject
    public CertRenewalServiceImpl(ClusterConfigService clusterConfigService, KeystoreMongoStorage keystoreMongoStorage, NodeService nodeService, DataNodeProvisioningService dataNodeProvisioningService, NotificationService notificationService, DBJobTriggerService jobTriggerService, CaService caService, JobSchedulerClock clock, @Named(value="password_secret") String passwordSecret) {
        this.clusterConfigService = clusterConfigService;
        this.keystoreMongoStorage = keystoreMongoStorage;
        this.nodeService = nodeService;
        this.dataNodeProvisioningService = dataNodeProvisioningService;
        this.notificationService = notificationService;
        this.jobTriggerService = jobTriggerService;
        this.clock = clock;
        this.caService = caService;
        this.passwordSecret = passwordSecret.toCharArray();
    }

    @VisibleForTesting
    CertRenewalServiceImpl(JobSchedulerClock clock) {
        this(null, null, null, null, null, null, null, clock, "dummy");
    }

    private RenewalPolicy getRenewalPolicy() {
        return this.clusterConfigService.get(RenewalPolicy.class);
    }

    boolean needsRenewal(DateTime nextRenewal, RenewalPolicy renewalPolicy, X509Certificate cert) {
        Date threshold = this.calculateThreshold(renewalPolicy.certificateLifetime());
        try {
            cert.checkValidity(threshold);
            cert.checkValidity(nextRenewal.toDate());
        }
        catch (CertificateExpiredException e) {
            LOG.debug("Certificate about to expire.");
            return true;
        }
        catch (CertificateNotYetValidException e) {
            LOG.debug("Certificate not yet valid - which is surprising, but ignoring it.");
        }
        return false;
    }

    Date convertToDateViaInstant(LocalDateTime dateToConvert) {
        return Date.from(dateToConvert.atZone(ZoneId.systemDefault()).toInstant());
    }

    Date calculateThreshold(String certificateLifetime) {
        Duration lifetime = Duration.parse(certificateLifetime).dividedBy(this.CERT_RENEWAL_THRESHOLD_PERCENTAGE);
        LocalDateTime validUntil = this.clock.now(ZoneId.systemDefault()).plus(lifetime).toLocalDateTime();
        return this.convertToDateViaInstant(validUntil);
    }

    @Override
    public void checkCertificatesForRenewal() {
        RenewalPolicy renewalPolicy = this.getRenewalPolicy();
        if (renewalPolicy != null) {
            this.checkDataNodesCertificatesForRenewal(renewalPolicy);
            this.checkCaCertificatesForRenewal(renewalPolicy);
        }
    }

    private DateTime getNextRenewal() {
        return this.jobTriggerService.getOneForJob("64a66741cb3275652764c937").map(JobTriggerDto::nextTime).orElse(DateTime.now((DateTimeZone)DateTimeZone.UTC).plusMinutes(30));
    }

    protected void checkCaCertificatesForRenewal(RenewalPolicy renewalPolicy) {
        try {
            Optional<KeyStore> keystore = this.caService.loadKeyStore();
            if (keystore.isPresent()) {
                Certificate caCert;
                Certificate rootCert;
                KeyStore ks = keystore.get();
                DateTime nextRenewal = this.getNextRenewal();
                if (this.needsRenewal(nextRenewal, renewalPolicy, (X509Certificate)(rootCert = ks.getCertificate("root")))) {
                    this.notificationService.fixed(Notification.Type.CERTIFICATE_NEEDS_RENEWAL, "root cert");
                }
                if (this.needsRenewal(nextRenewal, renewalPolicy, (X509Certificate)(caCert = ks.getCertificate("ca")))) {
                    this.notificationService.fixed(Notification.Type.CERTIFICATE_NEEDS_RENEWAL, "ca cert");
                }
            }
        }
        catch (KeyStoreException | NoSuchAlgorithmException | KeyStoreStorageException e) {
            LOG.error("Could not read CA keystore: {}", (Object)e.getMessage());
        }
    }

    private Optional<KeyStore> loadKeyStoreForNode(Node node) {
        try {
            return this.keystoreMongoStorage.readKeyStore(new KeystoreMongoLocation(node.getNodeId(), KeystoreMongoCollections.DATA_NODE_KEYSTORE_COLLECTION), this.passwordSecret);
        }
        catch (KeyStoreStorageException e) {
            return Optional.empty();
        }
    }

    private Optional<X509Certificate> getCertificateForNode(KeyStore keyStore) {
        try {
            return Optional.of((X509Certificate)keyStore.getCertificate("datanode"));
        }
        catch (KeyStoreException e) {
            return Optional.empty();
        }
    }

    protected List<Node> findNodesThatNeedCertificateRenewal(RenewalPolicy renewalPolicy) {
        DateTime nextRenewal = this.getNextRenewal();
        Map<String, Node> activeDataNodes = this.nodeService.allActive(Node.Type.DATANODE);
        return activeDataNodes.values().stream().map(node -> {
            Optional<KeyStore> keystore = this.loadKeyStoreForNode((Node)node);
            Optional certificate = keystore.flatMap(this::getCertificateForNode);
            if (certificate.isPresent() && this.needsRenewal(nextRenewal, renewalPolicy, (X509Certificate)certificate.get())) {
                return node;
            }
            return null;
        }).filter(Objects::nonNull).toList();
    }

    @Override
    public void initiateRenewalForNode(String nodeId) {
        DataNodeProvisioningConfig config = this.dataNodeProvisioningService.getPreflightConfigFor(nodeId);
        this.dataNodeProvisioningService.save(config.toBuilder().state(DataNodeProvisioningConfig.State.CONFIGURED).build());
    }

    @Override
    public List<CertRenewalService.DataNode> findNodes() {
        Map<String, Node> activeDataNodes = this.nodeService.allActive(Node.Type.DATANODE);
        return activeDataNodes.values().stream().map(node -> {
            Optional<KeyStore> keystore = this.loadKeyStoreForNode((Node)node);
            Optional<LocalDateTime> certificate = keystore.flatMap(this::getCertificateForNode);
            Optional<LocalDateTime> certValidUntil = certificate.map(cert -> cert.getNotAfter().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime());
            DataNodeProvisioningConfig config = this.getDataNodeProvisioningConfig((Node)node);
            return new CertRenewalService.DataNode(node.getNodeId(), node.getType(), node.getTransportAddress(), config.state(), config.errorMsg(), node.getHostname(), node.getShortNodeId(), certValidUntil.orElse(null));
        }).toList();
    }

    private DataNodeProvisioningConfig getDataNodeProvisioningConfig(Node node) {
        return this.dataNodeProvisioningService.getPreflightConfigFor(node.getNodeId());
    }

    private void notifyManualRenewalForNode(List<Node> nodes) {
        String key = String.join((CharSequence)",", nodes.stream().map(Node::getNodeId).toList());
        if (!this.notificationService.isFirst(Notification.Type.CERTIFICATE_NEEDS_RENEWAL)) {
            this.notificationService.fixed(Notification.Type.CERTIFICATE_NEEDS_RENEWAL);
        }
        Notification notification = this.notificationService.buildNow().addType(Notification.Type.CERTIFICATE_NEEDS_RENEWAL).addSeverity(Notification.Severity.URGENT).addKey(key).addDetail("nodes", key);
        this.notificationService.publishIfFirst(notification);
    }

    protected void checkDataNodesCertificatesForRenewal(RenewalPolicy renewalPolicy) {
        List<Node> nodes = this.findNodesThatNeedCertificateRenewal(renewalPolicy);
        if (nodes.isEmpty()) {
            return;
        }
        if (RenewalPolicy.Mode.AUTOMATIC.equals((Object)renewalPolicy.mode())) {
            nodes.forEach(node -> this.initiateRenewalForNode(node.getNodeId()));
        } else {
            this.notifyManualRenewalForNode(nodes);
        }
    }
}

