/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.plugins.map.geoip;

import com.codahale.metrics.Reservoir;
import com.codahale.metrics.Timer;
import com.codahale.metrics.UniformReservoir;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.AbstractIdleService;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.inject.Singleton;
import java.io.IOException;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.graylog.plugins.map.config.DatabaseType;
import org.graylog.plugins.map.config.DatabaseVendorType;
import org.graylog.plugins.map.config.GeoIpResolverConfig;
import org.graylog.plugins.map.config.S3DownloadException;
import org.graylog.plugins.map.config.S3GeoIpFileService;
import org.graylog.plugins.map.geoip.GeoIpDbFileChangedEvent;
import org.graylog.plugins.map.geoip.GeoIpVendorResolverService;
import org.graylog2.cluster.ClusterConfigChangedEvent;
import org.graylog2.notifications.Notification;
import org.graylog2.notifications.NotificationService;
import org.graylog2.plugin.cluster.ClusterConfigService;
import org.graylog2.plugin.utilities.FileInfo;
import org.graylog2.plugin.validate.ConfigValidationException;
import org.graylog2.rest.resources.system.GeoIpResolverConfigValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public final class GeoIpDbFileChangeMonitorService
extends AbstractIdleService {
    private static final Logger LOG = LoggerFactory.getLogger((String)GeoIpDbFileChangeMonitorService.class.getSimpleName());
    private ScheduledFuture<?> refreshTask;
    private final ScheduledExecutorService scheduler;
    private final GeoIpResolverConfigValidator geoIpResolverConfigValidator;
    private final S3GeoIpFileService s3GeoIpFileService;
    private final NotificationService notificationService;
    private final ClusterConfigService clusterConfigService;
    private final EventBus eventBus;
    private Duration dbRefreshInterval = Duration.ZERO;
    private FileInfo cityDbFileInfo = FileInfo.empty();
    private FileInfo asnDbFileInfo = FileInfo.empty();
    private GeoIpResolverConfig config;

    @Inject
    public GeoIpDbFileChangeMonitorService(@Named(value="daemonScheduler") ScheduledExecutorService scheduler, EventBus eventBus, ClusterConfigService clusterConfigService, GeoIpVendorResolverService geoIpVendorResolverService, S3GeoIpFileService s3GeoIpFileService, NotificationService notificationService) {
        this.scheduler = Objects.requireNonNull(scheduler);
        this.eventBus = Objects.requireNonNull(eventBus);
        this.s3GeoIpFileService = Objects.requireNonNull(s3GeoIpFileService);
        this.clusterConfigService = Objects.requireNonNull(clusterConfigService);
        this.geoIpResolverConfigValidator = new GeoIpResolverConfigValidator(geoIpVendorResolverService, s3GeoIpFileService, clusterConfigService);
        this.notificationService = notificationService;
    }

    @Subscribe
    public void onClusterConfigChanged(ClusterConfigChangedEvent event) {
        if (GeoIpResolverConfig.class.getCanonicalName().equals(event.type())) {
            this.scheduler.schedule(this::updateConfiguration, 0L, TimeUnit.SECONDS);
        }
    }

    protected void startUp() throws Exception {
        this.eventBus.register((Object)this);
        this.updateConfiguration();
    }

    protected void shutDown() throws Exception {
        this.eventBus.unregister((Object)this);
    }

    private void refreshDatabases() {
        LOG.debug("Starting GeoIP database refresh");
        Map<DatabaseType, FileInfo.Change> changes = this.checkForChanges();
        if (changes.isEmpty()) {
            LOG.debug("GeoIP Database files have not changed--will not refresh");
        } else {
            GeoIpDbFileChangedEvent event = GeoIpDbFileChangedEvent.create();
            this.eventBus.post((Object)event);
        }
    }

    private Map<DatabaseType, FileInfo.Change> checkForChanges() {
        if (this.config == null) {
            this.config = this.getCurrentConfig();
        }
        EnumMap<DatabaseType, FileInfo.Change> changes = new EnumMap<DatabaseType, FileInfo.Change>(DatabaseType.class);
        if (this.config.useS3() && this.s3GeoIpFileService.fileRefreshRequired(this.config)) {
            try {
                LOG.debug("Pulling DB files from S3");
                this.s3GeoIpFileService.downloadFilesToTempLocation(this.config);
                GeoIpResolverConfig tempConfig = this.config.toBuilder().cityDbPath(this.s3GeoIpFileService.getTempCityFile()).asnDbPath(this.config.asnDbPath().isEmpty() ? "" : this.s3GeoIpFileService.getTempAsnFile()).build();
                Timer timer = new Timer((Reservoir)new UniformReservoir());
                this.geoIpResolverConfigValidator.validateGeoIpLocationResolver(tempConfig, timer);
                this.geoIpResolverConfigValidator.validateGeoIpAsnResolver(tempConfig, timer);
                this.s3GeoIpFileService.moveTempFilesToActive();
                LOG.debug("Pulled new files from S3");
            }
            catch (IllegalArgumentException | IllegalStateException validationError) {
                String message = "Geo Processor DB files from S3 failed validation. Upload valid files to S3. Leaving old files in place on disk.";
                this.sendFailedSyncNotification(message);
                LOG.error(message);
                this.s3GeoIpFileService.cleanupTempFiles();
                return changes;
            }
            catch (IOException | S3DownloadException e) {
                String message = "Failed to download Geo Processor DB files from S3. Unable to refresh. Leaving old files in place on disk.";
                this.sendFailedSyncNotification(message);
                LOG.error(message);
                return changes;
            }
        }
        FileInfo.Change cityDbChange = this.cityDbFileInfo.checkForChange();
        FileInfo.Change asnDbChange = this.asnDbFileInfo.checkForChange();
        DatabaseVendorType vendorType = this.config.databaseVendorType();
        if (cityDbChange.isChanged()) {
            changes.put(vendorType.getCityDbType(), cityDbChange);
            this.cityDbFileInfo = cityDbChange.fileInfo();
        }
        if (asnDbChange.isChanged()) {
            changes.put(vendorType.getAsnDbType(), asnDbChange);
            this.asnDbFileInfo = asnDbChange.fileInfo();
        }
        return changes;
    }

    private void updateConfiguration() {
        block8: {
            try {
                this.config = this.getCurrentConfig();
                if (this.config.enabled()) {
                    this.reScheduleRefreshIfNeeded();
                    String asnFile = this.config.asnDbPath();
                    String cityFile = this.config.cityDbPath();
                    if (this.config.useS3()) {
                        cityFile = this.s3GeoIpFileService.getActiveCityFile();
                        asnFile = this.s3GeoIpFileService.getActiveAsnFile();
                        if (this.s3GeoIpFileService.fileRefreshRequired(this.config)) {
                            try {
                                if (this.s3GeoIpFileService.fileRefreshRequired(this.config)) {
                                    this.s3GeoIpFileService.downloadFilesToTempLocation(this.config);
                                    this.s3GeoIpFileService.moveTempFilesToActive();
                                }
                            }
                            catch (IOException | S3DownloadException e) {
                                String commonMessage = "Failed to pull new Geo-Location Processor database files from S3.";
                                this.sendFailedSyncNotification(commonMessage + " Geo-Location Processor may not be functional on all nodes.");
                                LOG.error("{} Geo-Location Processor will not be functional on this node.", (Object)commonMessage);
                                return;
                            }
                        }
                    }
                    this.geoIpResolverConfigValidator.validate(this.config);
                    this.cityDbFileInfo = this.getDbFileInfo(cityFile);
                    this.asnDbFileInfo = this.getDbFileInfo(asnFile);
                    break block8;
                }
                LOG.debug("GeoIP Processor is disabled.  Will not schedule GeoIP database file change monitor");
                this.cancelScheduledRefreshTask();
                this.dbRefreshInterval = Duration.ZERO;
            }
            catch (IllegalArgumentException | IllegalStateException | ConfigValidationException e) {
                LOG.error("Error validating GeoIP Database files. {}", (Object)e.getMessage(), (Object)e);
            }
        }
    }

    private void cancelScheduledRefreshTask() {
        if (this.refreshTask != null) {
            boolean canceled = this.refreshTask.cancel(true);
            if (canceled) {
                LOG.debug("The GeoIP database file change monitor was running.  It has been cancelled");
                this.refreshTask = null;
            } else {
                LOG.warn("The GeoIP database file change monitor was running and failed to stop it");
            }
        }
    }

    private void reScheduleRefreshIfNeeded() {
        if (!this.dbRefreshInterval.equals(this.config.refreshIntervalAsDuration())) {
            boolean reschedule;
            boolean bl = reschedule = this.refreshTask == null || this.refreshTask.cancel(true);
            if (reschedule) {
                this.dbRefreshInterval = this.config.refreshIntervalAsDuration();
                this.scheduleDbRefresh();
            } else {
                LOG.warn("Failed to Cancel existing GeoIp Database Refresh Task.  Will not update refresh interval.");
            }
        }
    }

    private FileInfo getDbFileInfo(String path) {
        try {
            return FileInfo.forPath(Paths.get(path, new String[0]));
        }
        catch (Exception e) {
            return FileInfo.empty();
        }
    }

    private void scheduleDbRefresh() {
        try {
            long millis = this.dbRefreshInterval.toMillis();
            this.refreshTask = this.scheduler.scheduleAtFixedRate(this::refreshDatabases, millis, millis, TimeUnit.MILLISECONDS);
            LOG.debug("Scheduled GeoIP database refresh every '{}' Milliseconds", (Object)millis);
        }
        catch (Exception e) {
            LOG.error("Error scheduling GeoIP database refresh job. {}", (Object)e.getMessage(), (Object)e);
        }
    }

    private GeoIpResolverConfig getCurrentConfig() {
        return this.clusterConfigService.getOrDefault(GeoIpResolverConfig.class, GeoIpResolverConfig.defaultConfig());
    }

    private void sendFailedSyncNotification(String description) {
        Notification notification = this.notificationService.buildNow().addType(Notification.Type.GENERIC).addSeverity(Notification.Severity.NORMAL).addDetail("title", "Geo-Location Processor S3 Sync Failure").addDetail("description", description);
        this.notificationService.publishIfFirst(notification);
    }
}

