/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.plugins.map.config;

import com.google.auto.value.AutoValue;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import org.graylog.plugins.map.config.AutoValue_S3GeoIpFileService_BucketsAndKeys;
import org.graylog.plugins.map.config.GeoIpProcessorConfig;
import org.graylog.plugins.map.config.GeoIpResolverConfig;
import org.graylog.plugins.map.config.S3DownloadException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.S3Object;

@Singleton
public class S3GeoIpFileService {
    private static final Logger LOG = LoggerFactory.getLogger(S3GeoIpFileService.class);
    public static final String S3_BUCKET_PREFIX = "s3://";
    public static final String ACTIVE_ASN_FILE = "asn-from-s3.mmdb";
    public static final String ACTIVE_CITY_FILE = "standard_location-from-s3.mmdb";
    public static final String TEMP_ASN_FILE = "temp-asn-from-s3.mmdb";
    public static final String TEMP_CITY_FILE = "temp-standard_location-from-s3.mmdb";
    public static final String NULL_S3_CLIENT_MESSAGE = "Unable to create DefaultCredentialsProvider for the S3 Client. Geo Location Processor S3 file refresh is disabled.";
    private S3Client s3Client;
    private final Path downloadDir;
    private final Path asnPath;
    private final Path cityPath;
    private final Path tempAsnPath;
    private final Path tempCityPath;
    private Instant asnFileLastModified = Instant.EPOCH;
    private Instant cityFileLastModified = Instant.EPOCH;
    private Instant tempAsnFileLastModified = null;
    private Instant tempCityFileLastModified = null;

    @Inject
    public S3GeoIpFileService(GeoIpProcessorConfig config) {
        this.downloadDir = config.getS3DownloadLocation();
        this.asnPath = this.downloadDir.resolve(ACTIVE_ASN_FILE);
        this.cityPath = this.downloadDir.resolve(ACTIVE_CITY_FILE);
        this.tempAsnPath = this.downloadDir.resolve(TEMP_ASN_FILE);
        this.tempCityPath = this.downloadDir.resolve(TEMP_CITY_FILE);
        if (Files.exists(this.cityPath, new LinkOption[0])) {
            this.cityFileLastModified = Instant.ofEpochMilli(this.cityPath.toFile().lastModified());
        }
        if (Files.exists(this.asnPath, new LinkOption[0])) {
            this.asnFileLastModified = Instant.ofEpochMilli(this.asnPath.toFile().lastModified());
        }
    }

    public void downloadFilesToTempLocation(GeoIpResolverConfig config) throws S3DownloadException {
        if (this.s3ClientIsNull() || !this.ensureDownloadDirectory()) {
            return;
        }
        try {
            this.cleanupTempFiles();
            BucketsAndKeys bucketsAndKeys = this.getBucketsAndKeys(config);
            GetObjectResponse cityResponse = this.getS3Client().getObject((GetObjectRequest)GetObjectRequest.builder().bucket(bucketsAndKeys.cityBucket()).key(bucketsAndKeys.cityKey()).build(), this.tempCityPath);
            this.setFilePermissions(this.tempCityPath);
            this.tempCityFileLastModified = cityResponse.lastModified();
            if (!config.asnDbPath().isEmpty()) {
                GetObjectResponse asnResponse = this.getS3Client().getObject((GetObjectRequest)GetObjectRequest.builder().bucket(bucketsAndKeys.asnBucket()).key(bucketsAndKeys.asnKey()).build(), this.tempAsnPath);
                this.setFilePermissions(this.tempAsnPath);
                this.tempAsnFileLastModified = asnResponse.lastModified();
            }
        }
        catch (Exception e) {
            LOG.error("Failed to retrieve S3 files. {}", (Object)e.toString());
            this.cleanupTempFiles();
            throw new S3DownloadException(e.getMessage());
        }
    }

    public boolean fileRefreshRequired(GeoIpResolverConfig config) {
        if (this.s3ClientIsNull()) {
            return false;
        }
        if (!Files.exists(this.cityPath, new LinkOption[0]) || !config.asnDbPath().isEmpty() && !Files.exists(this.asnPath, new LinkOption[0])) {
            return true;
        }
        BucketsAndKeys bucketsAndKeys = this.getBucketsAndKeys(config);
        S3Object cityObj = this.getS3Object(bucketsAndKeys.cityBucket(), bucketsAndKeys.cityKey());
        if (cityObj == null) {
            LOG.warn("No city database file '{}' found in S3 bucket '{}'. Aborting S3 file refresh.", (Object)bucketsAndKeys.cityKey(), (Object)bucketsAndKeys.cityBucket());
            return false;
        }
        boolean asnUpdated = false;
        if (!config.asnDbPath().isEmpty()) {
            S3Object asnObj = this.getS3Object(bucketsAndKeys.asnBucket(), bucketsAndKeys.asnKey());
            if (asnObj == null) {
                LOG.warn("No ASN database file '{}' found in S3 bucket '{}'. Aborting S3 file refresh.", (Object)bucketsAndKeys.asnKey(), (Object)bucketsAndKeys.asnBucket());
                return false;
            }
            asnUpdated = asnObj.lastModified().isAfter(this.asnFileLastModified);
        }
        return cityObj.lastModified().isAfter(this.cityFileLastModified) || asnUpdated;
    }

    public void moveTempFilesToActive() throws IOException {
        Files.move(this.tempCityPath, this.cityPath, StandardCopyOption.REPLACE_EXISTING);
        this.cityFileLastModified = this.tempCityFileLastModified;
        if (Files.exists(this.tempAsnPath, new LinkOption[0])) {
            Files.move(this.tempAsnPath, this.asnPath, StandardCopyOption.REPLACE_EXISTING);
            this.asnFileLastModified = this.tempAsnFileLastModified;
        }
        this.tempAsnFileLastModified = null;
        this.tempCityFileLastModified = null;
    }

    public String getTempAsnFile() {
        return this.tempAsnPath.toString();
    }

    public String getTempCityFile() {
        return this.tempCityPath.toString();
    }

    public String getActiveAsnFile() {
        return this.asnPath.toString();
    }

    public String getActiveCityFile() {
        return this.cityPath.toString();
    }

    public void cleanupTempFiles() {
        try {
            if (Files.exists(this.tempAsnPath, new LinkOption[0])) {
                Files.delete(this.tempAsnPath);
            }
            if (Files.exists(this.tempCityPath, new LinkOption[0])) {
                Files.delete(this.tempCityPath);
            }
            this.tempAsnFileLastModified = null;
            this.tempCityFileLastModified = null;
        }
        catch (IOException e) {
            LOG.error("Failed to delete temporary Geo Processor DB files. Manual cleanup of '{}' and '{}' may be necessary", (Object)this.getTempAsnFile(), (Object)this.getTempCityFile());
        }
    }

    public boolean s3ClientIsNull() {
        return this.getS3Client() == null;
    }

    private void setFilePermissions(Path filePath) {
        File tempFile = filePath.toFile();
        if (!(tempFile.setExecutable(true) && tempFile.setWritable(true) && tempFile.setReadable(true, false))) {
            LOG.warn("Failed to set file permissions on newly downloaded Geo Location Processor database file {}. Geo Location Processing may be unable to function correctly without these file permissions", (Object)filePath);
        }
    }

    private BucketsAndKeys getBucketsAndKeys(GeoIpResolverConfig config) {
        String cityFile = config.cityDbPath();
        int cityLastSlash = cityFile.lastIndexOf("/");
        String cityBucket = cityFile.substring(S3_BUCKET_PREFIX.length(), cityLastSlash);
        String cityKey = cityFile.substring(cityLastSlash + 1);
        LOG.debug("City Bucket = {}, City Key = {}", (Object)cityBucket, (Object)cityKey);
        String asnBucket = "";
        String asnKey = "";
        if (!config.asnDbPath().isEmpty()) {
            String asnFile = config.asnDbPath();
            int asnLastSlash = asnFile.lastIndexOf("/");
            asnBucket = asnFile.substring(S3_BUCKET_PREFIX.length(), asnLastSlash);
            asnKey = asnFile.substring(asnLastSlash + 1);
        }
        LOG.debug("ASN Bucket = {}, ASN Key = {}", (Object)asnBucket, (Object)asnKey);
        return BucketsAndKeys.create(asnBucket, asnKey, cityBucket, cityKey);
    }

    private S3Object getS3Object(String bucket, String key) {
        ListObjectsV2Request listObjectsRequest = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(bucket).prefix(key).build();
        ListObjectsV2Response listObjectsResponse = this.getS3Client().listObjectsV2(listObjectsRequest);
        S3Object obj = null;
        for (S3Object o : listObjectsResponse.contents()) {
            if (!o.key().equals(key)) continue;
            obj = o;
            break;
        }
        return obj;
    }

    private S3Client getS3Client() {
        if (this.s3Client == null) {
            try {
                this.s3Client = S3Client.create();
            }
            catch (Exception e) {
                LOG.warn(NULL_S3_CLIENT_MESSAGE);
                LOG.debug("If not trying to use the Geo Location Processor S3 file refresh feature, the following error can safely be ignored.\n\tERROR : {}", (Object)e.getMessage());
            }
        }
        return this.s3Client;
    }

    private boolean ensureDownloadDirectory() {
        if (!Files.exists(this.downloadDir, new LinkOption[0])) {
            try {
                Files.createDirectory(this.downloadDir, new FileAttribute[0]);
            }
            catch (IOException e) {
                LOG.error("Unable to create S3 download directory at {}. Geo-Location Processor S3 file refresh will be broken on this node.", (Object)this.downloadDir.toAbsolutePath());
            }
        }
        return Files.exists(this.downloadDir, new LinkOption[0]);
    }

    @AutoValue
    static abstract class BucketsAndKeys {
        BucketsAndKeys() {
        }

        public abstract String asnBucket();

        public abstract String asnKey();

        public abstract String cityBucket();

        public abstract String cityKey();

        public static BucketsAndKeys create(String asnBucket, String asnKey, String cityBucket, String cityKey) {
            return new AutoValue_S3GeoIpFileService_BucketsAndKeys(asnBucket, asnKey, cityBucket, cityKey);
        }
    }
}

