/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.system.jobs;

import com.codahale.metrics.InstrumentedScheduledExecutorService;
import com.codahale.metrics.MetricRegistry;
import com.eaio.uuid.UUID;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.instrumentation.annotations.WithSpan;
import jakarta.inject.Inject;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.graylog.tracing.GraylogSemanticAttributes;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJob;
import org.graylog2.system.jobs.SystemJobConcurrencyException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SystemJobManager {
    private static final Logger LOG = LoggerFactory.getLogger(SystemJobManager.class);
    private static final int THREAD_POOL_SIZE = 15;
    private final ActivityWriter activityWriter;
    private final ScheduledExecutorService executor;
    private final Map<String, SystemJob> jobs;

    @Inject
    public SystemJobManager(ActivityWriter activityWriter, MetricRegistry metricRegistry) {
        this.activityWriter = activityWriter;
        this.executor = this.executorService(metricRegistry);
        this.jobs = new ConcurrentHashMap<String, SystemJob>();
    }

    private ScheduledExecutorService executorService(MetricRegistry metricRegistry) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("systemjob-executor-%d").build();
        return new InstrumentedScheduledExecutorService(Executors.newScheduledThreadPool(15, threadFactory), metricRegistry, MetricRegistry.name(this.getClass(), (String[])new String[]{"executor-service"}));
    }

    public String submit(SystemJob job) throws SystemJobConcurrencyException {
        return this.submitWithDelay(job, 0L, TimeUnit.SECONDS);
    }

    public synchronized String submitWithDelay(final SystemJob job, final long delay, TimeUnit timeUnit) throws SystemJobConcurrencyException {
        if (delay == 0L) {
            this.checkAllowedConcurrency(job);
        }
        final String jobClass = job.getClass().getCanonicalName();
        job.setId(new UUID().toString());
        this.jobs.put(job.getId(), job);
        this.executor.schedule(new Runnable(){

            @Override
            public void run() {
                try {
                    if (delay > 0L) {
                        SystemJobManager.this.checkAllowedConcurrency(job);
                    }
                    job.markStarted();
                    Stopwatch x = Stopwatch.createStarted();
                    SystemJobManager.this.executeJob(job);
                    x.stop();
                    String msg = "SystemJob <" + job.getId() + "> [" + jobClass + "] finished in " + x.elapsed(TimeUnit.MILLISECONDS) + "ms.";
                    LOG.info(msg);
                    SystemJobManager.this.activityWriter.write(new Activity(msg, SystemJobManager.class));
                }
                catch (SystemJobConcurrencyException x) {
                }
                catch (Exception e) {
                    LOG.error("Unhandled error while running SystemJob <" + job.getId() + "> [" + jobClass + "]", (Throwable)e);
                }
                finally {
                    SystemJobManager.this.jobs.remove(job.getId());
                }
            }
        }, delay, timeUnit);
        LOG.info("Submitted SystemJob <{}> [{}]", (Object)job.getId(), (Object)jobClass);
        return job.getId();
    }

    @WithSpan
    private void executeJob(SystemJob job) {
        Span.current().setAttribute(GraylogSemanticAttributes.SYSTEM_JOB_TYPE, (Object)job.getClassName());
        job.execute();
    }

    protected void checkAllowedConcurrency(SystemJob job) throws SystemJobConcurrencyException {
        int concurrent = this.concurrentJobs(job.getClass());
        if (concurrent >= job.maxConcurrency()) {
            throw new SystemJobConcurrencyException("The maximum of parallel [" + job.getClass().getCanonicalName() + "]\u00a0is locked to <" + job.maxConcurrency() + "> but <" + concurrent + "> are running.");
        }
    }

    public Map<String, SystemJob> getRunningJobs() {
        return this.jobs;
    }

    public int concurrentJobs(Class jobClass) {
        int concurrent = 0;
        for (SystemJob job : this.jobs.values()) {
            if (!job.getClass().equals(jobClass)) continue;
            ++concurrent;
        }
        return concurrent;
    }
}

