/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.scheduler.worker;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.InstrumentedExecutorService;
import com.codahale.metrics.InstrumentedThreadFactory;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.inject.assistedinject.Assisted;
import jakarta.inject.Inject;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.graylog2.system.shutdown.GracefulShutdownHook;
import org.graylog2.system.shutdown.GracefulShutdownService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobWorkerPool
implements GracefulShutdownHook {
    private static final Logger LOG = LoggerFactory.getLogger(JobWorkerPool.class);
    private static final String NAME_PREFIX = "job-worker-pool";
    private static final String EXECUTOR_NAME = "job-worker-pool-executor";
    private static final Pattern NAME_PATTERN = Pattern.compile("[a-zA-Z0-9\\-]+");
    private final int poolSize;
    private final ExecutorService executor;
    private final Semaphore slots;
    private final Runnable shutdownCallback;

    @Inject
    public JobWorkerPool(@Assisted String name, @Assisted int poolSize, @Assisted Runnable shutdownCallback, GracefulShutdownService gracefulShutdownService, MetricRegistry metricRegistry) {
        this.shutdownCallback = shutdownCallback;
        this.poolSize = poolSize;
        Preconditions.checkArgument((boolean)NAME_PATTERN.matcher(name).matches(), (String)"Pool name must match %s", (Object)NAME_PATTERN);
        this.executor = JobWorkerPool.buildExecutor(name, poolSize, metricRegistry);
        this.slots = new Semaphore(poolSize, true);
        this.registerMetrics(metricRegistry, poolSize);
        gracefulShutdownService.register(this);
    }

    public int freeSlots() {
        return this.slots.availablePermits();
    }

    public boolean hasFreeSlots() {
        return this.freeSlots() > 0;
    }

    public boolean anySlotsUsed() {
        return this.poolSize != this.freeSlots();
    }

    public boolean execute(Runnable job) {
        boolean acquired = this.slots.tryAcquire();
        if (!acquired) {
            return false;
        }
        try {
            this.executor.execute(() -> {
                try {
                    job.run();
                }
                catch (Exception e) {
                    LOG.error("Unhandled job execution error", (Throwable)e);
                }
                finally {
                    this.slots.release();
                }
            });
            return true;
        }
        catch (RejectedExecutionException e) {
            this.slots.release();
            return false;
        }
    }

    @Override
    public void doGracefulShutdown() throws Exception {
        this.executor.shutdown();
        if (!this.executor.awaitTermination(60L, TimeUnit.SECONDS)) {
            LOG.warn("Timeout shutting down worker pool after 60 seconds");
        } else {
            this.shutdownCallback.run();
        }
    }

    private static ExecutorService buildExecutor(String name, int poolSize, MetricRegistry metricRegistry) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("job-worker-pool[" + name + "]-%d").setUncaughtExceptionHandler((t, e) -> LOG.error("Unhandled exception", e)).build();
        InstrumentedThreadFactory itf = new InstrumentedThreadFactory(threadFactory, metricRegistry, MetricRegistry.name(JobWorkerPool.class, (String[])new String[]{name}));
        SynchronousQueue<Runnable> workQueue = new SynchronousQueue<Runnable>();
        ThreadPoolExecutor executor = new ThreadPoolExecutor(1, poolSize, 60L, TimeUnit.SECONDS, workQueue, (ThreadFactory)itf);
        return new InstrumentedExecutorService((ExecutorService)executor, metricRegistry, MetricRegistry.name((String)EXECUTOR_NAME, (String[])new String[]{name}));
    }

    private void registerMetrics(MetricRegistry metricRegistry, int poolSize) {
        metricRegistry.register(MetricRegistry.name(this.getClass(), (String[])new String[]{"waiting_for_slots"}), (Metric)((Gauge)this.slots::getQueueLength));
        metricRegistry.register(MetricRegistry.name(this.getClass(), (String[])new String[]{"free_slots"}), (Metric)((Gauge)this::freeSlots));
        metricRegistry.register(MetricRegistry.name(this.getClass(), (String[])new String[]{"total_slots"}), (Metric)((Gauge)() -> poolSize));
    }

    public static interface Factory {
        public JobWorkerPool create(String var1, int var2, Runnable var3);
    }
}

