/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.shared.inputs;

import com.codahale.metrics.InstrumentedExecutorService;
import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import jakarta.inject.Inject;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import org.graylog2.Configuration;
import org.graylog2.cluster.leader.LeaderElectionService;
import org.graylog2.plugin.IOState;
import org.graylog2.plugin.InputFailureRecorder;
import org.graylog2.plugin.buffers.InputBuffer;
import org.graylog2.plugin.inputs.MessageInput;
import org.graylog2.shared.inputs.InputRegistry;
import org.graylog2.shared.inputs.PersistedInputs;
import org.graylog2.shared.utilities.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InputLauncher {
    private static final Logger LOG = LoggerFactory.getLogger(InputLauncher.class);
    private final IOState.Factory<MessageInput> inputStateFactory;
    private final InputBuffer inputBuffer;
    private final PersistedInputs persistedInputs;
    private final InputRegistry inputRegistry;
    private final ExecutorService executor;
    private final Configuration configuration;
    private final LeaderElectionService leaderElectionService;

    @Inject
    public InputLauncher(IOState.Factory<MessageInput> inputStateFactory, InputBuffer inputBuffer, PersistedInputs persistedInputs, InputRegistry inputRegistry, MetricRegistry metricRegistry, Configuration configuration, LeaderElectionService leaderElectionService) {
        this.inputStateFactory = inputStateFactory;
        this.inputBuffer = inputBuffer;
        this.persistedInputs = persistedInputs;
        this.inputRegistry = inputRegistry;
        this.executor = this.executorService(metricRegistry);
        this.configuration = configuration;
        this.leaderElectionService = leaderElectionService;
    }

    private ExecutorService executorService(MetricRegistry metricRegistry) {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("inputs-%d").build();
        return new InstrumentedExecutorService(Executors.newCachedThreadPool(threadFactory), metricRegistry, MetricRegistry.name(this.getClass(), (String[])new String[]{"executor-service"}));
    }

    public IOState<MessageInput> launch(final MessageInput input) {
        IOState<MessageInput> inputState;
        Preconditions.checkNotNull((Object)input);
        if (this.inputRegistry.getInputState(input.getId()) == null) {
            inputState = this.inputStateFactory.create(input);
            this.inputRegistry.add(inputState);
        } else {
            inputState = this.inputRegistry.getInputState(input.getId());
            switch (inputState.getState()) {
                case RUNNING: 
                case STARTING: 
                case FAILING: {
                    return inputState;
                }
            }
            inputState.setStoppable(input);
        }
        this.executor.submit(new Runnable(){

            @Override
            public void run() {
                LOG.debug("Starting [{}] input {}", (Object)input.getClass().getCanonicalName(), (Object)input.toIdentifier());
                try {
                    input.checkConfiguration();
                    inputState.setState(IOState.Type.STARTING);
                    input.launch(InputLauncher.this.inputBuffer, new InputFailureRecorder(inputState));
                    inputState.setState(IOState.Type.RUNNING);
                    String msg = "Completed starting [" + input.getClass().getCanonicalName() + "] input " + input.toIdentifier();
                    LOG.debug(msg);
                }
                catch (Exception e) {
                    InputLauncher.this.handleLaunchException(e, inputState);
                }
            }
        });
        return inputState;
    }

    protected void handleLaunchException(Throwable e, IOState<MessageInput> inputState) {
        MessageInput input = inputState.getStoppable();
        StringBuilder msg = new StringBuilder("The [" + input.getClass().getCanonicalName() + "] input " + input.toIdentifier() + " misfired. Reason: ");
        String causeMsg = ExceptionUtils.getRootCauseMessage(e);
        msg.append(causeMsg);
        LOG.error(msg.toString(), e);
        inputState.setState(IOState.Type.FAILED, causeMsg);
    }

    public void launchAllPersisted() {
        for (MessageInput input : this.persistedInputs) {
            if (this.leaderStatusInhibitsLaunch(input)) {
                LOG.info("Not launching 'onlyOnePerCluster' input {} because this node is not the leader.", (Object)input.toIdentifier());
                continue;
            }
            if (this.shouldStartAutomatically(input)) {
                LOG.info("Launching input {} - desired state is {}", (Object)input.toIdentifier(), (Object)input.getDesiredState());
                input.initialize();
                this.launch(input);
                continue;
            }
            LOG.info("Not auto-starting input {} - desired state is {}", (Object)input.toIdentifier(), (Object)input.getDesiredState());
        }
    }

    public boolean shouldStartAutomatically(MessageInput input) {
        return this.configuration.getAutoRestartInputs() || input.getDesiredState().equals((Object)IOState.Type.RUNNING);
    }

    public boolean leaderStatusInhibitsLaunch(MessageInput input) {
        return input.onlyOnePerCluster() && input.isGlobal() != false && !this.leaderElectionService.isLeader();
    }
}

