/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.scheduler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.AbstractExecutionThreadService;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.inject.Singleton;
import java.time.Duration;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.graylog.scheduler.JobExecutionEngine;
import org.graylog.scheduler.JobSchedulerConfig;
import org.graylog.scheduler.clock.JobSchedulerClock;
import org.graylog.scheduler.eventbus.JobCompletedEvent;
import org.graylog.scheduler.eventbus.JobSchedulerEventBus;
import org.graylog.scheduler.worker.JobWorkerPool;
import org.graylog2.plugin.ServerStatus;
import org.graylog2.plugin.Tools;
import org.graylog2.system.shutdown.GracefulShutdownHook;
import org.graylog2.system.shutdown.GracefulShutdownService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class JobSchedulerService
extends AbstractExecutionThreadService
implements GracefulShutdownHook {
    private static final Logger LOG = LoggerFactory.getLogger(JobSchedulerService.class);
    private final JobExecutionEngine jobExecutionEngine;
    private final JobSchedulerConfig schedulerConfig;
    private final JobSchedulerClock clock;
    private final JobSchedulerEventBus schedulerEventBus;
    private final ServerStatus serverStatus;
    private final JobWorkerPool workerPool;
    private final GracefulShutdownService gracefulShutdownService;
    private final Duration shutdownTimeout;
    private final com.github.joschi.jadconfig.util.Duration loopSleepDuration;
    private final InterruptibleSleeper sleeper = new InterruptibleSleeper();
    private final ScheduledExecutorService jobHeartbeatExecutor = this.createJobHeartbeatExecutor();
    private Thread executionThread;

    @Inject
    public JobSchedulerService(JobExecutionEngine.Factory engineFactory, JobWorkerPool.Factory workerPoolFactory, JobSchedulerConfig schedulerConfig, JobSchedulerClock clock, JobSchedulerEventBus schedulerEventBus, ServerStatus serverStatus, GracefulShutdownService gracefulShutdownService, @Named(value="shutdown_timeout") int shutdownTimeoutMs, @Named(value="job_scheduler_loop_sleep_duration") com.github.joschi.jadconfig.util.Duration loopSleepDuration) {
        this.workerPool = workerPoolFactory.create("system", schedulerConfig.numberOfWorkerThreads());
        this.jobExecutionEngine = engineFactory.create(this.workerPool);
        this.schedulerConfig = schedulerConfig;
        this.clock = clock;
        this.schedulerEventBus = schedulerEventBus;
        this.serverStatus = serverStatus;
        this.gracefulShutdownService = gracefulShutdownService;
        this.shutdownTimeout = Duration.ofMillis(shutdownTimeoutMs);
        this.loopSleepDuration = loopSleepDuration;
    }

    private ScheduledExecutorService createJobHeartbeatExecutor() {
        return Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("job-heartbeat-%d").setDaemon(true).setUncaughtExceptionHandler((Thread.UncaughtExceptionHandler)new Tools.LogUncaughtExceptionHandler(LOG)).build());
    }

    protected void startUp() throws Exception {
        this.jobHeartbeatExecutor.scheduleAtFixedRate(this::updateLockedJobs, 0L, 15L, TimeUnit.SECONDS);
        this.schedulerEventBus.register(this);
        this.gracefulShutdownService.register(this);
        this.executionThread = Thread.currentThread();
    }

    protected void run() throws Exception {
        LOG.debug("Waiting for server to enter RUNNING status before starting the scheduler loop");
        try {
            this.serverStatus.awaitRunning();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.debug("Was interrupted while waiting for server to enter RUNNING state. Aborting.");
            return;
        }
        LOG.debug("Server entered RUNNING state, starting scheduler loop");
        boolean executionEnabled = true;
        while (this.isRunning()) {
            if (!this.schedulerConfig.canExecute()) {
                executionEnabled = this.logExecutionConfigState(executionEnabled, false);
                this.clock.sleepUninterruptibly(1L, TimeUnit.SECONDS);
                continue;
            }
            executionEnabled = this.logExecutionConfigState(executionEnabled, true);
            LOG.debug("Starting scheduler loop iteration");
            try {
                if (!this.jobExecutionEngine.execute() && this.isRunning() && this.sleeper.sleep(this.loopSleepDuration.getQuantity(), this.loopSleepDuration.getUnit())) {
                    LOG.debug("Waited for {} {} because there are either no free worker threads or no runnable triggers", (Object)this.loopSleepDuration.getQuantity(), (Object)this.loopSleepDuration.getUnit());
                }
            }
            catch (InterruptedException e) {
                LOG.debug("Received interrupted exception", (Throwable)e);
            }
            catch (Exception e) {
                LOG.error("Error running job execution engine", (Throwable)e);
            }
            LOG.debug("Ending scheduler loop iteration");
        }
    }

    @Subscribe
    public void handleJobCompleted(JobCompletedEvent triggerCompletedEvent) {
        this.sleeper.interrupt();
    }

    @Override
    public void doGracefulShutdown() throws Exception {
        this.stopAsync().awaitTerminated(this.shutdownTimeout);
    }

    protected void triggerShutdown() {
        this.schedulerEventBus.unregister(this);
        this.jobExecutionEngine.shutdown();
        try {
            this.workerPool.shutdown(this.shutdownTimeout);
        }
        catch (InterruptedException e) {
            LOG.debug("Interrupted while waiting for worker pool shutdown");
            Thread.currentThread().interrupt();
        }
        this.jobHeartbeatExecutor.shutdown();
        if (this.gracefulShutdownService.isRunning()) {
            this.gracefulShutdownService.unregister(this);
        }
        this.executionThread.interrupt();
    }

    private void updateLockedJobs() {
        this.jobExecutionEngine.updateLockedJobs();
    }

    private boolean logExecutionConfigState(boolean previouslyEnabled, boolean nowEnabled) {
        if (previouslyEnabled && !nowEnabled) {
            LOG.info("Job scheduler execution is disabled. Waiting and trying again until enabled.");
        } else if (!previouslyEnabled && nowEnabled) {
            LOG.info("Job scheduler execution is now enabled. Proceeding.");
        }
        return nowEnabled;
    }

    @VisibleForTesting
    static class InterruptibleSleeper {
        private final Semaphore semaphore;

        InterruptibleSleeper() {
            this(new Semaphore(1));
        }

        @VisibleForTesting
        InterruptibleSleeper(Semaphore semaphore) {
            this.semaphore = semaphore;
        }

        public boolean sleep(long duration, TimeUnit unit) throws InterruptedException {
            this.semaphore.drainPermits();
            return !this.semaphore.tryAcquire(duration, unit);
        }

        public void interrupt() {
            this.semaphore.release();
        }
    }
}

