/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer.rotation.strategies;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nullable;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.graylog2.configuration.ElasticsearchConfiguration;
import org.graylog2.indexer.IndexSet;
import org.graylog2.indexer.indexset.IndexSetConfig;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.rotation.common.IndexRotator;
import org.graylog2.indexer.rotation.strategies.TimeBasedRotationStrategyConfig;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.indexer.rotation.RotationStrategy;
import org.graylog2.plugin.indexer.rotation.RotationStrategyConfig;
import org.joda.time.DateTime;
import org.joda.time.DateTimeField;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadablePeriod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class TimeBasedRotationStrategy
implements RotationStrategy {
    public static final String NAME = "time";
    public static final String OVERRIDE_HINT = "(elasticsearch_max_write_index_age overrides configured period)";
    private static final Logger log = LoggerFactory.getLogger(TimeBasedRotationStrategy.class);
    private final Indices indices;
    private final ElasticsearchConfiguration elasticsearchConfiguration;
    private final IndexRotator indexRotator;
    private Map<String, DateTime> anchor;

    @Inject
    public TimeBasedRotationStrategy(Indices indices, ElasticsearchConfiguration elasticsearchConfiguration, IndexRotator indexRotator) {
        this.indices = indices;
        this.elasticsearchConfiguration = elasticsearchConfiguration;
        this.indexRotator = indexRotator;
        this.anchor = new ConcurrentHashMap<String, DateTime>();
    }

    static DateTime determineRotationPeriodAnchor(@Nullable DateTime lastAnchor, Period period) {
        Period normalized = period.normalizedStandard();
        int years = normalized.getYears();
        int months = normalized.getMonths();
        int weeks = normalized.getWeeks();
        int days = normalized.getDays();
        int hours = normalized.getHours();
        int minutes = normalized.getMinutes();
        int seconds = normalized.getSeconds();
        if (years == 0 && months == 0 && weeks == 0 && days == 0 && hours == 0 && minutes == 0 && seconds == 0) {
            throw new IllegalArgumentException("Invalid rotation period specified");
        }
        DateTimeFieldType largestStrideType = null;
        if (seconds > 0) {
            largestStrideType = DateTimeFieldType.secondOfMinute();
        }
        if (minutes > 0) {
            largestStrideType = DateTimeFieldType.minuteOfHour();
        }
        if (hours > 0) {
            largestStrideType = DateTimeFieldType.hourOfDay();
        }
        if (days > 0) {
            largestStrideType = DateTimeFieldType.dayOfMonth();
        }
        if (weeks > 0) {
            largestStrideType = DateTimeFieldType.weekOfWeekyear();
        }
        if (months > 0) {
            largestStrideType = DateTimeFieldType.monthOfYear();
        }
        if (years > 0) {
            largestStrideType = DateTimeFieldType.year();
        }
        if (largestStrideType == null) {
            throw new IllegalArgumentException("Could not determine rotation stride length.");
        }
        DateTime anchorTime = TimeBasedRotationStrategy.anchorTimeFrom(lastAnchor);
        DateTimeField field = largestStrideType.getField(anchorTime.getChronology());
        int periodValue = normalized.get(largestStrideType.getDurationType());
        long fieldValue = field.roundFloor(anchorTime.getMillis());
        int fieldValueInUnit = field.get(fieldValue);
        if (periodValue == 0) {
            log.warn("Determining stride length failed because of a 0 period. Defaulting back to 1 period to avoid crashing, but this is a bug!");
            periodValue = 1;
        }
        long difference = fieldValueInUnit % periodValue;
        long newValue = field.add(fieldValue, -1L * difference);
        return new DateTime(newValue, DateTimeZone.UTC);
    }

    private static DateTime anchorTimeFrom(@Nullable DateTime lastAnchor) {
        if (lastAnchor != null && !lastAnchor.getZone().equals((Object)DateTimeZone.UTC)) {
            return lastAnchor.withZone(DateTimeZone.UTC);
        }
        return (DateTime)MoreObjects.firstNonNull((Object)lastAnchor, (Object)Tools.nowUTC());
    }

    private static boolean isLonger(Period p1, Period p2) {
        Instant now = Instant.now();
        Duration d1 = p1.toDurationTo((ReadableInstant)now);
        Duration d2 = p2.toDurationTo((ReadableInstant)now);
        return d1.isLongerThan((ReadableDuration)d2);
    }

    @Override
    public void rotate(IndexSet indexSet) {
        this.indexRotator.rotate(indexSet, this::shouldRotate);
    }

    @Override
    public Class<? extends RotationStrategyConfig> configurationClass() {
        return TimeBasedRotationStrategyConfig.class;
    }

    @Override
    public RotationStrategyConfig defaultConfiguration() {
        return TimeBasedRotationStrategyConfig.builder().maxRotationPeriod(this.elasticsearchConfiguration.getMaxWriteIndexAge()).build();
    }

    public void reset() {
        this.anchor.clear();
    }

    private IndexRotator.Result createResult(boolean shouldRotate, String message) {
        return IndexRotator.createResult(shouldRotate, message, this.getClass().getCanonicalName());
    }

    @Nullable
    private IndexRotator.Result shouldRotate(String index, IndexSet indexSet) {
        IndexSetConfig indexSetConfig = Objects.requireNonNull(indexSet.getConfig(), "Index set configuration must not be null");
        String indexSetId = indexSetConfig.id();
        Preconditions.checkState((!Strings.isNullOrEmpty((String)index) ? 1 : 0) != 0, (Object)"Index name must not be null or empty");
        Preconditions.checkState((!Strings.isNullOrEmpty((String)indexSetId) ? 1 : 0) != 0, (Object)"Index set ID must not be null or empty");
        Preconditions.checkState((boolean)(indexSetConfig.rotationStrategyConfig() instanceof TimeBasedRotationStrategyConfig), (String)"Invalid rotation strategy config <%s> for index set <%s>", (Object)indexSetConfig.rotationStrategyConfig().getClass().getCanonicalName(), (Object)indexSet);
        TimeBasedRotationStrategyConfig config = (TimeBasedRotationStrategyConfig)indexSetConfig.rotationStrategyConfig();
        Pair<Period, Boolean> normalizedRotationPeriod = this.getNormalizedRotationPeriod(config);
        Period normalizedPeriod = (Period)normalizedRotationPeriod.getLeft();
        boolean overriding = (Boolean)normalizedRotationPeriod.getRight();
        if (!this.anchor.containsKey(indexSetId)) {
            this.indices.indexCreationDate(index).ifPresent(creationDate -> {
                DateTime currentAnchor = TimeBasedRotationStrategy.determineRotationPeriodAnchor(creationDate, normalizedPeriod);
                this.anchor.put(indexSetId, currentAnchor);
            });
            if (!this.anchor.containsKey(indexSetId)) {
                return this.createResult(true, "No known previous rotation time, forcing index rotation now.");
            }
        }
        DateTime now = Tools.nowUTC();
        DateTime currentAnchor = this.anchor.get(indexSetId);
        DateTime nextRotation = currentAnchor.plus((ReadablePeriod)normalizedPeriod);
        if (nextRotation.isAfter((ReadableInstant)now)) {
            String message = new MessageFormat("Next rotation at {0} {1}", Locale.ENGLISH).format(new Object[]{nextRotation, overriding ? OVERRIDE_HINT : ""});
            return this.createResult(false, message);
        }
        DateTime nextAnchor = this.calculateNextAnchor(currentAnchor, normalizedPeriod, now);
        this.anchor.put(indexSetId, nextAnchor);
        if (!config.rotateEmptyIndexSet() && this.isEmptyIndexSet(indexSet)) {
            log.debug("Index set {} contains no messages, skipping rotation!", (Object)indexSet);
            String message = new MessageFormat("Index set contains no messages, skipping rotation! Next rotation at {0} {1}", Locale.ENGLISH).format(new Object[]{nextAnchor, overriding ? OVERRIDE_HINT : ""});
            return this.createResult(false, message);
        }
        String message = new MessageFormat("Rotation period {0} elapsed, next rotation at {1} {2}", Locale.ENGLISH).format(new Object[]{now, nextAnchor, overriding ? OVERRIDE_HINT : ""});
        return this.createResult(true, message);
    }

    private boolean isEmptyIndexSet(IndexSet indexSet) {
        return this.indices.getIndices(indexSet, new String[0]).stream().filter(this.indices::isOpen).noneMatch(indexName -> this.indices.numberOfMessages((String)indexName) > 0L);
    }

    private Pair<Period, Boolean> getNormalizedRotationPeriod(TimeBasedRotationStrategyConfig config) {
        Period rotationPeriod = config.rotationPeriod();
        Period maxPeriod = this.elasticsearchConfiguration.getMaxWriteIndexAge();
        boolean overriding = false;
        if (maxPeriod != null && TimeBasedRotationStrategy.isLonger(rotationPeriod, maxPeriod)) {
            log.debug("Max rotation limit {} overrides configured period {}", (Object)maxPeriod, (Object)rotationPeriod);
            rotationPeriod = maxPeriod;
            overriding = true;
        }
        Period normalizedPeriod = rotationPeriod.normalizedStandard();
        return new ImmutablePair((Object)normalizedPeriod, (Object)overriding);
    }

    private DateTime calculateNextAnchor(DateTime currentAnchor, Period normalizedPeriod, DateTime now) {
        DateTime tmpAnchor;
        int multiplicator = 0;
        while ((tmpAnchor = currentAnchor.withPeriodAdded((ReadablePeriod)normalizedPeriod, ++multiplicator)).isBefore((ReadableInstant)now)) {
        }
        return currentAnchor.withPeriodAdded((ReadablePeriod)normalizedPeriod, multiplicator - 1);
    }

    @Override
    public String getStrategyName() {
        return NAME;
    }
}

