/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.integrations.ipfix;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.primitives.Longs;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.apache.commons.codec.binary.Hex;
import org.graylog.integrations.ipfix.Flow;
import org.graylog.integrations.ipfix.InformationElement;
import org.graylog.integrations.ipfix.InformationElementDefinition;
import org.graylog.integrations.ipfix.InformationElementDefinitions;
import org.graylog.integrations.ipfix.InvalidMessageVersion;
import org.graylog.integrations.ipfix.IpfixException;
import org.graylog.integrations.ipfix.IpfixMessage;
import org.graylog.integrations.ipfix.MessageHeader;
import org.graylog.integrations.ipfix.OptionsTemplateRecord;
import org.graylog.integrations.ipfix.ShallowDataSet;
import org.graylog.integrations.ipfix.ShallowOptionsTemplateSet;
import org.graylog.integrations.ipfix.ShallowTemplateSet;
import org.graylog.integrations.ipfix.TemplateRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IpfixParser {
    private static final Logger LOG = LoggerFactory.getLogger(IpfixParser.class);
    private static final int SETID_RESERVED0 = 0;
    private static final int SETID_RESERVED1 = 1;
    private static final int SETID_TEMPLATE = 2;
    private static final int SETID_OPTIONSTEMPLATE = 3;
    private final InformationElementDefinitions infoElemDefs;

    public IpfixParser(InformationElementDefinitions informationElementDefinitions) {
        this.infoElemDefs = informationElementDefinitions;
    }

    public MessageDescription shallowParseMessage(ByteBuf packet) {
        ByteBuf buffer = packet.readSlice(16);
        LOG.debug("Shallow parse header\n{}", (Object)ByteBufUtil.prettyHexDump((ByteBuf)buffer));
        MessageHeader header = this.parseMessageHeader(buffer);
        MessageDescription messageDescription = new MessageDescription(header);
        if (header.length() != packet.readableBytes() + 16) {
            throw new IllegalArgumentException("Buffer does not contain the complete IPFIX message");
        }
        block5: while (packet.isReadable()) {
            int setId = packet.readUnsignedShort();
            int setLength = packet.readUnsignedShort();
            ByteBuf setContent = packet.readSlice(setLength - 4);
            switch (setId) {
                case 0: 
                case 1: {
                    throw new IpfixException("Invalid set id in IPFIX message: " + setId);
                }
                case 2: {
                    ShallowTemplateSet templateSet = this.shallowParseTemplateSet(setContent);
                    messageDescription.addTemplateSet(templateSet);
                    continue block5;
                }
                case 3: {
                    ShallowOptionsTemplateSet optionsTemplateSet = this.shallowParseOptionsTemplateSet(setContent);
                    messageDescription.addOptionsTemplateSet(optionsTemplateSet);
                    continue block5;
                }
            }
            ShallowDataSet dataSet = this.shallowParseDataSet(setId, setLength, setContent, header.exportTime());
            messageDescription.addDataSet(dataSet);
        }
        return messageDescription;
    }

    private ShallowTemplateSet shallowParseTemplateSet(ByteBuf setContent) {
        LOG.debug("Attempting a shallow parse on template set.");
        ImmutableList.Builder builder = ImmutableList.builder();
        while (setContent.isReadable()) {
            setContent.markReaderIndex();
            int lowerReaderIndex = setContent.readerIndex();
            int templateId = setContent.readUnsignedShort();
            int fieldCount = setContent.readUnsignedShort();
            for (int i = 0; i < fieldCount; ++i) {
                this.parseInformationElement(setContent);
            }
            int upperReaderIndex = setContent.readerIndex();
            byte[] recordBytes = new byte[upperReaderIndex - lowerReaderIndex];
            setContent.resetReaderIndex();
            setContent.readBytes(recordBytes);
            builder.add((Object)new ShallowTemplateSet.Record(templateId, recordBytes));
        }
        return ShallowTemplateSet.create((ImmutableList<ShallowTemplateSet.Record>)builder.build());
    }

    private ShallowOptionsTemplateSet shallowParseOptionsTemplateSet(ByteBuf setContent) {
        LOG.debug("Attempting a shallow parse on options template set.");
        ImmutableList.Builder builder = ImmutableList.builder();
        while (setContent.isReadable()) {
            setContent.markReaderIndex();
            int lowerReaderIndex = setContent.readerIndex();
            int templateId = setContent.readUnsignedShort();
            int fieldCount = setContent.readUnsignedShort();
            int scopeFieldCount = setContent.readUnsignedShort();
            for (int i = 0; i < fieldCount; ++i) {
                this.parseInformationElement(setContent);
            }
            int upperReaderIndex = setContent.readerIndex();
            byte[] recordBytes = new byte[upperReaderIndex - lowerReaderIndex];
            setContent.resetReaderIndex();
            setContent.readBytes(recordBytes);
            builder.add((Object)new ShallowOptionsTemplateSet.Record(templateId, recordBytes));
        }
        return ShallowOptionsTemplateSet.create((ImmutableList<ShallowOptionsTemplateSet.Record>)builder.build());
    }

    private ShallowDataSet shallowParseDataSet(int id, int length, ByteBuf setContent, ZonedDateTime exportTime) {
        LOG.debug("Attempting a shallow parse on dataset.");
        byte[] setBytes = new byte[length - 4];
        setContent.readBytes(setBytes);
        return ShallowDataSet.create(id, exportTime.toEpochSecond(), setBytes);
    }

    private InformationElement parseInformationElement(ByteBuf buffer) {
        int idAndEnterpriseBit;
        int id = idAndEnterpriseBit = buffer.readUnsignedShort();
        long enterpriseNumber = 0L;
        int length = buffer.readUnsignedShort();
        if (idAndEnterpriseBit > 32768) {
            id -= 32768;
            enterpriseNumber = buffer.readUnsignedInt();
        }
        return InformationElement.create(id, length, enterpriseNumber);
    }

    private MessageHeader parseMessageHeader(ByteBuf buffer) {
        LOG.debug("Attempting to parse message header.");
        int versionNumber = buffer.readUnsignedShort();
        if (versionNumber != 10) {
            throw new InvalidMessageVersion(versionNumber);
        }
        int packetLength = buffer.readUnsignedShort();
        long exportTime = buffer.readUnsignedInt();
        long sequenceNumber = buffer.readUnsignedInt();
        long observationDomainId = buffer.readUnsignedInt();
        return MessageHeader.create(packetLength, ZonedDateTime.ofInstant(Instant.ofEpochSecond(exportTime), ZoneOffset.UTC), sequenceNumber, observationDomainId);
    }

    public IpfixMessage parseMessage(ByteBuf packet) {
        LOG.debug("Attempting to parse message.");
        LOG.debug("IPFIX message\n{}", (Object)ByteBufUtil.prettyHexDump((ByteBuf)packet));
        IpfixMessage.Builder builder = IpfixMessage.builder();
        ByteBuf headerBuffer = packet.readSlice(16);
        LOG.debug("Message header buffer\n{}", (Object)ByteBufUtil.prettyHexDump((ByteBuf)headerBuffer));
        MessageHeader header = this.parseMessageHeader(headerBuffer);
        if (header.length() > packet.readableBytes() + 16) {
            LOG.error("Buffer does not contain expected IPFIX message:\n{}", (Object)ByteBufUtil.prettyHexDump((ByteBuf)packet));
            throw new IpfixException("Buffer does not contain the complete IPFIX message");
        }
        HashMap templates = Maps.newHashMap();
        HashMap optionsTemplates = Maps.newHashMap();
        block5: while (packet.isReadable()) {
            ImmutableList informationElements;
            int setId = packet.readUnsignedShort();
            int setLength = packet.readUnsignedShort();
            LOG.debug("Set id {} buffer\n{}", (Object)setId, (Object)ByteBufUtil.prettyHexDump((ByteBuf)packet, (int)(packet.readerIndex() - 4), (int)setLength));
            ByteBuf setContent = packet.readSlice(setLength - 4);
            switch (setId) {
                case 0: 
                case 1: {
                    throw new IpfixException("Invalid set id in IPFIX message: " + setId);
                }
                case 2: {
                    Set<TemplateRecord> templateRecords = this.parseTemplateSet(setContent);
                    builder.addAllTemplates(templateRecords);
                    templateRecords.forEach(r -> templates.put(r.templateId(), r));
                    continue block5;
                }
                case 3: {
                    Set<OptionsTemplateRecord> optionsTemplateRecords = this.parseOptionsTemplateSet(setContent);
                    builder.addAllOptionsTemplateSet(optionsTemplateRecords);
                    optionsTemplateRecords.forEach(r -> optionsTemplates.put(r.templateId(), r));
                    continue block5;
                }
            }
            if (templates.containsKey(setId)) {
                informationElements = ((TemplateRecord)templates.get(setId)).informationElements();
            } else if (optionsTemplates.containsKey(setId)) {
                OptionsTemplateRecord record = (OptionsTemplateRecord)optionsTemplates.get(setId);
                informationElements = ImmutableList.builder().addAll(record.scopeFields()).addAll(record.optionFields()).build();
            } else {
                throw new IpfixException("Missing template for data set using template id " + setId + ". Cannot parse data set.");
            }
            Set<Flow> flows = this.parseDataSet(informationElements, templates, setContent);
            builder.addAllFlows(flows);
        }
        return builder.build();
    }

    private Set<TemplateRecord> parseTemplateSet(ByteBuf setContent) {
        LOG.debug("Attempting to parse template set.");
        ImmutableSet.Builder b = ImmutableSet.builder();
        while (setContent.isReadable()) {
            b.add((Object)this.parseTemplateRecord(setContent));
        }
        return b.build();
    }

    private Set<OptionsTemplateRecord> parseOptionsTemplateSet(ByteBuf setContent) {
        ImmutableSet.Builder b = ImmutableSet.builder();
        while (setContent.isReadable()) {
            int i;
            OptionsTemplateRecord.Builder recordBuilder = OptionsTemplateRecord.builder();
            int templateId = setContent.readUnsignedShort();
            recordBuilder.templateId(templateId);
            int fieldCount = setContent.readUnsignedShort();
            int scopeFieldCount = setContent.readUnsignedShort();
            for (i = 0; i < scopeFieldCount; ++i) {
                recordBuilder.scopeFieldsBuilder().add((Object)this.parseInformationElement(setContent));
            }
            for (i = 0; i < fieldCount - scopeFieldCount; ++i) {
                recordBuilder.optionFieldsBuilder().add((Object)this.parseInformationElement(setContent));
            }
            b.add((Object)recordBuilder.build());
        }
        return b.build();
    }

    public Set<Flow> parseDataSet(ImmutableList<InformationElement> informationElements, Map<Integer, TemplateRecord> templateMap, ByteBuf setContent) {
        ImmutableSet.Builder flowBuilder = ImmutableSet.builder();
        int flowNum = 1;
        while (setContent.isReadable()) {
            try {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Parsing flow {}", (Object)flowNum);
                }
                ++flowNum;
                ImmutableMap.Builder fields = ImmutableMap.builder();
                for (InformationElement informationElement : informationElements) {
                    int firstByte = setContent.readerIndex();
                    InformationElementDefinition desc = this.infoElemDefs.getDefinition(informationElement.id(), informationElement.enterpriseNumber());
                    block3 : switch (desc.dataType()) {
                        case UNSIGNED8: 
                        case UNSIGNED16: 
                        case UNSIGNED32: 
                        case UNSIGNED64: {
                            fields.put((Object)desc.fieldName(), (Object)(switch (informationElement.length()) {
                                case 1 -> setContent.readUnsignedByte();
                                case 2 -> setContent.readUnsignedShort();
                                case 3 -> setContent.readUnsignedMedium();
                                case 4 -> setContent.readUnsignedInt();
                                case 5, 6, 7, 8 -> {
                                    byte[] bytesBigEndian = new byte[]{0, 0, 0, 0, 0, 0, 0, 0};
                                    int firstIndex = 8 - informationElement.length();
                                    setContent.readBytes(bytesBigEndian, firstIndex, informationElement.length());
                                    yield Longs.fromByteArray((byte[])bytesBigEndian);
                                }
                                default -> throw new IpfixException("Unexpected length for unsigned integer");
                            }));
                            break;
                        }
                        case SIGNED8: 
                        case SIGNED16: 
                        case SIGNED32: 
                        case SIGNED64: {
                            fields.put((Object)desc.fieldName(), (Object)(switch (informationElement.length()) {
                                case 1 -> setContent.readByte();
                                case 2 -> setContent.readShort();
                                case 3 -> setContent.readMedium();
                                case 4 -> setContent.readUnsignedInt();
                                case 5, 6, 7, 8 -> {
                                    byte[] bytesBigEndian = new byte[]{0, 0, 0, 0, 0, 0, 0, 0};
                                    int firstIndex = 8 - informationElement.length() - 1;
                                    setContent.readBytes(bytesBigEndian, firstIndex, informationElement.length());
                                    yield Longs.fromByteArray((byte[])bytesBigEndian);
                                }
                                default -> throw new IpfixException("Unexpected length for unsigned integer");
                            }));
                            break;
                        }
                        case FLOAT32: 
                        case FLOAT64: {
                            fields.put((Object)desc.fieldName(), (Object)(switch (informationElement.length()) {
                                case 4 -> setContent.readFloat();
                                case 8 -> setContent.readDouble();
                                default -> throw new IpfixException("Unexpected length for float value: " + informationElement.length());
                            }));
                            break;
                        }
                        case MACADDRESS: {
                            byte[] macBytes = new byte[6];
                            setContent.readBytes(macBytes);
                            fields.put((Object)desc.fieldName(), (Object)String.format(Locale.ROOT, "%02x:%02x:%02x:%02x:%02x:%02x", macBytes[0], macBytes[1], macBytes[2], macBytes[3], macBytes[4], macBytes[5]));
                            break;
                        }
                        case IPV4ADDRESS: {
                            byte[] ipv4Bytes = new byte[4];
                            setContent.readBytes(ipv4Bytes);
                            try {
                                fields.put((Object)desc.fieldName(), (Object)InetAddress.getByAddress(ipv4Bytes).getHostAddress());
                                break;
                            }
                            catch (UnknownHostException e) {
                                throw new IpfixException("Unable to parse IPV4 address", e);
                            }
                        }
                        case IPV6ADDRESS: {
                            byte[] ipv6Bytes = new byte[16];
                            setContent.readBytes(ipv6Bytes);
                            try {
                                fields.put((Object)desc.fieldName(), (Object)InetAddress.getByAddress(ipv6Bytes).getHostAddress());
                                break;
                            }
                            catch (UnknownHostException e) {
                                throw new IpfixException("Unable to parse IPV6 address", e);
                            }
                        }
                        case BOOLEAN: {
                            byte booleanByte = setContent.readByte();
                            switch (booleanByte) {
                                case 1: {
                                    fields.put((Object)desc.fieldName(), (Object)true);
                                    break block3;
                                }
                                case 2: {
                                    fields.put((Object)desc.fieldName(), (Object)false);
                                    break block3;
                                }
                            }
                            throw new IpfixException("Invalid value for boolean: " + booleanByte);
                        }
                        case STRING: {
                            CharSequence charSequence;
                            if (informationElement.length() == 65535) {
                                int length = this.getVarLength(setContent);
                                charSequence = setContent.readCharSequence(length, StandardCharsets.UTF_8);
                            } else {
                                charSequence = setContent.readCharSequence(informationElement.length(), StandardCharsets.UTF_8);
                            }
                            fields.put((Object)desc.fieldName(), (Object)String.valueOf(charSequence).replace("\u0000", ""));
                            break;
                        }
                        case OCTETARRAY: {
                            byte[] octetArray;
                            if (informationElement.length() == 65535) {
                                int length = this.getVarLength(setContent);
                                octetArray = new byte[length];
                            } else {
                                octetArray = new byte[informationElement.length()];
                            }
                            setContent.readBytes(octetArray);
                            fields.put((Object)desc.fieldName(), (Object)Hex.encodeHexString((byte[])octetArray));
                            break;
                        }
                        case DATETIMESECONDS: {
                            long dateTimeSeconds = setContent.readUnsignedInt();
                            fields.put((Object)desc.fieldName(), (Object)ZonedDateTime.ofInstant(Instant.ofEpochSecond(dateTimeSeconds), ZoneOffset.UTC));
                            break;
                        }
                        case DATETIMEMILLISECONDS: {
                            long dateTimeMills = setContent.readLong();
                            fields.put((Object)desc.fieldName(), (Object)ZonedDateTime.ofInstant(Instant.ofEpochMilli(dateTimeMills), ZoneOffset.UTC));
                            break;
                        }
                        case DATETIMEMICROSECONDS: 
                        case DATETIMENANOSECONDS: {
                            long seconds = setContent.readUnsignedInt();
                            long fraction = setContent.readUnsignedInt();
                            if (desc.dataType() == InformationElementDefinition.DataType.DATETIMEMICROSECONDS) {
                                fraction &= 0xFFFFFFFFFFFFF800L;
                            }
                            fields.put((Object)desc.fieldName(), (Object)ZonedDateTime.ofInstant(Instant.ofEpochSecond(seconds, fraction), ZoneOffset.UTC));
                            break;
                        }
                        case BASICLIST: {
                            int length = informationElement.length() == 65535 ? this.getVarLength(setContent) : setContent.readUnsignedByte();
                            ByteBuf listBuffer = setContent.readSlice(length);
                            short semantic = listBuffer.readUnsignedByte();
                            InformationElement element = this.parseInformationElement(listBuffer);
                            InformationElementDefinition def = this.infoElemDefs.getDefinition(element.id(), element.enterpriseNumber());
                            if (def == null) {
                                LOG.error("Unable to find information element definition in basicList: id {} PEN {}, this is a bug, cannot parse packet.", (Object)element.id(), (Object)element.enterpriseNumber());
                                break;
                            }
                            LOG.warn("Skipping basicList data ({} bytes)", (Object)informationElement.length());
                            while (listBuffer.isReadable()) {
                                listBuffer.skipBytes(element.length());
                            }
                            break;
                        }
                        case SUBTEMPLATELIST: {
                            int length = informationElement.length() == 65535 ? this.getVarLength(setContent) : setContent.readUnsignedByte();
                            length -= 3;
                            LOG.debug("Remaining data buffer:\n{}", (Object)ByteBufUtil.prettyHexDump((ByteBuf)setContent));
                            short semantic = setContent.readUnsignedByte();
                            int templateId = setContent.readUnsignedShort();
                            TemplateRecord templateRecord = templateMap.get(templateId);
                            if (templateRecord == null) {
                                LOG.error("Unable to parse subtemplateList, because we don't have the template for it: {}, skipping data ({} bytes)", (Object)templateId, (Object)length);
                                setContent.skipBytes(length);
                                break;
                            }
                            ByteBuf listContent = setContent.readSlice(length);
                            ImmutableList.Builder flowsBuilder = ImmutableList.builder();
                            if (listContent.isReadable()) {
                                flowsBuilder.addAll(this.parseDataSet(templateRecord.informationElements(), templateMap, listContent));
                            }
                            ImmutableList flows = flowsBuilder.build();
                            for (int i = 0; i < flows.size(); ++i) {
                                String fieldPrefix = desc.fieldName() + "_" + i + "_";
                                ((Flow)flows.get(i)).fields().forEach((field, value) -> fields.put((Object)(fieldPrefix + field), value));
                            }
                            break;
                        }
                        case SUBTEMPLATEMULTILIST: {
                            int length = informationElement.length() == 65535 ? this.getVarLength(setContent) : setContent.readUnsignedByte();
                            setContent.skipBytes(length);
                            LOG.warn("subtemplateMultilist support is not implemented, skipping data ({} bytes)", (Object)length);
                            break;
                        }
                    }
                    if (!LOG.isTraceEnabled()) continue;
                    LOG.trace("Field {}:\n{}", (Object)informationElement.id(), (Object)ByteBufUtil.prettyHexDump((ByteBuf)setContent, (int)firstByte, (int)(setContent.readerIndex() - firstByte)));
                }
                flowBuilder.add((Object)Flow.create((ImmutableMap<String, Object>)fields.build()));
            }
            catch (IndexOutOfBoundsException ioobe) {
                if (setContent.forEachByte(value -> value == 0) == -1) break;
                LOG.error("Verify data record padding failed, trailing bytes were not all 0x00");
                throw ioobe;
            }
        }
        return flowBuilder.build();
    }

    private int getVarLength(ByteBuf setContent) {
        int firstLengthByte = setContent.readUnsignedByte();
        int length = firstLengthByte == 255 ? setContent.readUnsignedShort() : firstLengthByte;
        return length;
    }

    public TemplateRecord parseTemplateRecord(ByteBuf bytes) {
        TemplateRecord.Builder recordBuilder = TemplateRecord.builder();
        int templateId = bytes.readUnsignedShort();
        recordBuilder.templateId(templateId);
        int fieldCount = bytes.readUnsignedShort();
        for (int i = 0; i < fieldCount; ++i) {
            recordBuilder.addInformationElement(this.parseInformationElement(bytes));
        }
        return recordBuilder.build();
    }

    public class MessageDescription {
        private final Map<Integer, ShallowTemplateSet.Record> templates = Maps.newHashMap();
        private final Map<Integer, ShallowOptionsTemplateSet.Record> optionsTemplates = Maps.newHashMap();
        private final Multimap<Integer, ShallowDataSet> dataSets = ArrayListMultimap.create();
        private final MessageHeader header;

        public MessageDescription(MessageHeader header) {
            this.header = header;
        }

        public MessageHeader getHeader() {
            return this.header;
        }

        public void addTemplateSet(ShallowTemplateSet templateSet) {
            for (ShallowTemplateSet.Record record : templateSet.records()) {
                this.templates.put(record.getTemplateId(), record);
            }
        }

        public ShallowTemplateSet.Record getTemplateRecord(int templateId) {
            return this.templates.get(templateId);
        }

        public void addOptionsTemplateSet(ShallowOptionsTemplateSet optionsTemplateSet) {
            for (ShallowOptionsTemplateSet.Record record : optionsTemplateSet.records()) {
                this.optionsTemplates.put(record.getTemplateId(), record);
            }
        }

        public void addDataSet(ShallowDataSet dataSet) {
            this.dataSets.put((Object)dataSet.templateId(), (Object)dataSet);
        }

        public Set<Integer> referencedTemplateIds() {
            return ImmutableSet.copyOf((Collection)this.dataSets.keySet());
        }

        public Set<ShallowDataSet> dataSets() {
            return ImmutableSet.copyOf((Collection)this.dataSets.values());
        }

        public Set<Integer> declaredTemplateIds() {
            return ImmutableSet.copyOf(this.templates.keySet());
        }

        public Set<ShallowTemplateSet.Record> templateRecords() {
            return ImmutableSet.copyOf(this.templates.values());
        }

        public Set<Integer> declaredOptionsTemplateIds() {
            return ImmutableSet.copyOf(this.optionsTemplates.keySet());
        }

        public Set<ShallowOptionsTemplateSet.Record> optionsTemplateRecords() {
            return ImmutableSet.copyOf(this.optionsTemplates.values());
        }
    }
}

