/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.cluster.certificates;

import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Indexes;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import jakarta.annotation.Nonnull;
import jakarta.inject.Inject;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.LinkedList;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.graylog.security.certutil.cert.CertificateChain;
import org.graylog2.cluster.certificates.CertificateExchange;
import org.graylog2.cluster.certificates.CertificateExchangeType;
import org.graylog2.cluster.certificates.CertificateSigningRequest;
import org.graylog2.database.MongoConnection;
import org.graylog2.security.encryption.EncryptedValue;
import org.graylog2.security.encryption.EncryptedValueService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateExchangeImpl
implements CertificateExchange {
    private static final Logger LOG = LoggerFactory.getLogger(CertificateExchangeImpl.class);
    private static final String ENCRYPTED_VALUE_SUBFIELD = "encrypted_value";
    private static final String SALT_SUBFIELD = "salt";
    private static final String COLLECTION_NAME = "certificate_exchange";
    public static final String FIELD_NODE_ID = "node_id";
    public static final String FIELD_ENTRY_TYPE = "type";
    public static final String FIELD_ENCRYPTED_VALUE = "value";
    private final MongoDatabase mongoDatabase;
    private final EncryptedValueService encryptionService;

    @Inject
    public CertificateExchangeImpl(MongoConnection mongoConnection, EncryptedValueService encryptionService) {
        this.mongoDatabase = mongoConnection.getMongoDatabase();
        this.encryptionService = encryptionService;
        CertificateExchangeImpl.createIndex(this.mongoDatabase);
    }

    private static void createIndex(MongoDatabase mongoDatabase) {
        MongoCollection dbCollection = mongoDatabase.getCollection(COLLECTION_NAME);
        dbCollection.createIndex(Indexes.ascending((String[])new String[]{FIELD_NODE_ID, FIELD_ENTRY_TYPE}), new IndexOptions().unique(true));
    }

    @Override
    public void requestCertificate(CertificateSigningRequest request) throws IOException {
        this.writeToDatabase(request.nodeId(), CertificateExchangeImpl.serializeCsr(request.request()), CertificateExchangeType.CSR);
    }

    @Override
    public void signPendingCertificateRequests(Function<CertificateSigningRequest, CertificateChain> signingFunction) throws IOException {
        MongoCollection dbCollection = this.mongoDatabase.getCollection(COLLECTION_NAME);
        FindIterable objects = dbCollection.find(Filters.eq((String)FIELD_ENTRY_TYPE, (Object)((Object)CertificateExchangeType.CSR)));
        try (MongoCursor cursor = objects.cursor();){
            while (cursor.hasNext()) {
                try {
                    CertificateSigningRequest csr = this.documentToCsr((Document)cursor.next());
                    this.writeCertificate(csr.nodeId(), signingFunction.apply(csr));
                    this.removeCertificateSigningRequest(csr.nodeId());
                }
                catch (Exception e) {
                    LOG.error("Failed to sign CSR for node, skipping it for now.", (Throwable)e);
                }
            }
        }
    }

    @Override
    public void pollCertificate(String nodeId, Consumer<CertificateChain> chainConsumer) {
        this.readCertChain(nodeId).ifPresent(certificateChain -> {
            chainConsumer.accept((CertificateChain)certificateChain);
            this.removeCertificateChain(nodeId);
        });
    }

    private void writeCertificate(String nodeId, CertificateChain certificateChain) throws IOException {
        this.writeToDatabase(nodeId, CertificateExchangeImpl.serializeChain(certificateChain), CertificateExchangeType.CERT_CHAIN);
    }

    private void writeToDatabase(String nodeId, String serializedValue, CertificateExchangeType type) {
        boolean updated;
        MongoCollection dbCollection = this.mongoDatabase.getCollection(COLLECTION_NAME);
        EncryptedValue encrypted = this.encryptionService.encrypt(serializedValue);
        UpdateResult result = dbCollection.updateOne(Filters.and((Bson[])new Bson[]{Filters.eq((String)FIELD_NODE_ID, (Object)nodeId), Filters.eq((String)FIELD_ENTRY_TYPE, (Object)((Object)type))}), Updates.combine((Bson[])new Bson[]{Updates.set((String)FIELD_NODE_ID, (Object)nodeId), Updates.set((String)"value.encrypted_value", (Object)encrypted.value()), Updates.set((String)"value.salt", (Object)encrypted.salt())}), new UpdateOptions().upsert(true));
        boolean bl = updated = result.getModifiedCount() > 0L || result.getUpsertedId() != null;
        if (!updated) {
            throw new RuntimeException("Failed to write entry to certificate exchange collection!");
        }
    }

    private CertificateSigningRequest documentToCsr(Document document) {
        String nodeID = document.getString((Object)FIELD_NODE_ID);
        String decryptedValue = this.decryptedValue(document);
        return new CertificateSigningRequest(nodeID, this.parseCSR(decryptedValue));
    }

    @Nonnull
    private Optional<CertificateChain> readCertChain(String nodeId) {
        return this.findCertChain(nodeId).map(document -> {
            String decryptedValue = this.decryptedValue((Document)document);
            return this.parseCertificateChain(decryptedValue);
        });
    }

    @Nonnull
    private Optional<Document> findCertChain(String nodeId) {
        MongoCollection dbCollection = this.mongoDatabase.getCollection(COLLECTION_NAME);
        Bson filter = Filters.and((Bson[])new Bson[]{Filters.eq((String)FIELD_NODE_ID, (Object)nodeId), Filters.eq((String)FIELD_ENTRY_TYPE, (Object)((Object)CertificateExchangeType.CERT_CHAIN))});
        FindIterable objects = dbCollection.find(filter).limit(1);
        return Optional.ofNullable((Document)objects.first());
    }

    private String decryptedValue(Document document) {
        return Optional.ofNullable((Document)document.get((Object)FIELD_ENCRYPTED_VALUE, Document.class)).map(encryptedDocument -> EncryptedValue.builder().value(encryptedDocument.getString((Object)ENCRYPTED_VALUE_SUBFIELD)).salt(encryptedDocument.getString((Object)SALT_SUBFIELD)).isDeleteValue(false).isKeepValue(false).build()).map(this.encryptionService::decrypt).orElseThrow(() -> new IllegalStateException("This document should contain encrypted value! " + document.toJson()));
    }

    private void removeCertificateChain(String nodeId) {
        this.removeEntry(nodeId, CertificateExchangeType.CERT_CHAIN);
    }

    private void removeCertificateSigningRequest(String nodeId) {
        this.removeEntry(nodeId, CertificateExchangeType.CSR);
    }

    private void removeEntry(String nodeId, CertificateExchangeType type) {
        MongoCollection dbCollection = this.mongoDatabase.getCollection(COLLECTION_NAME);
        Bson filter = Filters.and((Bson[])new Bson[]{Filters.eq((String)FIELD_NODE_ID, (Object)nodeId), Filters.eq((String)FIELD_ENTRY_TYPE, (Object)((Object)type))});
        DeleteResult result = dbCollection.deleteOne(filter);
        if (result.getDeletedCount() != 1L) {
            throw new IllegalStateException("removeEntry hasn't deleted any entry, should delete one!");
        }
    }

    private static String serializeCsr(PKCS10CertificationRequest csr) throws IOException {
        StringWriter writer = new StringWriter();
        try (JcaPEMWriter jcaPEMWriter = new JcaPEMWriter((Writer)writer);){
            jcaPEMWriter.writeObject((Object)csr);
        }
        return writer.toString();
    }

    private static String serializeChain(CertificateChain certChain) throws IOException {
        StringWriter writer = new StringWriter();
        try (JcaPEMWriter jcaPEMWriter = new JcaPEMWriter((Writer)writer);){
            for (Certificate c : certChain.toCertificateChainArray()) {
                jcaPEMWriter.writeObject((Object)c);
            }
        }
        return writer.toString();
    }

    public CertificateChain parseCertificateChain(String certificate) {
        try {
            BufferedReader pemReader = new BufferedReader(new StringReader(certificate));
            PEMParser pemParser = new PEMParser((Reader)pemReader);
            LinkedList<X509Certificate> caCerts = new LinkedList<X509Certificate>();
            X509Certificate signedCert = this.readSingleCert(pemParser);
            X509Certificate caCert = this.readSingleCert(pemParser);
            while (caCert != null) {
                caCerts.add(caCert);
                caCert = this.readSingleCert(pemParser);
            }
            return new CertificateChain(signedCert, caCerts);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private X509Certificate readSingleCert(PEMParser pemParser) throws IOException, CertificateException {
        Object parsedObj = pemParser.readObject();
        if (parsedObj == null) {
            return null;
        }
        if (parsedObj instanceof X509Certificate) {
            return (X509Certificate)parsedObj;
        }
        if (parsedObj instanceof X509CertificateHolder) {
            return new JcaX509CertificateConverter().getCertificate((X509CertificateHolder)parsedObj);
        }
        throw new IllegalArgumentException("Cannot read certificate from PEMParser. Containing object is of unexpected type " + String.valueOf(parsedObj.getClass()));
    }

    private PKCS10CertificationRequest parseCSR(String csr) {
        try {
            BufferedReader pemReader = new BufferedReader(new StringReader(csr));
            PEMParser pemParser = new PEMParser((Reader)pemReader);
            Object parsedObj = pemParser.readObject();
            return (PKCS10CertificationRequest)parsedObj;
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}

