/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer.messages;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.function.Supplier;
import org.graylog2.indexer.ElasticsearchException;
import org.graylog2.indexer.messages.DynamicSizeListPartitioner;
import org.graylog2.indexer.messages.IndexingError;
import org.graylog2.indexer.messages.IndexingRequest;
import org.graylog2.indexer.messages.IndexingResults;
import org.graylog2.indexer.messages.RetryWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkedBulkIndexer {
    private static final Logger LOG = LoggerFactory.getLogger(ChunkedBulkIndexer.class);
    private static final RetryWait retryWait = new RetryWait(100);

    public IndexingResults index(List<IndexingRequest> messageList, BulkIndex bulkIndex) throws IOException {
        if (messageList.isEmpty()) {
            return IndexingResults.empty();
        }
        int chunkSize = messageList.size();
        int offset = 0;
        IndexingResults.Builder accumulatedResults = IndexingResults.Builder.create();
        int attempt = 0;
        boolean allowResettingChunkSize = false;
        while (true) {
            try {
                IndexingResults results = this.bulkIndexChunked(new Chunk(messageList, offset, chunkSize), allowResettingChunkSize, bulkIndex);
                accumulatedResults.addResults(results);
                return accumulatedResults.build();
            }
            catch (EntityTooLargeException e) {
                CircuitBreakerException cbe;
                boolean retryForever = e instanceof CircuitBreakerException && (cbe = (CircuitBreakerException)e).isTransient();
                LOG.warn("Bulk index failed with '{}' error. Retrying by splitting up batch size <{}>.", (Object)e.description(), (Object)chunkSize);
                if (chunkSize == messageList.size()) {
                    LOG.warn("Consider lowering the \"output_batch_size\" setting. Or resizing your Search cluster");
                }
                offset += e.indexedSuccessfully;
                chunkSize = Math.max(chunkSize / 2, retryForever ? 1 : 0);
                accumulatedResults.addResults(e.previousResults);
                if (!retryForever || chunkSize != 1) continue;
                allowResettingChunkSize = true;
                retryWait.waitBeforeRetrying(attempt++);
                if (chunkSize != 0) continue;
                throw new ElasticsearchException("Bulk index cannot split output batch any further.");
            }
            break;
        }
    }

    private IndexingResults bulkIndexChunked(Chunk command, boolean allowResettingChunkSize, BulkIndex bulkIndex) throws EntityTooLargeException, IOException {
        List<IndexingRequest> messageList = command.requests;
        int offset = command.offset;
        int chunkSize = command.size;
        IndexingResults.Builder accumulatedResults = IndexingResults.Builder.create();
        if (messageList.isEmpty()) {
            return accumulatedResults.build();
        }
        List<IndexingRequest> remainingMessages = messageList.subList(offset, messageList.size());
        DynamicSizeListPartitioner<IndexingRequest> partitioner = new DynamicSizeListPartitioner<IndexingRequest>(remainingMessages);
        int chunkCount = 1;
        int indexedSuccessfully = 0;
        while (partitioner.hasNext()) {
            List<IndexingRequest> chunk = partitioner.nextPartition(chunkSize);
            BulkIndexResult response = bulkIndex.apply(indexedSuccessfully, accumulatedResults.build(), chunk);
            indexedSuccessfully += chunk.size();
            IndexingResults results = response.indexingResults();
            accumulatedResults.addResults(results);
            if (allowResettingChunkSize && chunkSize != command.requests().size()) {
                LOG.warn("Indexing successful again - resetting chunk size!");
                chunkSize = command.requests().size();
            }
            this.logDebugInfo(messageList, offset, chunkSize, chunkCount, response.indexedMessages(), (List<IndexingError>)results.errors());
            this.logFailures(response.failureMessage(), results.errors().size());
            ++chunkCount;
        }
        return accumulatedResults.build();
    }

    private void logFailures(Supplier<String> failureMessage, int failureCount) {
        if (failureCount > 0) {
            LOG.error("Failed to index [{}] messages. Please check the index error log in your web interface for the reason. Error: {}", (Object)failureCount, (Object)failureMessage.get());
        }
    }

    private void logDebugInfo(List<IndexingRequest> messageList, int offset, int chunkSize, int chunkCount, int indexedMessages, List<IndexingError> failures) {
        if (LOG.isDebugEnabled()) {
            String chunkInfo = "";
            if (chunkSize != messageList.size()) {
                chunkInfo = String.format(Locale.ROOT, " (chunk %d/%d offset %d)", chunkCount, (int)Math.ceil((double)messageList.size() / (double)chunkSize), offset);
            }
            LOG.debug("Index: Bulk indexed {} messages{}, failures: {}", new Object[]{indexedMessages, chunkInfo, failures.size()});
        }
    }

    public record Chunk(List<IndexingRequest> requests, int offset, int size) {
    }

    public static interface BulkIndex {
        public BulkIndexResult apply(int var1, IndexingResults var2, List<IndexingRequest> var3) throws EntityTooLargeException, IOException;
    }

    public static class EntityTooLargeException
    extends Exception {
        public final int indexedSuccessfully;
        public final IndexingResults previousResults;

        String description() {
            return "Request Entity Too Large";
        }

        public EntityTooLargeException(int indexedSuccessfully, IndexingResults previousResults) {
            this.indexedSuccessfully = indexedSuccessfully;
            this.previousResults = previousResults;
        }
    }

    public static class CircuitBreakerException
    extends EntityTooLargeException {
        private final Durability durability;

        private boolean isTransient() {
            return this.durability == Durability.Transient;
        }

        @Override
        String description() {
            return "Data too large";
        }

        public CircuitBreakerException(int indexedSuccessfully, IndexingResults previousResults, Durability durability) {
            super(indexedSuccessfully, previousResults);
            this.durability = durability;
        }

        public static enum Durability {
            Transient,
            Permanent;

        }
    }

    public record BulkIndexResult(IndexingResults indexingResults, Supplier<String> failureMessage, int indexedMessages) {
    }

    public static class TooManyRequestsException
    extends EntityTooLargeException {
        @Override
        String description() {
            return "Too many requests";
        }

        public TooManyRequestsException(int indexedSuccessfully, IndexingResults previousResults) {
            super(indexedSuccessfully, previousResults);
        }
    }
}

