/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.plugin.inputs.transports.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Iterators;
import com.google.common.io.BaseEncoding;
import com.google.common.io.ByteStreams;
import io.netty.handler.ssl.PemPrivateKey;
import io.netty.handler.ssl.PemX509Certificate;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.PKCS8Generator;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.openssl.jcajce.JcaPKCS8Generator;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8DecryptorProviderBuilder;
import org.bouncycastle.openssl.jcajce.JceOpenSSLPKCS8EncryptorBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.OutputEncryptor;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyUtil {
    private static final Logger LOG = LoggerFactory.getLogger(KeyUtil.class);
    private static final Joiner JOINER = Joiner.on((String)",").skipNulls();
    private static final Pattern KEY_PATTERN = Pattern.compile("-{5}BEGIN (?:(RSA|DSA|EC)? )?(ENCRYPTED )?PRIVATE KEY-{5}\\r?\\n([A-Z0-9a-z+/\\r\\n]+={0,2})\\r?\\n-{5}END (?:(?:RSA|DSA|EC)? )?(?:ENCRYPTED )?PRIVATE KEY-{5}\\r?\\n$", 8);

    public static X509Certificate[] loadX509Certificates(Path certificatePath) throws CertificateException, IOException {
        return (X509Certificate[])KeyUtil.loadCertificates(certificatePath).stream().filter(certificate -> certificate instanceof X509Certificate).map(certificate -> (X509Certificate)certificate).toArray(X509Certificate[]::new);
    }

    public static Collection<? extends Certificate> loadCertificates(Path certificatePath) throws CertificateException, IOException {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        File certFile = certificatePath.toFile();
        if (certFile.isDirectory()) {
            ByteArrayOutputStream certStream = new ByteArrayOutputStream();
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(certFile.toPath());){
                for (Path f : ds) {
                    certStream.write(Files.readAllBytes(f));
                }
            }
            return cf.generateCertificates(new ByteArrayInputStream(certStream.toByteArray()));
        }
        try (InputStream inputStream = Files.newInputStream(certificatePath, new OpenOption[0]);){
            Collection<? extends Certificate> collection = cf.generateCertificates(inputStream);
            return collection;
        }
    }

    public static KeyManager[] initKeyStore(File tlsKeyFile, File tlsCertFile, String tlsKeyPassword) throws IOException, GeneralSecurityException {
        KeyStore ks = KeyStore.getInstance("JKS");
        ks.load(null, null);
        Collection<? extends Certificate> certChain = KeyUtil.loadCertificates(tlsCertFile.toPath());
        PrivateKey privateKey = KeyUtil.loadPrivateKey(tlsKeyFile, tlsKeyPassword);
        char[] password = Strings.nullToEmpty((String)tlsKeyPassword).toCharArray();
        ks.setKeyEntry("key", privateKey, password, certChain.toArray(new Certificate[certChain.size()]));
        if (LOG.isDebugEnabled()) {
            LOG.debug("Private key file: {}", (Object)tlsKeyFile);
            LOG.debug("Certificate file: {}", (Object)tlsCertFile);
            LOG.debug("Aliases: {}", (Object)KeyUtil.join(ks.aliases()));
        }
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
        kmf.init(ks, password);
        return kmf.getKeyManagers();
    }

    private static String join(Enumeration<String> aliases) {
        return JOINER.join((Iterator)Iterators.forEnumeration(aliases));
    }

    @VisibleForTesting
    protected static PrivateKey loadPrivateKey(File file, String password) throws IOException, GeneralSecurityException {
        try (InputStream is = Files.newInputStream(file.toPath(), new OpenOption[0]);){
            String[] keyAlgorithms;
            PKCS8EncodedKeySpec keySpec;
            byte[] keyBytes = ByteStreams.toByteArray((InputStream)is);
            String keyString = new String(keyBytes, StandardCharsets.US_ASCII);
            Matcher m = KEY_PATTERN.matcher(keyString);
            byte[] encoded = keyBytes;
            if (m.matches()) {
                if (!Strings.isNullOrEmpty((String)m.group(1))) {
                    throw new IllegalArgumentException("Unsupported key type PKCS#1, please convert to PKCS#8");
                }
                encoded = BaseEncoding.base64().decode((CharSequence)m.group(3).replaceAll("[\\r\\n]", ""));
            }
            if ((keySpec = KeyUtil.createKeySpec(encoded, password)) == null) {
                throw new IllegalArgumentException("Unsupported key type: " + String.valueOf(file));
            }
            for (String keyAlgorithm : keyAlgorithms = new String[]{"RSA", "DSA", "EC"}) {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance(keyAlgorithm);
                    PrivateKey privateKey = keyFactory.generatePrivate(keySpec);
                    return privateKey;
                }
                catch (InvalidKeySpecException e) {
                    LOG.debug("Loading {} private key from \"{}\" failed", new Object[]{keyAlgorithm, file, e});
                }
            }
            throw new IllegalArgumentException("Unsupported key type: " + String.valueOf(file));
        }
    }

    private static PKCS8EncodedKeySpec createKeySpec(byte[] keyBytes, String password) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException {
        if (Strings.isNullOrEmpty((String)password)) {
            return new PKCS8EncodedKeySpec(keyBytes);
        }
        EncryptedPrivateKeyInfo pkInfo = new EncryptedPrivateKeyInfo(keyBytes);
        SecretKeyFactory kf = SecretKeyFactory.getInstance(pkInfo.getAlgName());
        PBEKeySpec keySpec = new PBEKeySpec(password.toCharArray());
        SecretKey secretKey = kf.generateSecret(keySpec);
        Cipher cipher = Cipher.getInstance(pkInfo.getAlgName());
        cipher.init(2, (Key)secretKey, pkInfo.getAlgParameters());
        return pkInfo.getKeySpec(cipher);
    }

    public static X509Certificate readCertificate(Path path) throws IOException {
        byte[] bytes = Files.readAllBytes(path);
        return PemX509Certificate.valueOf((byte[])bytes);
    }

    public static PrivateKey readPrivateKey(Path path) throws IOException {
        byte[] bytes = Files.readAllBytes(path);
        return PemPrivateKey.valueOf((byte[])bytes);
    }

    public static File generatePKCS8FromPrivateKey(Path tmpDir, char[] password, PrivateKey key) throws GeneralSecurityException {
        try {
            JceOpenSSLPKCS8EncryptorBuilder encryptorBuilder = new JceOpenSSLPKCS8EncryptorBuilder(PKCS8Generator.AES_256_CBC).setRandom(new SecureRandom()).setPasssword(password);
            OutputEncryptor encryptor = encryptorBuilder.build();
            PemObject pemObj = new JcaPKCS8Generator(key, encryptor).generate();
            StringWriter stringWriter = new StringWriter();
            try (JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)stringWriter);){
                pemWriter.writeObject((PemObjectGenerator)pemObj);
            }
            String pkcs8Key = stringWriter.toString();
            File tmpFile = Files.createTempFile(tmpDir, "pkcs8", ".key", new FileAttribute[0]).toFile();
            try (FileOutputStream fos = new FileOutputStream(tmpFile);){
                fos.write(pkcs8Key.getBytes(StandardCharsets.UTF_8));
                tmpFile.deleteOnExit();
            }
            return tmpFile;
        }
        catch (IOException | OperatorCreationException e) {
            throw new GeneralSecurityException(e);
        }
    }

    public static PrivateKey privateKeyFromFile(String password, File keyFile) throws IOException, PKCSException, OperatorCreationException {
        PrivateKey privateKey;
        Object object;
        JcaPEMKeyConverter converter = new JcaPEMKeyConverter().setProvider("BC");
        try (InputStream inputStream = Files.newInputStream(keyFile.toPath(), new OpenOption[0]);
             InputStreamReader fileReader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
             PEMParser pemParser = new PEMParser((Reader)fileReader);){
            object = pemParser.readObject();
        }
        if (object instanceof PKCS8EncryptedPrivateKeyInfo) {
            PKCS8EncryptedPrivateKeyInfo pInfo = (PKCS8EncryptedPrivateKeyInfo)object;
            JceOpenSSLPKCS8DecryptorProviderBuilder providerBuilder = new JceOpenSSLPKCS8DecryptorProviderBuilder();
            InputDecryptorProvider provider = providerBuilder.build(Strings.nullToEmpty((String)password).toCharArray());
            PrivateKeyInfo info = pInfo.decryptPrivateKeyInfo(provider);
            privateKey = converter.getPrivateKey(info);
        } else if (object instanceof PrivateKeyInfo) {
            privateKey = converter.getPrivateKey((PrivateKeyInfo)object);
        } else if (object instanceof PEMKeyPair) {
            privateKey = converter.getPrivateKey(((PEMKeyPair)object).getPrivateKeyInfo());
        } else {
            throw new PKCSException("Encountered unexpected object type: " + object.getClass().getName());
        }
        return privateKey;
    }
}

