/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.security.certutil;

import jakarta.annotation.Nonnull;
import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.inject.Singleton;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nullable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.glassfish.jersey.media.multipart.BodyPart;
import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.graylog.security.certutil.CaConfiguration;
import org.graylog.security.certutil.CaKeystoreWithPassword;
import org.graylog.security.certutil.CertificateAuthorityChangedEvent;
import org.graylog.security.certutil.ca.CAKeyPair;
import org.graylog.security.certutil.ca.PemCaReader;
import org.graylog.security.certutil.ca.exceptions.CACreationException;
import org.graylog.security.certutil.ca.exceptions.KeyStoreStorageException;
import org.graylog2.Configuration;
import org.graylog2.bootstrap.preflight.web.resources.model.CAType;
import org.graylog2.bootstrap.preflight.web.resources.model.CertificateAuthorityInformation;
import org.graylog2.cluster.certificates.EncryptedCaKeystore;
import org.graylog2.events.ClusterEventBus;
import org.graylog2.plugin.cluster.ClusterConfigService;
import org.graylog2.security.encryption.EncryptedValueService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
class CaPersistenceService {
    private static final Logger LOG;
    public static final String CA_KEYSTORE_ID = "GRAYLOG CA";
    private final CaConfiguration configuration;
    private final String passwordSecret;
    private final ClusterEventBus eventBus;
    private final ClusterConfigService clusterConfigService;
    private final EncryptedValueService encryptionService;

    @Inject
    public CaPersistenceService(Configuration configuration, @Named(value="password_secret") String passwordSecret, ClusterEventBus eventBus, ClusterConfigService clusterConfigService, EncryptedValueService encryptionService) {
        this.configuration = configuration;
        this.clusterConfigService = clusterConfigService;
        this.encryptionService = encryptionService;
        this.passwordSecret = passwordSecret;
        this.eventBus = eventBus;
    }

    public Optional<CertificateAuthorityInformation> get() throws KeyStoreStorageException {
        if (this.configuration.configuredCaExists()) {
            return Optional.of(new CertificateAuthorityInformation("local CA", CAType.LOCAL));
        }
        return this.readFromDatabase().map(c -> new CertificateAuthorityInformation(CA_KEYSTORE_ID, CAType.GENERATED));
    }

    public CertificateAuthorityInformation create(String organization, Integer daysValid) throws CACreationException, KeyStoreStorageException, KeyStoreException {
        Duration certificateValidity = Duration.ofDays(daysValid == null || daysValid == 0 ? 3650L : (long)daysValid.intValue());
        KeyStore keyStore = CAKeyPair.create(organization, this.passwordSecret.toCharArray(), certificateValidity).toKeyStore();
        this.writeToDatabase(keyStore);
        LOG.debug("Generated a new CA.");
        this.triggerCaChangedEvent();
        return this.get().orElseThrow(() -> new IllegalStateException("Failed to obtain CA information, but a CA has been just created. Inconsistent state!"));
    }

    public void upload(@Nullable String password, List<FormDataBodyPart> parts) throws CACreationException {
        char[] providedPassword = password == null ? null : password.toCharArray();
        try {
            KeyStore keyStore = KeyStore.getInstance("PKCS12", "BC");
            keyStore.load(null, null);
            for (BodyPart bodyPart : parts) {
                InputStream is = (InputStream)bodyPart.getEntityAs(InputStream.class);
                byte[] bytes = is.readAllBytes();
                String pem = new String(bytes, StandardCharsets.UTF_8);
                if (pem.contains("-----BEGIN CERTIFICATE")) {
                    PemCaReader.CA ca = PemCaReader.readCA(pem, password);
                    keyStore.setKeyEntry("ca", ca.privateKey(), providedPassword, ca.certificates().toArray(new Certificate[0]));
                    continue;
                }
                ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
                keyStore.load(bais, providedPassword);
            }
            this.writeToDatabase(this.adaptUploadedKeystore(keyStore, providedPassword));
            this.triggerCaChangedEvent();
        }
        catch (IOException | GeneralSecurityException | KeyStoreStorageException ex) {
            LOG.error("Could not write CA: " + ex.getMessage(), (Throwable)ex);
            throw new CACreationException("Could not write CA: " + ex.getMessage(), ex);
        }
    }

    @Nonnull
    private KeyStore adaptUploadedKeystore(KeyStore existingKeystore, char[] existingPassword) throws GeneralSecurityException, IOException {
        String alias;
        KeyStore adaptedKeystore = KeyStore.getInstance("PKCS12");
        adaptedKeystore.load(null, this.passwordSecret.toCharArray());
        ArrayList<String> aliases = Collections.list(existingKeystore.aliases());
        if (aliases.isEmpty()) {
            throw new IllegalStateException("Provided keystore is empty!");
        }
        if (aliases.size() == 1) {
            alias = aliases.iterator().next();
            if (!existingKeystore.isKeyEntry(alias)) {
                throw new IllegalStateException("Only one alias " + alias + " found in the keystore and it doesn't have a key assigned");
            }
        } else {
            throw new IllegalStateException("Keystores with multiple keys not supported yet!");
        }
        LOG.info("Found one alias " + alias + ", extracting and persisting key and certificate chain.");
        Key key = existingKeystore.getKey(alias, existingPassword);
        Certificate[] certChain = existingKeystore.getCertificateChain(alias);
        adaptedKeystore.setKeyEntry("ca", key, this.passwordSecret.toCharArray(), certChain);
        return adaptedKeystore;
    }

    public void startOver() {
        this.clusterConfigService.remove(EncryptedCaKeystore.class);
    }

    private void triggerCaChangedEvent() {
        this.eventBus.post(new CertificateAuthorityChangedEvent());
    }

    public Optional<CaKeystoreWithPassword> loadKeyStore() throws KeyStoreStorageException {
        if (this.configuration.configuredCaExists()) {
            return this.readFromFS();
        }
        return this.readFromDatabase();
    }

    private void writeToDatabase(KeyStore keyStore) throws KeyStoreStorageException {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            keyStore.store(baos, this.passwordSecret.toCharArray());
            String keystoreDataAsString = Base64.getEncoder().encodeToString(baos.toByteArray());
            this.clusterConfigService.write(new EncryptedCaKeystore(this.encryptionService.encrypt(keystoreDataAsString)));
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreStorageException("Failed to save keystore to cluster config service", e);
        }
    }

    private Optional<CaKeystoreWithPassword> readFromDatabase() {
        return Optional.ofNullable(this.clusterConfigService.get(EncryptedCaKeystore.class)).map(EncryptedCaKeystore::keystore).map(this.encryptionService::decrypt).map(Base64.getDecoder()::decode).map(this::parseKEystoreFromString).map(ks -> new CaKeystoreWithPassword((KeyStore)ks, this.passwordSecret));
    }

    @Nonnull
    private KeyStore parseKEystoreFromString(byte[] keystoreAsString) {
        KeyStore keyStore;
        ByteArrayInputStream bais = new ByteArrayInputStream(keystoreAsString);
        try {
            KeyStore keyStore2 = KeyStore.getInstance("PKCS12");
            keyStore2.load(bais, this.passwordSecret.toCharArray());
            keyStore = keyStore2;
        }
        catch (Throwable throwable) {
            try {
                try {
                    bais.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception ex) {
                throw new RuntimeException("Failed to load keystore from Mongo collection", ex);
            }
        }
        bais.close();
        return keyStore;
    }

    private Optional<CaKeystoreWithPassword> readFromFS() throws KeyStoreStorageException {
        Optional<CaKeystoreWithPassword> optional;
        block8: {
            InputStream in = Files.newInputStream(this.configuration.getCaKeystoreFile(), new OpenOption[0]);
            try {
                KeyStore caKeystore = KeyStore.getInstance("PKCS12");
                caKeystore.load(in, this.configuration.getCaPassword().toCharArray());
                optional = Optional.of(caKeystore).map(ks -> new CaKeystoreWithPassword((KeyStore)ks, this.configuration.getCaPassword()));
                if (in == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (in != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException | GeneralSecurityException ex) {
                    throw new KeyStoreStorageException("Could not read keystore: " + ex.getMessage(), ex);
                }
            }
            in.close();
        }
        return optional;
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
        LOG = LoggerFactory.getLogger(CaPersistenceService.class);
    }
}

