package org.gridvise.coherence.cache.entity;

import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.log4j.Logger;
import org.gridvise.coherence.cache.entity.aggregator.DistinctSetValuesAggregator;

import com.tangosol.net.CacheFactory;
import com.tangosol.net.NamedCache;
import com.tangosol.util.Filter;
import com.tangosol.util.InvocableMap.EntryAggregator;
import com.tangosol.util.InvocableMap.EntryProcessor;
import com.tangosol.util.MapListener;
import com.tangosol.util.ValueExtractor;
import com.tangosol.util.aggregator.DistinctValues;
import com.tangosol.util.aggregator.GroupAggregator;
import com.tangosol.util.filter.PresentFilter;

public abstract class AbstractCache<K, V> implements ICache<K, V> {

	private final static Logger LOG = Logger.getLogger(AbstractCache.class);

	protected static final Filter PRESENT_FILTER = new PresentFilter();

	private NamedCache namedCache;

	private boolean indexAdded = false;

	public abstract String getCacheName();

	protected abstract void addIndexes(NamedCache namedCache);

	@Override
	public void addIndex(ValueExtractor valueExtractor) {
		this.getCache().addIndex(valueExtractor, false, null);

	}

	@Override
	public void addMapListener(MapListener mapListener) {
		this.getCache().addMapListener(mapListener);
	}
	@Override
	public void addMapListener(MapListener mapListener, Filter filter, boolean lite) {
		this.getCache().addMapListener(mapListener, filter, lite);
	}
	@Override
	public void addMapListener(MapListener mapListener, K key, boolean lite) {
		this.getCache().addMapListener(mapListener, key, lite);
	}
	@Override
	public void addMapListener(MapListener mapListener, Filter filter) {
		addMapListener(mapListener, filter, false);
	}
	@Override
	public void addMapListener(MapListener mapListener, K key) {
		addMapListener(mapListener, key, false);
	}
	
	@Override
	public void removeMapListener(MapListener mapListener) {
		this.getCache().removeMapListener(mapListener);
	}
	
	@Override
	public void removeMapListener(MapListener mapListener, K key) {
		this.getCache().removeMapListener(mapListener, key);
	}
	
	@Override
	public void removeMapListener(MapListener mapListener, Filter filter) {
		removeMapListener(mapListener, filter);
	}

	
	@Override
	public void remove(Collection<K> keys) {
		LOG.info("removing " + keys.size() + " from " + this.getCacheName());
		NamedCache cache = getCache();
		for (Object key : keys) {
			cache.remove(key);
		}
	}

	@Override
	public void remove(K key) {
		NamedCache cache = getCache();
		cache.remove(key);
	}

	@Override
	public void remove(Filter filter) {
		Set<K> keysToBeRemoved = this.keySet(filter);
		this.remove(keysToBeRemoved);
	}

	public void clearCache() {
		LOG.info("Clearing cache " + this.getCacheName());
		this.getCache().clear();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Set<K> keySet(Filter filter) {
		return this.getCache().keySet(filter);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Collection<Entry<K, V>> entrySet(Filter filter) {
		return this.getCache().entrySet();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<K,V> values(Filter filter) {
		Collection<K> keys = this.getCache().keySet(filter);
		return (Map<K,V>) this.getCache().getAll(keys);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Collection<V> values() {
		return this.getCache().values();
	}

	@Override
	public void clear() {
		this.getCache().clear();
	}

	@Override
	public int size() {
		return this.getCache().size();
	}

	@Override
	public NamedCache getCache() {
		if (this.namedCache == null) {
			this.namedCache = CacheFactory.getCache(getCacheName());
		}
		if (!this.indexAdded) {
			this.addIndexes(this.namedCache);
			this.indexAdded = true;
		}
		return this.namedCache;
	}

	@SuppressWarnings("unchecked")
	@Override
	public V get(K key) {
		return (V) this.getCache().get(key);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<K, V> getAll(Collection<K> keys) {
		return this.getCache().getAll(keys);
	}

	@Override
	public void put(K key, V value) {
		this.getCache().put(key, value);
	}

	@Override
	public void putAll(Map<K, V> map) {
		this.getCache().putAll(map);
	}
	
	public <E> Collection<E> distictValues(ValueExtractor valueExtractor){
		Filter f = null;
		return (Collection<E>) this.getCache().aggregate( f,
				new DistinctValues(valueExtractor));
	}


	@Override
	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictValues(Filter filter, Enum<?> extractMethodName) {
		return (Collection<E>) this.getCache().aggregate(filter,
				new DistinctValues(extractMethodName.toString()));
	}

	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictValues(Filter filter, ValueExtractor valueExtractor) {
		return (Collection<E>) this.getCache().aggregate(filter,
				new DistinctValues(valueExtractor));
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictSetValues(Filter filter,
			Enum<?> setExtractMethodName) {
		return (Collection<E>) this.getCache()
				.aggregate(
						filter,
						new DistinctSetValuesAggregator(setExtractMethodName
								.toString()));
	}

	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictSetValues(Filter filter,
			ValueExtractor valueExtractor) {
		return (Collection<E>) this.getCache().aggregate(filter,
				new DistinctSetValuesAggregator(valueExtractor));
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictValues(Collection<K> keys,
			Enum<?> extractMethodName) {
		return (Collection<E>) this.getCache().aggregate(keys,
				new DistinctValues(extractMethodName.toString()));
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictValues(Collection<K> keys,
			String extractMethodName) {
		return (Collection<E>) this.getCache().aggregate(keys,
				new DistinctValues(extractMethodName.toString()));
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> Collection<E> distictSetValues(Collection<K> keys,
			Enum<?> setExtractMethodName) {
		return (Collection<E>) this.getCache()
				.aggregate(
						keys,
						new DistinctSetValuesAggregator(setExtractMethodName
								.toString()));
	}

	@Override
	@SuppressWarnings("unchecked")
	public <G, E> Map<G, Collection<E>> distictValues(Filter filter,
			Enum<?> extractMethodName, Enum<?> groupByExtractMethodName) {
		GroupAggregator groupAggregator = getDistinctGroupByAggregator(
				extractMethodName, groupByExtractMethodName);
		return (Map<G, Collection<E>>) this.getCache().aggregate(filter,
				groupAggregator);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <G, E> Map<G, Collection<E>> distictSetValues(Filter filter,
			Enum<?> setExtractMethodName, Enum<?> groupByExtractMethodName) {
		GroupAggregator groupSetAggregator = getDistinctSetGroupByAggregator(
				setExtractMethodName, groupByExtractMethodName);
		return (Map<G, Collection<E>>) this.getCache().aggregate(filter,
				groupSetAggregator);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <G, E> Map<G, Collection<E>> distictValues(Collection<K> keys,
			Enum<?> extractMethodName, Enum<?> groupByExtractMethodName) {
		GroupAggregator groupAggregator = getDistinctGroupByAggregator(
				extractMethodName, groupByExtractMethodName);
		return (Map<G, Collection<E>>) this.getCache().aggregate(keys,
				groupAggregator);
	}

	@SuppressWarnings("unchecked")
	public <G, E> Map<G, Collection<E>> distictSetValues(Collection<K> keys,
			Enum<?> setExtractMethodName, Enum<?> groupByExtractMethodName) {
		GroupAggregator groupSetAggregator = getDistinctSetGroupByAggregator(
				setExtractMethodName, groupByExtractMethodName);
		return (Map<G, Collection<E>>) this.getCache().aggregate(keys,
				groupSetAggregator);
	}

	private GroupAggregator getDistinctGroupByAggregator(
			Enum<?> extractMethodName, Enum<?> groupByExtractMethodName) {
		return GroupAggregator.createInstance(groupByExtractMethodName
				.toString(), new DistinctValues(extractMethodName.toString()));
	}

	private GroupAggregator getDistinctSetGroupByAggregator(
			Enum<?> extractMethodName, Enum<?> groupByExtractMethodName) {
		return GroupAggregator.createInstance(
				groupByExtractMethodName.toString(),
				new DistinctSetValuesAggregator(extractMethodName.toString()));
	}

	protected void addIndex(ValueExtractor valueExtractor, NamedCache cache) {
		this.getCache().addIndex(valueExtractor, false, null);
	}

	@Override
	public Object invoke(K key, EntryProcessor entryprocessor) {
		return this.getCache().invoke(key, entryprocessor);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<K, ?> invokeAll(Collection<K> keys, EntryProcessor entryprocessor) {
		return this.getCache().invokeAll(keys, entryprocessor);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object aggregate(Collection<K> keys, EntryAggregator aggregator) {
		return this.getCache().aggregate(keys, aggregator);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object aggregate(Filter filter, EntryAggregator aggregator) {
		return this.getCache().aggregate(filter, aggregator);
	}

}
