//
// --------------------------------------------------------------------------
//  Gurux Ltd
// 
//
//
// Filename:        $HeadURL$
//
// Version:         $Revision$,
//                  $Date$
//                  $Author$
//
// Copyright (c) Gurux Ltd
//
//---------------------------------------------------------------------------
//
//  DESCRIPTION
//
// This file is a part of Gurux Device Framework.
//
// Gurux Device Framework is Open Source software; you can redistribute it
// and/or modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation; version 2 of the License.
// Gurux Device Framework is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
// See the GNU General Public License for more details.
//
// More information of Gurux products: https://www.gurux.org
//
// This code is licensed under the GNU General Public License v2. 
// Full text may be retrieved at http://www.gnu.org/licenses/gpl-2.0.txt
//---------------------------------------------------------------------------

package gurux.dlms.asn;

import java.util.HashMap;

import gurux.dlms.enums.BerType;

final class GXAsn1Settings {
    private int count;
    /**
     * Are comments used.
     */
    private boolean comments;

    private final StringBuilder sb = new StringBuilder();
    private HashMap<Short, String> tags = new HashMap<Short, String>();
    private HashMap<String, Short> tagbyName = new HashMap<String, Short>();

    private void addTag(final int key, final String value) {
        tags.put((short) key, value);
        tagbyName.put(value.toLowerCase(), (short) key);
    }

    /**
     * Constructor.
     */
    GXAsn1Settings() {
        addTag(BerType.APPLICATION, "Application");
        addTag(BerType.CONSTRUCTED | BerType.CONTEXT, "Context");
        addTag(BerType.CONSTRUCTED | BerType.SEQUENCE, "Sequence");
        addTag(BerType.CONSTRUCTED | BerType.SET, "Set");
        addTag(BerType.OBJECT_IDENTIFIER, "ObjectIdentifier");
        addTag(BerType.PRINTABLE_STRING, "String");
        addTag(BerType.UTF8STRING, "UTF8");
        addTag(BerType.IA5_STRING, "IA5");
        addTag(BerType.INTEGER, "Integer");
        addTag(BerType.NULL, "Null");
        addTag(BerType.BIT_STRING, "BitString");
        addTag(BerType.UTC_TIME, "UtcTime");
        addTag(BerType.GENERALIZED_TIME, "GeneralizedTime");
        addTag(BerType.OCTET_STRING, "OctetString");
        addTag(BerType.BOOLEAN, "Bool");
        addTag(-1, "Byte");
        addTag(-2, "Short");
        addTag(-4, "Int");
        addTag(-8, "Long");
    }

    public String getTag(final short value) {
        return tags.get(value);
    }

    public short getTag(final String value) {
        return tagbyName.get(value);
    }

    /**
     * @return XML length.
     */
    public int getXmlLength() {
        return sb.length();
    }

    /**
     * Add comment.
     * 
     * @param offset
     *            Offset.
     * @param value
     *            Comment value.
     */
    public void appendComment(final int offset, final String value) {
        if (comments) {
            boolean empty = sb.length() == 0;
            StringBuilder tmp;
            if (empty) {
                tmp = sb;
            } else {
                tmp = new StringBuilder();
            }
            for (int pos = 0; pos < count - 1; ++pos) {
                tmp.append(' ');
            }
            tmp.append("<!--");
            tmp.append(value);
            tmp.append("-->\r\n");
            if (!empty) {
                sb.insert(offset, tmp);
            }
        }
    }

    /*
     * Append spaces to the buffer.
     */
    public void appendSpaces() {
        for (int pos = 0; pos != count; ++pos) {
            sb.append(' ');
        }
    }

    public void append(final String value) {
        sb.append(value);
    }

    public void increase() {
        ++count;
        append("\r\n");
    }

    public void decrease() {
        --count;
        appendSpaces();
    }

    @Override
    public String toString() {
        return sb.toString();
    }

    /**
     * @return Are comments used.
     */
    public boolean isComments() {
        return comments;
    }

    /**
     * @param value
     *            Are comments used.
     */
    public void setComments(final boolean value) {
        comments = value;
    }
}
