package org.hammerlab.test.matchers.seqs

import org.scalatest.matchers.{ MatchResult, Matcher }

import scala.collection.SortedMap
import scala.collection.mutable.ArrayBuffer

/**
 * Custom [[Matcher]] for [[Seq]]s of key-value pairs. Prints nicely-formatted messages about tuples that differ between
 * one collection and another in specific ways:
 *
 *   - keys (and hance values) present in one and not the other.
 *   - keys that are present in both but with differing values.
 *   - pairs that are present in both but in different orders.
 */
case class PairSeqMatcher[K: Ordering, V: Ordering](expected: Seq[(K, V)], matchOrder: Boolean = true)
  extends Matcher[Seq[(K, V)]] {

  override def apply(actual: Seq[(K, V)]): MatchResult = {

    val expectedMap: Map[K, Set[V]] =
      expected
        .groupBy(_._1)
        .mapValues(_.map(_._2).toSet)

    val actualMap: Map[K, Set[V]] =
      actual
        .groupBy(_._1)
        .mapValues(_.map(_._2).toSet)

    val keys = expectedMap.keySet ++ actualMap.keySet

    val errors = ArrayBuffer[String]()
    errors += "Sequences didn't match!"
    errors += ""

    val differingElemsBuilder = SortedMap.newBuilder[K, (String, String)]
    val extraElemsBuilder = SortedMap.newBuilder[K, String]
    val missingElemsBuilder = SortedMap.newBuilder[K, String]
    for {
      key <- keys
      expectedValues = expectedMap.getOrElse(key, Set())
      actualValues = actualMap.getOrElse(key, Set())
      if expectedValues != actualValues

      missingValues = expectedValues.diff(actualValues).toSeq.sorted
      extraValues = actualValues.diff(expectedValues).toSeq.sorted

      extrasStr = extraValues.mkString(",")
      missingsStr = missingValues.mkString(",")
    } {
      (extraValues.nonEmpty, missingValues.nonEmpty) match {
        case (true, true) =>
          differingElemsBuilder += key -> (extrasStr, missingsStr)
        case (true, false) =>
          extraElemsBuilder += key -> extrasStr
        case (false, true) =>
          missingElemsBuilder += key -> missingsStr
        case (false, false) =>
          // Can't get here.
      }
    }

    val differingElems = differingElemsBuilder.result()
    val extraElems = extraElemsBuilder.result()
    val missingElems = missingElemsBuilder.result()

    if (extraElems.nonEmpty || missingElems.nonEmpty || differingElems.nonEmpty) {

      if (extraElems.nonEmpty) {
        errors += s"Extra elems:"
        errors += extraElems.mkString("\t", "\n\t", "\n")
      }

      if (missingElems.nonEmpty) {
        errors += s"Missing elems:"
        errors += missingElems.mkString("\t", "\n\t", "\n")
      }

      if (differingElems.nonEmpty) {
        val diffLines =
          for {
            (k, (actualValue, expectedValue)) <- differingElems
          } yield
            s"$k: actual: $actualValue, expected: $expectedValue"

        errors += s"Differing values:"
        errors += diffLines.mkString("\t", "\n\t", "\n")
      }
    } else if (matchOrder && actual != expected) {
      errors += s"Elements out of order:"
      errors += "Expected:"
      errors += expected.mkString("\t", "\n\t", "\n")
      errors += "Actual:"
      errors += actual.mkString("\t", "\n\t", "\n")
    }

    val matched =
      if (matchOrder)
        actual == expected
      else
        actual.toSet == expected.toSet

    MatchResult(
      matched,
      errors.mkString("\n"),
      s"$actual matched; was supposed to not."
    )
  }
}

object PairSeqMatcher {
  def pairsMatch[K: Ordering, V: Ordering](expected: Iterable[(K, V)]): Matcher[Seq[(K, V)]] = PairSeqMatcher[K, V](expected.toSeq)
  def pairsMatch[K: Ordering, V: Ordering](expected: Array[(K, V)]): Matcher[Seq[(K, V)]] = PairSeqMatcher[K, V](expected.toList)
  def pairsMatch[K: Ordering, V: Ordering](expected: Iterator[(K, V)]): Matcher[Seq[(K, V)]] = PairSeqMatcher[K, V](expected.toSeq)
}
