/*
 * Copyright 2015-2017 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.hawkular.agent.javaagent.config;

import com.fasterxml.jackson.annotation.JsonValue;

/**
 * Provides a way to serialize and deserialize YAML value as a String but maintain
 * an object of type <T> to be returned by the getter and set by the setter.
 */
public abstract class AbstractStringifiedProperty<T> {

    protected final T defaultValue;
    protected String valueAsString;

    public AbstractStringifiedProperty() {
        this.defaultValue = null;
        set(null);
    }

    public AbstractStringifiedProperty(T defaultVal) {
        this.defaultValue = defaultVal;
        set(defaultVal);
    }

    public AbstractStringifiedProperty(String str) {
        this.defaultValue = null;
        setValueAsString(str);
    }

    // copy-constructor
    public AbstractStringifiedProperty(AbstractStringifiedProperty<T> original) {
        if (original != null) {
            this.defaultValue = original.defaultValue;
            this.valueAsString = original.valueAsString;
        } else {
            this.defaultValue = null;
            set(null);
        }
    }

    public T get() {
        return deserialize(this.valueAsString);
    }

    public void set(T value) {
        setValueAsString(serialize(value != null ? value : this.defaultValue));
    }

    public String getValueAsString() {
        return this.valueAsString;
    }

    public void setValueAsString(String str) {
        if (str == null) {
            throw new IllegalArgumentException("Expression string must not be null");
        }
        this.valueAsString = str;
    }

    @JsonValue
    @Override
    public String toString() {
        return getValueAsString();
    }

    protected abstract T deserialize(String valueAsString);

    protected abstract String serialize(T value);

}