/*
 * Copyright 2015-2016 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.hawkular.apm.api.services;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.hawkular.apm.api.model.trace.CorrelationIdentifier;

/**
 * This class represents the base query criteria.
 *
 * @author gbrown
 */
public class Criteria {

    private static final Logger log = Logger.getLogger(Criteria.class.getName());

    private long startTime = 0L;
    private long endTime = 0L;
    private String businessTransaction;
    private Set<PropertyCriteria> properties = new HashSet<PropertyCriteria>();
    private Set<CorrelationIdentifier> correlationIds = new HashSet<CorrelationIdentifier>();
    private Set<FaultCriteria> faults = new HashSet<FaultCriteria>();
    private String hostName;
    private long upperBound;
    private long lowerBound;
    private String principal;
    private String uri;
    private String operation;

    /**  */
    private static int DEFAULT_RESPONSE_SIZE = 100000;

    /**  */
    private static long DEFAULT_TIMEOUT = 10000L;

    private long timeout = DEFAULT_TIMEOUT;

    private int maxResponseSize = DEFAULT_RESPONSE_SIZE;

    /**
     * @return the startTime, or 0 meaning 1 hours ago
     */
    public long getStartTime() {
        return startTime;
    }

    /**
     * @param startTime the startTime to set
     * @return The criteria
     */
    public Criteria setStartTime(long startTime) {
        this.startTime = startTime;
        return this;
    }

    /**
     * This method calculates the start time to use. If
     * the configured value is 0, then it will default to
     * an hour before the end time. If negative, then the
     * value will be deducted from the end time.
     *
     * @return The calculated start time
     */
    public long calculateStartTime() {
        if (startTime == 0) {
            // Set to 1 hour before end time
            return calculateEndTime() - 3600000;
        } else if (startTime < 0) {
            return calculateEndTime() + startTime;
        }
        return startTime;
    }

    /**
     * @return the endTime, or 0 meaning 'current time'
     */
    public long getEndTime() {
        return endTime;
    }

    /**
     * @param endTime the endTime to set
     * @return The criteria
     */
    public Criteria setEndTime(long endTime) {
        this.endTime = endTime;
        return this;
    }

    /**
     * This method returns an end time based on the configured
     * value. If end time is less or equal to 0, then its value
     * will be deducted from the current time.
     *
     * @return The calculated end time
     */
    public long calculateEndTime() {
        if (endTime == 0) {
            return System.currentTimeMillis();
        } else if (endTime < 0) {
            return System.currentTimeMillis() - endTime;
        }
        return endTime;
    }

    /**
     * @return the business transaction
     */
    public String getBusinessTransaction() {
        return businessTransaction;
    }

    /**
     * If a null name is used, then it will match any transaction whether it has
     * a name or not. If the supplied name is an empty string, then it will match
     * only transactions that don't have a name. If a name is specified, then
     * only transactions with that business transaction name will be selected.
     *
     * @param name the business transaction name to set
     * @return The criteria
     */
    public Criteria setBusinessTransaction(String name) {
        this.businessTransaction = name;
        return this;
    }

    /**
     * @return the properties
     */
    public Set<PropertyCriteria> getProperties() {
        return properties;
    }

    /**
     * @param properties the properties to set
     * @return The criteria
     */
    public Criteria setProperties(Set<PropertyCriteria> properties) {
        this.properties = properties;
        return this;
    }

    /**
     * This method adds a new property criteria.
     *
     * @param name The property name
     * @param value The property value
     * @param operator The property operator
     * @return The criteria
     */
    public Criteria addProperty(String name, String value, Operator operator) {
        properties.add(new PropertyCriteria(name, value, operator));
        return this;
    }

    /**
     * @return the correlationIds
     */
    public Set<CorrelationIdentifier> getCorrelationIds() {
        return correlationIds;
    }

    /**
     * @param correlationIds the correlationIds to set
     * @return The criteria
     */
    public Criteria setCorrelationIds(Set<CorrelationIdentifier> correlationIds) {
        this.correlationIds = correlationIds;
        return this;
    }

    /**
     * @return the hostName
     */
    public String getHostName() {
        return hostName;
    }

    /**
     * @param hostName the hostName to set
     * @return The criteria
     */
    public Criteria setHostName(String hostName) {
        this.hostName = hostName;
        return this;
    }

    /**
     * @return the upperBound
     */
    public long getUpperBound() {
        return upperBound;
    }

    /**
     * @param upperBound the upperBound to set
     */
    public void setUpperBound(long upperBound) {
        this.upperBound = upperBound;
    }

    /**
     * @return the lowerBound
     */
    public long getLowerBound() {
        return lowerBound;
    }

    /**
     * @param lowerBound the lowerBound to set
     */
    public void setLowerBound(long lowerBound) {
        this.lowerBound = lowerBound;
    }

    /**
     * @return the principal
     */
    public String getPrincipal() {
        return principal;
    }

    /**
     * @param principal the principal to set
     * @return The criteria
     */
    public Criteria setPrincipal(String principal) {
        this.principal = principal;
        return this;
    }

    /**
     * @return the uri
     */
    public String getUri() {
        return uri;
    }

    /**
     * @param uri the uri to set
     * @return The criteria
     */
    public Criteria setUri(String uri) {
        this.uri = uri;
        return this;
    }

    /**
     * @return the operation
     */
    public String getOperation() {
        return operation;
    }

    /**
     * @param operation the operation to set
     * @return The criteria
     */
    public Criteria setOperation(String operation) {
        this.operation = operation;
        return this;
    }

    /**
     * @return the faults
     */
    public Set<FaultCriteria> getFaults() {
        return faults;
    }

    /**
     * @param fault the fault to set
     * @return The criteria
     */
    public Criteria setFaults(Set<FaultCriteria> faults) {
        this.faults = faults;
        return this;
    }

    /**
     * @return the timeout
     */
    public long getTimeout() {
        return timeout;
    }

    /**
     * @param timeout the timeout to set
     */
    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    /**
     * @return the maxResponseSize
     */
    public int getMaxResponseSize() {
        return maxResponseSize;
    }

    /**
     * @param maxResponseSize the maxResponseSize to set
     */
    public void setMaxResponseSize(int maxResponseSize) {
        this.maxResponseSize = maxResponseSize;
    }

    /**
     * This method returns the criteria as a map of name/value pairs.
     * The properties and correlation ids are returned as a single
     * entry with | separators.
     *
     * @return The criteria parameters
     */
    public Map<String, String> parameters() {
        Map<String, String> ret = new HashMap<String, String>();

        if (getBusinessTransaction() != null) {
            ret.put("businessTransaction", getBusinessTransaction());
        }

        if (getStartTime() > 0) {
            ret.put("startTime", "" + getStartTime());
        }

        if (getEndTime() > 0) {
            ret.put("endTime", "" + getEndTime());
        }

        if (!getProperties().isEmpty()) {
            boolean first = true;
            StringBuilder buf = new StringBuilder();

            for (PropertyCriteria pc : getProperties()) {
                if (first) {
                    first = false;
                } else {
                    buf.append(',');
                }
                buf.append(pc.encoded());
            }

            ret.put("properties", buf.toString());
        }

        if (hostName != null) {
            ret.put("hostName", hostName);
        }

        // Only relevant for trace fragments
        if (!getCorrelationIds().isEmpty()) {
            boolean first = true;
            StringBuilder buf = new StringBuilder();

            for (CorrelationIdentifier cid : getCorrelationIds()) {
                if (first) {
                    first = false;
                } else {
                    buf.append(',');
                }
                buf.append(cid.getScope().name());
                buf.append('|');
                buf.append(cid.getValue());
            }

            ret.put("correlations", buf.toString());
        }

        // Only relevant for completion time queries
        if (!getFaults().isEmpty()) {
            boolean first = true;
            StringBuilder buf = new StringBuilder();

            for (FaultCriteria pc : getFaults()) {
                if (first) {
                    first = false;
                } else {
                    buf.append(',');
                }
                buf.append(pc.encoded());
            }

            ret.put("faults", buf.toString());
        }

        if (principal != null) {
            ret.put("principal", principal);
        }

        if (uri != null) {
            ret.put("uri", uri);
        }

        if (operation != null) {
            ret.put("operation", operation);
        }

        if (log.isLoggable(Level.FINEST)) {
            log.finest("Criteria parameters [" + ret + "]");
        }

        return ret;
    }

    /**
     * This method determines if the specified criteria are relevant to all fragments within
     * an end to end transaction.
     *
     * @return Whether the criteria would apply to all fragments in a transaction
     */
    public boolean transactionWide() {
        return !(!properties.isEmpty() || !correlationIds.isEmpty() || !faults.isEmpty() || hostName != null
                || uri != null || operation != null);
    }

    /**
     * This method returns the transaction wide version of the current criteria.
     *
     * @return The transaction wide version
     */
    public Criteria deriveTransactionWide() {
        Criteria ret = new Criteria();
        ret.setStartTime(startTime);
        ret.setEndTime(endTime);
        ret.setBusinessTransaction(businessTransaction);
        ret.setPrincipal(principal);
        return ret;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "Criteria [startTime=" + startTime + ", endTime=" + endTime + ", businessTransaction="
                + businessTransaction + ", properties=" + properties + ", correlationIds=" + correlationIds
                + ", faults=" + faults + ", hostName=" + hostName + ", upperBound=" + upperBound + ", lowerBound="
                + lowerBound + ", principal=" + principal + ", uri=" + uri + ", operation=" + operation + ", timeout="
                + timeout + ", maxResponseSize=" + maxResponseSize + "]";
    }

    /**
     * The enum for the comparison operators. The operators are specific
     * to the property type (e.g. Text, Number)
     */
    public static enum Operator {

        /* Text value - matching property/fault operator */
        HAS,

        /* Text value - no matching property/fault operator */
        HASNOT,

        /* Number value - property equality operator */
        EQ,

        /* Number value - property inequality operator */
        NE,

        /* Number value - property greater-than operator */
        GT,

        /* Number value - property greater-than-or-equal operator */
        GTE,

        /* Number value - property less-than operator */
        LT,

        /* Number value - property less-than-or-equal operator */
        LTE

    }

    /**
     * This class represents the property criteria.
     */
    public static class PropertyCriteria {

        private String name;
        private String value;

        private Operator operator = Operator.HAS;

        /**
         * This is the default constructor.
         */
        public PropertyCriteria() {
        }

        /**
         * This constructor initialises the fields.
         *
         * @param name The name
         * @param value The value
         * @param operator The comparison operator
         */
        public PropertyCriteria(String name, String value, Operator operator) {
            this.name = name;
            this.value = value;
            this.setOperator(operator);
        }

        /**
         * @return the name
         */
        public String getName() {
            return name;
        }

        /**
         * @param name the name to set
         */
        public void setName(String name) {
            this.name = name;
        }

        /**
         * @return the value
         */
        public String getValue() {
            return value;
        }

        /**
         * @param value the value to set
         */
        public void setValue(String value) {
            this.value = value;
        }

        /**
         * @return the operator
         */
        public Operator getOperator() {
            return operator;
        }

        /**
         * @param operator the operator to set
         */
        public void setOperator(Operator operator) {
            if (operator == null) {
                operator = Operator.HAS;
            }
            this.operator = operator;
        }

        /**
         * This method returns an encoded form for the
         * property criteria.
         *
         * @return The encoded form
         */
        public String encoded() {
            StringBuilder buf = new StringBuilder();
            buf.append(getName());
            buf.append('|');
            buf.append(getValue());
            if (getOperator() != Operator.HAS) {
                buf.append('|');
                buf.append(getOperator());
            }
            return buf.toString();
        }

        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "PropertyCriteria [name=" + name + ", value=" + value + ", operator=" + operator + "]";
        }

    }

    /**
     * This class represents the fault criteria.
     */
    public static class FaultCriteria {

        private String value;

        private Operator operator = Operator.HAS;

        /**
         * This is the default constructor.
         */
        public FaultCriteria() {
        }

        /**
         * This constructor initialises the fields.
         *
         * @param value The value
         * @param operator The comparison operator
         */
        public FaultCriteria(String value, Operator operator) {
            this.value = value;
            setOperator(operator);
        }

        /**
         * @return the value
         */
        public String getValue() {
            return value;
        }

        /**
         * @param value the value to set
         */
        public void setValue(String value) {
            this.value = value;
        }

        /**
         * @return the operator
         */
        public Operator getOperator() {
            return operator;
        }

        /**
         * @param operator the operator to set
         */
        public void setOperator(Operator operator) {
            if (operator == null) {
                operator = Operator.HAS;
            }
            this.operator = operator;
        }

        /**
         * This method returns an encoded form for the
         * property criteria.
         *
         * @return The encoded form
         */
        public String encoded() {
            StringBuilder buf = new StringBuilder();
            buf.append(getValue());
            if (getOperator() != Operator.HAS) {
                buf.append('|');
                buf.append(getOperator());
            }
            return buf.toString();
        }

        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "FaultCriteria [value=" + value + ", operator=" + operator + "]";
        }
    }
}
