/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.codahale.metrics.Timer;
import com.github.tjake.ICRC32;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CompressedSegment;
import org.apache.cassandra.db.commitlog.MemoryMappedSegment;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.CRC32Factory;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.cliffc.high_scale_lib.NonBlockingHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static final long idBase;
    private static final AtomicInteger nextId;
    public static final int ENTRY_OVERHEAD_SIZE = 12;
    static final int SYNC_MARKER_SIZE = 8;
    private final OpOrder appendOrder = new OpOrder();
    private final AtomicInteger allocatePosition = new AtomicInteger();
    private volatile int lastSyncedOffset;
    private int endOfBuffer;
    private final WaitQueue syncComplete = new WaitQueue();
    private final NonBlockingHashMap<UUID, AtomicInteger> cfDirty = new NonBlockingHashMap(1024);
    private final ConcurrentHashMap<UUID, AtomicInteger> cfClean = new ConcurrentHashMap();
    public final long id;
    final File logFile;
    final FileChannel channel;
    final int fd;
    ByteBuffer buffer;
    final CommitLog commitLog;
    public final CommitLogDescriptor descriptor;

    static CommitLogSegment createSegment(CommitLog commitLog) {
        return commitLog.compressor != null ? new CompressedSegment(commitLog) : new MemoryMappedSegment(commitLog);
    }

    static long getNextId() {
        return idBase + (long)nextId.getAndIncrement();
    }

    CommitLogSegment(CommitLog commitLog) {
        this.commitLog = commitLog;
        this.id = CommitLogSegment.getNextId();
        this.descriptor = new CommitLogDescriptor(this.id, commitLog.compressorClass);
        this.logFile = new File(commitLog.location, this.descriptor.fileName());
        try {
            this.channel = FileChannel.open(this.logFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE);
            this.fd = CLibrary.getfd(this.channel);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.logFile);
        }
        this.buffer = this.createBuffer(commitLog);
        CommitLogDescriptor.writeHeader(this.buffer, this.descriptor);
        this.endOfBuffer = this.buffer.capacity();
        this.lastSyncedOffset = this.buffer.position();
        this.allocatePosition.set(this.lastSyncedOffset + 8);
    }

    abstract ByteBuffer createBuffer(CommitLog var1);

    Allocation allocate(Mutation mutation, int size) {
        OpOrder.Group opGroup = this.appendOrder.start();
        try {
            int position = this.allocate(size);
            if (position < 0) {
                opGroup.close();
                return null;
            }
            this.markDirty(mutation, position);
            return new Allocation(this, opGroup, position, (ByteBuffer)this.buffer.duplicate().position(position).limit(position + size));
        }
        catch (Throwable t) {
            opGroup.close();
            throw t;
        }
    }

    private int allocate(int size) {
        int next;
        int prev;
        do {
            if ((next = (prev = this.allocatePosition.get()) + size) < this.endOfBuffer) continue;
            return -1;
        } while (!this.allocatePosition.compareAndSet(prev, next));
        assert (this.buffer != null);
        return prev;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    void discardUnusedTail() {
        try (OpOrder.Group group = this.appendOrder.start();){
            while (true) {
                int next;
                int prev;
                if ((prev = this.allocatePosition.get()) >= (next = this.endOfBuffer + 1)) {
                    assert (this.buffer == null || prev == this.buffer.capacity() + 1);
                    return;
                }
                if (this.allocatePosition.compareAndSet(prev, next)) {
                    this.endOfBuffer = prev;
                    assert (this.buffer != null && next == this.buffer.capacity() + 1);
                    return;
                }
                continue;
                break;
            }
        }
    }

    void waitForModifications() {
        this.appendOrder.awaitNewBarrier();
    }

    synchronized void sync() {
        boolean close = false;
        if (this.allocatePosition.get() <= this.lastSyncedOffset + 8) {
            return;
        }
        assert (this.buffer != null);
        int startMarker = this.lastSyncedOffset;
        int nextMarker = this.allocate(8);
        if (nextMarker < 0) {
            this.discardUnusedTail();
            close = true;
            nextMarker = this.buffer.capacity();
        }
        this.waitForModifications();
        int sectionEnd = close ? this.endOfBuffer : nextMarker;
        this.write(startMarker, sectionEnd);
        this.lastSyncedOffset = nextMarker;
        if (close) {
            this.internalClose();
        }
        this.syncComplete.signalAll();
    }

    protected void writeSyncMarker(ByteBuffer buffer, int offset, int filePos, int nextMarker) {
        ICRC32 crc = CRC32Factory.instance.create();
        crc.updateInt((int)(this.id & 0xFFFFFFFFL));
        crc.updateInt((int)(this.id >>> 32));
        crc.updateInt(filePos);
        buffer.putInt(offset, nextMarker);
        buffer.putInt(offset + 4, crc.getCrc());
    }

    abstract void write(int var1, int var2);

    public boolean isStillAllocating() {
        return this.allocatePosition.get() < this.endOfBuffer;
    }

    void discard(boolean deleteFile) {
        this.close();
        if (deleteFile) {
            FileUtils.deleteWithConfirm(this.logFile);
        }
        this.commitLog.allocator.addSize(-this.onDiskSize());
    }

    public ReplayPosition getContext() {
        return new ReplayPosition(this.id, this.allocatePosition.get());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    void waitForFinalSync() {
        WaitQueue.Signal signal;
        while (true) {
            signal = this.syncComplete.register();
            if (this.lastSyncedOffset >= this.endOfBuffer) break;
            signal.awaitUninterruptibly();
        }
        signal.cancel();
    }

    void waitForSync(int position, Timer waitingOnCommit) {
        while (this.lastSyncedOffset < position) {
            WaitQueue.Signal signal;
            WaitQueue.Signal signal2 = signal = waitingOnCommit != null ? this.syncComplete.register(waitingOnCommit.time()) : this.syncComplete.register();
            if (this.lastSyncedOffset < position) {
                signal.awaitUninterruptibly();
                continue;
            }
            signal.cancel();
        }
    }

    synchronized void close() {
        this.discardUnusedTail();
        this.sync();
        assert (this.buffer == null);
    }

    protected void internalClose() {
        try {
            this.channel.close();
            this.buffer = null;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    void markDirty(Mutation mutation, int allocatedPosition) {
        for (ColumnFamily columnFamily : mutation.getColumnFamilies()) {
            CFMetaData cfm = columnFamily.metadata();
            if (cfm.isPurged()) {
                logger.error("Attempted to write commit log entry for unrecognized table: {}", (Object)columnFamily.id());
                continue;
            }
            CommitLogSegment.ensureAtleast(this.cfDirty, cfm.cfId, allocatedPosition);
        }
    }

    public synchronized void markClean(UUID cfId, ReplayPosition context) {
        if (!this.cfDirty.containsKey(cfId)) {
            return;
        }
        if (context.segment == this.id) {
            this.markClean(cfId, context.position);
        } else if (context.segment > this.id) {
            this.markClean(cfId, Integer.MAX_VALUE);
        }
    }

    private void markClean(UUID cfId, int position) {
        CommitLogSegment.ensureAtleast(this.cfClean, cfId, position);
        this.removeCleanFromDirty();
    }

    private static void ensureAtleast(ConcurrentMap<UUID, AtomicInteger> map, UUID cfId, int value) {
        int cur;
        AtomicInteger i2;
        AtomicInteger i = (AtomicInteger)map.get(cfId);
        if (i == null && (i2 = map.putIfAbsent(cfId, i = new AtomicInteger())) != null) {
            i = i2;
        }
        while ((cur = i.get()) <= value && !i.compareAndSet(cur, value)) {
        }
    }

    private void removeCleanFromDirty() {
        if (this.isStillAllocating()) {
            return;
        }
        Iterator iter2 = this.cfClean.entrySet().iterator();
        while (iter2.hasNext()) {
            Map.Entry clean = iter2.next();
            UUID cfId = (UUID)clean.getKey();
            AtomicInteger cleanPos = (AtomicInteger)clean.getValue();
            AtomicInteger dirtyPos = this.cfDirty.get(cfId);
            if (dirtyPos == null || dirtyPos.intValue() > cleanPos.intValue()) continue;
            this.cfDirty.remove(cfId);
            iter2.remove();
        }
    }

    public synchronized Collection<UUID> getDirtyCFIDs() {
        if (this.cfClean.isEmpty() || this.cfDirty.isEmpty()) {
            return this.cfDirty.keySet();
        }
        ArrayList<UUID> r = new ArrayList<UUID>(this.cfDirty.size());
        for (Map.Entry<UUID, AtomicInteger> dirty : this.cfDirty.entrySet()) {
            UUID cfId = dirty.getKey();
            AtomicInteger dirtyPos = dirty.getValue();
            AtomicInteger cleanPos = (AtomicInteger)this.cfClean.get(cfId);
            if (cleanPos != null && cleanPos.intValue() >= dirtyPos.intValue()) continue;
            r.add(dirty.getKey());
        }
        return r;
    }

    public synchronized boolean isUnused() {
        if (this.isStillAllocating()) {
            return false;
        }
        this.removeCleanFromDirty();
        return this.cfDirty.isEmpty();
    }

    public boolean contains(ReplayPosition context) {
        return context.segment == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (UUID cfId : this.getDirtyCFIDs()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    public abstract long onDiskSize();

    public long contentSize() {
        return this.lastSyncedOffset;
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    static {
        nextId = new AtomicInteger(1);
        long maxId = Long.MIN_VALUE;
        for (File file : new File(DatabaseDescriptor.getCommitLogLocation()).listFiles()) {
            if (!CommitLogDescriptor.isValid(file.getName())) continue;
            maxId = Math.max(CommitLogDescriptor.fromFileName((String)file.getName()).id, maxId);
        }
        idBase = Math.max(System.currentTimeMillis(), maxId + 1L);
    }

    static class Allocation {
        private final CommitLogSegment segment;
        private final OpOrder.Group appendOp;
        private final int position;
        private final ByteBuffer buffer;

        Allocation(CommitLogSegment segment, OpOrder.Group appendOp, int position, ByteBuffer buffer) {
            this.segment = segment;
            this.appendOp = appendOp;
            this.position = position;
            this.buffer = buffer;
        }

        CommitLogSegment getSegment() {
            return this.segment;
        }

        ByteBuffer getBuffer() {
            return this.buffer;
        }

        void markWritten() {
            this.appendOp.close();
        }

        void awaitDiskSync(Timer waitingOnCommit) {
            this.segment.waitForSync(this.position, waitingOnCommit);
        }

        public ReplayPosition getReplayPosition() {
            return new ReplayPosition(this.segment.id, this.buffer.limit());
        }
    }

    public static class CommitLogSegmentFileComparator
    implements Comparator<File> {
        @Override
        public int compare(File f, File f2) {
            CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(f.getName());
            CommitLogDescriptor desc2 = CommitLogDescriptor.fromFileName(f2.getName());
            return Long.compare(desc.id, desc2.id);
        }
    }
}

