/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.primitives.Longs;
import java.io.BufferedOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.compress.CompressionParameters;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.Memory;
import org.apache.cassandra.io.util.SafeMemory;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.Transactional;

public class CompressionMetadata {
    public final long dataLength;
    public final long compressedFileLength;
    private final Memory chunkOffsets;
    private final long chunkOffsetsSize;
    public final String indexFilePath;
    public final CompressionParameters parameters;

    public static CompressionMetadata create(String dataFilePath) {
        Descriptor desc = Descriptor.fromFilename(dataFilePath);
        return new CompressionMetadata(desc.filenameFor(Component.COMPRESSION_INFO), new File(dataFilePath).length());
    }

    @VisibleForTesting
    CompressionMetadata(String indexFilePath, long compressedLength) {
        this.indexFilePath = indexFilePath;
        try (DataInputStream stream = new DataInputStream(new FileInputStream(indexFilePath));){
            String compressorName = stream.readUTF();
            int optionCount = stream.readInt();
            HashMap<String, String> options = new HashMap<String, String>(optionCount);
            for (int i = 0; i < optionCount; ++i) {
                String key = stream.readUTF();
                String value = stream.readUTF();
                options.put(key, value);
            }
            int chunkLength = stream.readInt();
            try {
                this.parameters = new CompressionParameters(compressorName, (Integer)chunkLength, options);
            }
            catch (ConfigurationException e) {
                throw new RuntimeException("Cannot create CompressionParameters for stored parameters", e);
            }
            this.dataLength = stream.readLong();
            this.compressedFileLength = compressedLength;
            this.chunkOffsets = this.readChunkOffsets(stream);
        }
        catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new CorruptSSTableException((Exception)e, indexFilePath);
        }
        this.chunkOffsetsSize = this.chunkOffsets.size();
    }

    private CompressionMetadata(String filePath, CompressionParameters parameters, SafeMemory offsets, long offsetsSize, long dataLength, long compressedLength) {
        this.indexFilePath = filePath;
        this.parameters = parameters;
        this.dataLength = dataLength;
        this.compressedFileLength = compressedLength;
        this.chunkOffsets = offsets;
        this.chunkOffsetsSize = offsetsSize;
    }

    public ICompressor compressor() {
        return this.parameters.sstableCompressor;
    }

    public int chunkLength() {
        return this.parameters.chunkLength();
    }

    public long offHeapSize() {
        return this.chunkOffsets.size();
    }

    private Memory readChunkOffsets(DataInput input) {
        int chunkCount;
        try {
            chunkCount = input.readInt();
            if (chunkCount <= 0) {
                throw new IOException("Compressed file with 0 chunks encountered: " + input);
            }
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, this.indexFilePath);
        }
        Memory offsets = Memory.allocate((long)chunkCount * 8L);
        int i = 0;
        try {
            for (i = 0; i < chunkCount; ++i) {
                offsets.setLong((long)i * 8L, input.readLong());
            }
            return offsets;
        }
        catch (IOException e) {
            if (offsets != null) {
                offsets.close();
            }
            if (e instanceof EOFException) {
                String msg = String.format("Corrupted Index File %s: read %d but expected %d chunks.", this.indexFilePath, i, chunkCount);
                throw new CorruptSSTableException((Exception)new IOException(msg, e), this.indexFilePath);
            }
            throw new FSReadError((Throwable)e, this.indexFilePath);
        }
    }

    public Chunk chunkFor(long position) {
        int idx = 8 * (int)(position / (long)this.parameters.chunkLength());
        if ((long)idx >= this.chunkOffsetsSize) {
            throw new CorruptSSTableException((Exception)new EOFException(), this.indexFilePath);
        }
        long chunkOffset = this.chunkOffsets.getLong(idx);
        long nextChunkOffset = (long)(idx + 8) == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(idx + 8);
        return new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L));
    }

    public Chunk[] getChunksForSections(Collection<Pair<Long, Long>> sections) {
        TreeSet<Chunk> offsets = new TreeSet<Chunk>(new Comparator<Chunk>(){

            @Override
            public int compare(Chunk o1, Chunk o2) {
                return Longs.compare(o1.offset, o2.offset);
            }
        });
        for (Pair<Long, Long> section : sections) {
            int startIndex = (int)((Long)section.left / (long)this.parameters.chunkLength());
            int endIndex = (int)((Long)section.right / (long)this.parameters.chunkLength());
            endIndex = (Long)section.right % (long)this.parameters.chunkLength() == 0L ? endIndex - 1 : endIndex;
            for (int i = startIndex; i <= endIndex; ++i) {
                long offset = (long)i * 8L;
                long chunkOffset = this.chunkOffsets.getLong(offset);
                long nextChunkOffset = offset + 8L == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(offset + 8L);
                offsets.add(new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L)));
            }
        }
        return offsets.toArray(new Chunk[offsets.size()]);
    }

    public void close() {
        this.chunkOffsets.close();
    }

    static class ChunkSerializer
    implements IVersionedSerializer<Chunk> {
        ChunkSerializer() {
        }

        @Override
        public void serialize(Chunk chunk, DataOutputPlus out, int version) throws IOException {
            out.writeLong(chunk.offset);
            out.writeInt(chunk.length);
        }

        @Override
        public Chunk deserialize(DataInput in, int version) throws IOException {
            return new Chunk(in.readLong(), in.readInt());
        }

        @Override
        public long serializedSize(Chunk chunk, int version) {
            long size = TypeSizes.NATIVE.sizeof(chunk.offset);
            return size += (long)TypeSizes.NATIVE.sizeof(chunk.length);
        }
    }

    public static class Chunk {
        public static final IVersionedSerializer<Chunk> serializer = new ChunkSerializer();
        public final long offset;
        public final int length;

        public Chunk(long offset, int length) {
            assert (length > 0);
            this.offset = offset;
            this.length = length;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Chunk chunk = (Chunk)o;
            return this.length == chunk.length && this.offset == chunk.offset;
        }

        public int hashCode() {
            int result = (int)(this.offset ^ this.offset >>> 32);
            result = 31 * result + this.length;
            return result;
        }

        public String toString() {
            return String.format("Chunk<offset: %d, length: %d>", this.offset, this.length);
        }
    }

    public static class Writer
    extends Transactional.AbstractTransactional
    implements Transactional {
        private final CompressionParameters parameters;
        private final String filePath;
        private int maxCount = 100;
        private SafeMemory offsets = new SafeMemory((long)this.maxCount * 8L);
        private int count = 0;
        private long dataLength;
        private long chunkCount;

        private Writer(CompressionParameters parameters, String path) {
            this.parameters = parameters;
            this.filePath = path;
        }

        public static Writer open(CompressionParameters parameters, String path) {
            return new Writer(parameters, path);
        }

        public void addOffset(long offset) {
            if (this.count == this.maxCount) {
                this.maxCount = (int)((long)this.maxCount * 2L);
                SafeMemory newOffsets = this.offsets.copy((long)this.maxCount * 8L);
                this.offsets.close();
                this.offsets = newOffsets;
            }
            this.offsets.setLong(8L * (long)this.count++, offset);
        }

        private void writeHeader(DataOutput out, long dataLength, int chunks) {
            try {
                out.writeUTF(this.parameters.sstableCompressor.getClass().getSimpleName());
                out.writeInt(this.parameters.otherOptions.size());
                for (Map.Entry<String, String> entry : this.parameters.otherOptions.entrySet()) {
                    out.writeUTF(entry.getKey());
                    out.writeUTF(entry.getValue());
                }
                out.writeInt(this.parameters.chunkLength());
                out.writeLong(dataLength);
                out.writeInt(chunks);
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, this.filePath);
            }
        }

        public Writer finalizeLength(long dataLength, int chunkCount) {
            this.dataLength = dataLength;
            this.chunkCount = chunkCount;
            return this;
        }

        @Override
        public void doPrepare() {
            assert (this.chunkCount == (long)this.count);
            if (this.offsets.size() != (long)this.count * 8L) {
                SafeMemory tmp = this.offsets;
                this.offsets = this.offsets.copy((long)this.count * 8L);
                tmp.free();
            }
            try (DataOutputStream out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.filePath)));){
                this.writeHeader(out, this.dataLength, this.count);
                for (int i = 0; i < this.count; ++i) {
                    out.writeLong(this.offsets.getLong((long)i * 8L));
                }
            }
            catch (IOException e) {
                throw Throwables.propagate(e);
            }
        }

        public CompressionMetadata open(long dataLength, long compressedLength) {
            SafeMemory offsets = this.offsets.sharedCopy();
            int count = (int)(dataLength / (long)this.parameters.chunkLength());
            if (dataLength % (long)this.parameters.chunkLength() != 0L) {
                ++count;
            }
            assert (count > 0);
            if (count < this.count) {
                compressedLength = offsets.getLong((long)count * 8L);
            }
            return new CompressionMetadata(this.filePath, this.parameters, offsets, (long)count * 8L, dataLength, compressedLength);
        }

        public long chunkOffsetBy(int chunkIndex) {
            return this.offsets.getLong((long)chunkIndex * 8L);
        }

        public void resetAndTruncate(int chunkIndex) {
            this.count = chunkIndex;
        }

        @Override
        protected Throwable doPreCleanup(Throwable failed) {
            return this.offsets.close(failed);
        }

        @Override
        protected Throwable doCommit(Throwable accumulate) {
            return accumulate;
        }

        @Override
        protected Throwable doAbort(Throwable accumulate) {
            return FileUtils.deleteWithConfirm(this.filePath, false, accumulate);
        }
    }
}

