/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.CRC32;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.UnknownColumnFamilyException;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.commitlog.SegmentReader;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.SerializationHelper;
import org.apache.cassandra.io.util.ChannelProxy;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.commons.lang3.StringUtils;
import org.cliffc.high_scale_lib.NonBlockingHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogReplayer {
    @VisibleForTesting
    public static long MAX_OUTSTANDING_REPLAY_BYTES = Long.getLong("cassandra.commitlog_max_outstanding_replay_bytes", 0x4000000L);
    @VisibleForTesting
    public static MutationInitiator mutationInitiator = new MutationInitiator();
    static final String IGNORE_REPLAY_ERRORS_PROPERTY = "cassandra.commitlog.ignorereplayerrors";
    private static final Logger logger = LoggerFactory.getLogger(CommitLogReplayer.class);
    private static final int MAX_OUTSTANDING_REPLAY_COUNT = Integer.getInteger("cassandra.commitlog_max_outstanding_replay_count", 1024);
    private static final int LEGACY_END_OF_SEGMENT_MARKER = 0;
    private final Set<Keyspace> keyspacesRecovered = new NonBlockingHashSet<Keyspace>();
    private final Queue<Future<Integer>> futures = new ArrayDeque<Future<Integer>>();
    private final Map<UUID, AtomicInteger> invalidMutations;
    private final AtomicInteger replayedCount;
    private final Map<UUID, ReplayPosition> cfPositions;
    private final ReplayPosition globalPosition;
    private final CRC32 checksum;
    private byte[] buffer = new byte[4096];
    private long pendingMutationBytes = 0L;
    private final ReplayFilter replayFilter;
    private final CommitLogArchiver archiver;

    CommitLogReplayer(CommitLog commitLog, ReplayPosition globalPosition, Map<UUID, ReplayPosition> cfPositions, ReplayFilter replayFilter) {
        this.invalidMutations = new HashMap<UUID, AtomicInteger>();
        this.replayedCount = new AtomicInteger();
        this.checksum = new CRC32();
        this.cfPositions = cfPositions;
        this.globalPosition = globalPosition;
        this.replayFilter = replayFilter;
        this.archiver = commitLog.archiver;
    }

    public static CommitLogReplayer construct(CommitLog commitLog) {
        HashMap<UUID, ReplayPosition> cfPositions = new HashMap<UUID, ReplayPosition>();
        Ordering<ReplayPosition> replayPositionOrdering = Ordering.from(ReplayPosition.comparator);
        ReplayFilter replayFilter = ReplayFilter.create();
        for (ColumnFamilyStore cfs : ColumnFamilyStore.all()) {
            ReplayPosition rp = ReplayPosition.getReplayPosition(cfs.getSSTables(SSTableSet.CANONICAL));
            ReplayPosition truncatedAt = SystemKeyspace.getTruncatedPosition(cfs.metadata.cfId);
            if (truncatedAt != null) {
                long restoreTime = commitLog.archiver.restorePointInTime;
                long truncatedTime = SystemKeyspace.getTruncatedAt(cfs.metadata.cfId);
                if (truncatedTime > restoreTime) {
                    if (replayFilter.includes(cfs.metadata)) {
                        logger.info("Restore point in time is before latest truncation of table {}.{}. Clearing truncation record.", (Object)cfs.metadata.ksName, (Object)cfs.metadata.cfName);
                        SystemKeyspace.removeTruncationRecord(cfs.metadata.cfId);
                    }
                } else {
                    rp = replayPositionOrdering.max(Arrays.asList(rp, truncatedAt));
                }
            }
            cfPositions.put(cfs.metadata.cfId, rp);
        }
        ReplayPosition globalPosition = (ReplayPosition)replayPositionOrdering.min(cfPositions.values());
        logger.trace("Global replay position is {} from columnfamilies {}", (Object)globalPosition, (Object)FBUtilities.toString(cfPositions));
        return new CommitLogReplayer(commitLog, globalPosition, cfPositions, replayFilter);
    }

    public void recover(File[] clogs) throws IOException {
        for (int i = 0; i < clogs.length; ++i) {
            this.recover(clogs[i], i + 1 == clogs.length);
        }
    }

    public int blockForWrites() {
        for (Map.Entry<UUID, AtomicInteger> entry : this.invalidMutations.entrySet()) {
            logger.warn(String.format("Skipped %d mutations from unknown (probably removed) CF with id %s", entry.getValue().intValue(), entry.getKey()));
        }
        FBUtilities.waitOnFutures(this.futures);
        logger.trace("Finished waiting on mutations from recovery");
        this.futures.clear();
        boolean flushingSystem = false;
        ArrayList futures = new ArrayList();
        for (Keyspace keyspace : this.keyspacesRecovered) {
            if (keyspace.getName().equals("system")) {
                flushingSystem = true;
            }
            futures.addAll(keyspace.flush());
        }
        if (!flushingSystem) {
            futures.add(Keyspace.open("system").getColumnFamilyStore("batches").forceFlush());
        }
        FBUtilities.waitOnFutures(futures);
        return this.replayedCount.get();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void recover(File file, boolean tolerateTruncation) throws IOException {
        CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(file.getName());
        try (ChannelProxy channel = new ChannelProxy(file);
             RandomAccessReader reader = RandomAccessReader.open(channel);){
            SegmentReader segmentReader;
            if (desc.version < 4) {
                if (this.logAndCheckIfShouldSkip(file, desc)) {
                    return;
                }
                if (this.globalPosition.segment == desc.id) {
                    reader.seek(this.globalPosition.position);
                }
                this.replaySyncSection(reader, (int)reader.length(), desc, desc.fileName(), tolerateTruncation);
                return;
            }
            long segmentId = desc.id;
            try {
                desc = CommitLogDescriptor.readHeader(reader, DatabaseDescriptor.getEncryptionContext());
            }
            catch (Exception e) {
                desc = null;
            }
            if (desc == null) {
                CommitLogReplayer.handleReplayError(false, "Could not read commit log descriptor in file %s", file);
                return;
            }
            if (segmentId != desc.id) {
                CommitLogReplayer.handleReplayError(false, "Segment id mismatch (filename %d, descriptor %d) in file %s", segmentId, desc.id, file);
            }
            if (this.logAndCheckIfShouldSkip(file, desc)) {
                return;
            }
            try {
                segmentReader = new SegmentReader(desc, reader, tolerateTruncation);
            }
            catch (Exception e) {
                CommitLogReplayer.handleReplayError(false, "unable to create segment reader for commit log file: %s", e);
                if (reader != null) {
                    if (var7_9 != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable) {
                            var7_9.addSuppressed(throwable);
                        }
                    } else {
                        reader.close();
                    }
                }
                if (channel == null) return;
                if (var5_5 == null) {
                    channel.close();
                    return;
                }
                try {
                    channel.close();
                    return;
                }
                catch (Throwable throwable) {
                    var5_5.addSuppressed(throwable);
                    return;
                }
            }
            try {
                boolean tolerateErrorsInSection = tolerateTruncation;
                for (SegmentReader.SyncSegment syncSegment : segmentReader) {
                    String errorContext;
                    if (desc.id == this.globalPosition.segment && syncSegment.endPosition < this.globalPosition.position || this.replaySyncSection(syncSegment.input, syncSegment.endPosition, desc, errorContext = String.format("next section at %d in %s", syncSegment.fileStartPosition, desc.fileName()), tolerateErrorsInSection &= syncSegment.toleratesErrorsInSection)) continue;
                    break;
                }
            }
            catch (RuntimeException re) {
                if (!(re.getCause() instanceof IOException)) throw re;
                throw (IOException)re.getCause();
            }
            logger.debug("Finished reading {}", (Object)file);
            return;
        }
    }

    public boolean logAndCheckIfShouldSkip(File file, CommitLogDescriptor desc) {
        logger.debug("Replaying {} (CL version {}, messaging version {}, compression {})", file.getPath(), desc.version, desc.getMessagingVersion(), desc.compression);
        if (this.globalPosition.segment > desc.id) {
            logger.trace("skipping replay of fully-flushed {}", (Object)file);
            return true;
        }
        return false;
    }

    private boolean replaySyncSection(FileDataInput reader, int end, CommitLogDescriptor desc, String errorContext, boolean tolerateErrors) throws IOException {
        while (reader.getFilePointer() < (long)end && !reader.isEOF()) {
            long claimedCRC32;
            int serializedSize;
            long mutationStart = reader.getFilePointer();
            if (logger.isTraceEnabled()) {
                logger.trace("Reading mutation at {}", (Object)mutationStart);
            }
            try {
                serializedSize = reader.readInt();
                if (serializedSize == 0) {
                    logger.trace("Encountered end of segment marker at {}", (Object)reader.getFilePointer());
                    return false;
                }
                if (serializedSize < 10) {
                    CommitLogReplayer.handleReplayError(tolerateErrors, "Invalid mutation size %d at %d in %s", serializedSize, mutationStart, errorContext);
                    return false;
                }
                long claimedSizeChecksum = desc.version < 4 ? reader.readLong() : (long)reader.readInt() & 0xFFFFFFFFL;
                this.checksum.reset();
                if (desc.version < 3) {
                    this.checksum.update(serializedSize);
                } else {
                    FBUtilities.updateChecksumInt(this.checksum, serializedSize);
                }
                if (this.checksum.getValue() != claimedSizeChecksum) {
                    CommitLogReplayer.handleReplayError(tolerateErrors, "Mutation size checksum failure at %d in %s", mutationStart, errorContext);
                    return false;
                }
                if (serializedSize > this.buffer.length) {
                    this.buffer = new byte[(int)(1.2 * (double)serializedSize)];
                }
                reader.readFully(this.buffer, 0, serializedSize);
                claimedCRC32 = desc.version < 4 ? reader.readLong() : (long)reader.readInt() & 0xFFFFFFFFL;
            }
            catch (EOFException eof) {
                CommitLogReplayer.handleReplayError(tolerateErrors, "Unexpected end of segment", mutationStart, errorContext);
                return false;
            }
            this.checksum.update(this.buffer, 0, serializedSize);
            if (claimedCRC32 != this.checksum.getValue()) {
                CommitLogReplayer.handleReplayError(tolerateErrors, "Mutation checksum failure at %d in %s", mutationStart, errorContext);
                continue;
            }
            this.replayMutation(this.buffer, serializedSize, reader.getFilePointer(), desc);
        }
        return true;
    }

    void replayMutation(byte[] inputBuffer, int size, long entryLocation, CommitLogDescriptor desc) throws IOException {
        Mutation mutation;
        try (DataInputBuffer bufIn = new DataInputBuffer(inputBuffer, 0, size);){
            mutation = Mutation.serializer.deserialize(bufIn, desc.getMessagingVersion(), SerializationHelper.Flag.LOCAL);
            for (PartitionUpdate upd : mutation.getPartitionUpdates()) {
                upd.validate();
            }
        }
        catch (UnknownColumnFamilyException ex) {
            if (ex.cfId == null) {
                return;
            }
            AtomicInteger i = this.invalidMutations.get(ex.cfId);
            if (i == null) {
                i = new AtomicInteger(1);
                this.invalidMutations.put(ex.cfId, i);
            } else {
                i.incrementAndGet();
            }
            return;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            File f = File.createTempFile("mutation", "dat");
            try (DataOutputStream out = new DataOutputStream(new FileOutputStream(f));){
                out.write(inputBuffer, 0, size);
            }
            CommitLogReplayer.handleReplayError(false, "Unexpected error deserializing mutation; saved to %s.  This may be caused by replaying a mutation against a table with the same name but incompatible schema.  Exception follows: %s", f.getAbsolutePath(), t);
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("replaying mutation for {}.{}: {}", mutation.getKeyspaceName(), mutation.key(), "{" + StringUtils.join(mutation.getPartitionUpdates().iterator(), ", ") + "}");
        }
        this.pendingMutationBytes += (long)size;
        this.futures.offer(mutationInitiator.initiateMutation(mutation, desc.id, size, entryLocation, this));
        while (this.futures.size() > MAX_OUTSTANDING_REPLAY_COUNT || this.pendingMutationBytes > MAX_OUTSTANDING_REPLAY_BYTES || !this.futures.isEmpty() && this.futures.peek().isDone()) {
            this.pendingMutationBytes -= (long)FBUtilities.waitOnFuture(this.futures.poll()).intValue();
        }
    }

    protected boolean pointInTimeExceeded(Mutation fm) {
        long restoreTarget = this.archiver.restorePointInTime;
        for (PartitionUpdate upd : fm.getPartitionUpdates()) {
            if (this.archiver.precision.toMillis(upd.maxTimestamp()) <= restoreTarget) continue;
            return true;
        }
        return false;
    }

    static void handleReplayError(boolean permissible, String message, Object ... messageArgs) throws IOException {
        String msg = String.format(message, messageArgs);
        CommitLogReplayException e = new CommitLogReplayException(msg);
        if (permissible) {
            logger.error("Ignoring commit log replay error likely due to incomplete flush to disk", e);
        } else if (Boolean.getBoolean(IGNORE_REPLAY_ERRORS_PROPERTY)) {
            logger.error("Ignoring commit log replay error", e);
        } else if (!CommitLog.handleCommitError("Failed commit log replay", e)) {
            logger.error("Replay stopped. If you wish to override this error and continue starting the node ignoring commit log replay problems, specify -Dcassandra.commitlog.ignorereplayerrors=true on the command line");
            throw e;
        }
    }

    public static class CommitLogReplayException
    extends IOException {
        public CommitLogReplayException(String message, Throwable cause) {
            super(message, cause);
        }

        public CommitLogReplayException(String message) {
            super(message);
        }
    }

    private static class CustomReplayFilter
    extends ReplayFilter {
        private Multimap<String, String> toReplay;

        public CustomReplayFilter(Multimap<String, String> toReplay) {
            this.toReplay = toReplay;
        }

        @Override
        public Iterable<PartitionUpdate> filter(Mutation mutation) {
            final Collection<String> cfNames = this.toReplay.get(mutation.getKeyspaceName());
            if (cfNames == null) {
                return Collections.emptySet();
            }
            return Iterables.filter(mutation.getPartitionUpdates(), new Predicate<PartitionUpdate>(){

                @Override
                public boolean apply(PartitionUpdate upd) {
                    return cfNames.contains(upd.metadata().cfName);
                }
            });
        }

        @Override
        public boolean includes(CFMetaData metadata) {
            return this.toReplay.containsEntry(metadata.ksName, metadata.cfName);
        }
    }

    private static class AlwaysReplayFilter
    extends ReplayFilter {
        private AlwaysReplayFilter() {
        }

        @Override
        public Iterable<PartitionUpdate> filter(Mutation mutation) {
            return mutation.getPartitionUpdates();
        }

        @Override
        public boolean includes(CFMetaData metadata) {
            return true;
        }
    }

    static abstract class ReplayFilter {
        ReplayFilter() {
        }

        public abstract Iterable<PartitionUpdate> filter(Mutation var1);

        public abstract boolean includes(CFMetaData var1);

        public static ReplayFilter create() {
            if (System.getProperty("cassandra.replayList") == null) {
                return new AlwaysReplayFilter();
            }
            HashMultimap<String, String> toReplay = HashMultimap.create();
            for (String rawPair : System.getProperty("cassandra.replayList").split(",")) {
                String[] pair = StringUtils.split(rawPair.trim(), '.');
                if (pair.length != 2) {
                    throw new IllegalArgumentException("Each table to be replayed must be fully qualified with keyspace name, e.g., 'system.peers'");
                }
                Keyspace ks = Schema.instance.getKeyspaceInstance(pair[0]);
                if (ks == null) {
                    throw new IllegalArgumentException("Unknown keyspace " + pair[0]);
                }
                ColumnFamilyStore cfs = ks.getColumnFamilyStore(pair[1]);
                if (cfs == null) {
                    throw new IllegalArgumentException(String.format("Unknown table %s.%s", pair[0], pair[1]));
                }
                toReplay.put(pair[0], pair[1]);
            }
            return new CustomReplayFilter(toReplay);
        }
    }

    @VisibleForTesting
    public static class MutationInitiator {
        protected Future<Integer> initiateMutation(final Mutation mutation, final long segmentId, int serializedSize, final long entryLocation, final CommitLogReplayer clr) {
            WrappedRunnable runnable = new WrappedRunnable(){

                @Override
                public void runMayThrow() {
                    if (Schema.instance.getKSMetaData(mutation.getKeyspaceName()) == null) {
                        return;
                    }
                    if (clr.pointInTimeExceeded(mutation)) {
                        return;
                    }
                    Keyspace keyspace = Keyspace.open(mutation.getKeyspaceName());
                    Mutation newMutation = null;
                    for (PartitionUpdate update : clr.replayFilter.filter(mutation)) {
                        if (Schema.instance.getCF(update.metadata().cfId) == null) continue;
                        ReplayPosition rp = (ReplayPosition)clr.cfPositions.get(update.metadata().cfId);
                        if (segmentId <= rp.segment && (segmentId != rp.segment || entryLocation <= (long)rp.position)) continue;
                        if (newMutation == null) {
                            newMutation = new Mutation(mutation.getKeyspaceName(), mutation.key());
                        }
                        newMutation.add(update);
                        clr.replayedCount.incrementAndGet();
                    }
                    if (newMutation != null) {
                        assert (!newMutation.isEmpty());
                        try {
                            Uninterruptibles.getUninterruptibly(Keyspace.open(newMutation.getKeyspaceName()).applyFromCommitLog(newMutation));
                        }
                        catch (ExecutionException e) {
                            throw Throwables.propagate(e.getCause());
                        }
                        clr.keyspacesRecovered.add(keyspace);
                    }
                }
            };
            return StageManager.getStage(Stage.MUTATION).submit(runnable, serializedSize);
        }
    }
}

