/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.collect.ImmutableMap;
import java.io.DataInput;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import javax.crypto.Cipher;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Hex;
import org.json.simple.JSONValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDescriptor {
    private static final Logger logger = LoggerFactory.getLogger(HintsDescriptor.class);
    static final int VERSION_30 = 1;
    static final int CURRENT_VERSION = 1;
    static final String COMPRESSION = "compression";
    static final String ENCRYPTION = "encryption";
    static final Pattern pattern = Pattern.compile("^[a-fA-F0-9]{8}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{4}\\-[a-fA-F0-9]{12}\\-(\\d+)\\-(\\d+)\\.hints$");
    final UUID hostId;
    final int version;
    final long timestamp;
    final ImmutableMap<String, Object> parameters;
    final ParameterizedClass compressionConfig;
    private final Cipher cipher;
    private final ICompressor compressor;

    HintsDescriptor(UUID hostId, int version, long timestamp, ImmutableMap<String, Object> parameters) {
        this.hostId = hostId;
        this.version = version;
        this.timestamp = timestamp;
        this.compressionConfig = HintsDescriptor.createCompressionConfig(parameters);
        EncryptionData encryption = HintsDescriptor.createEncryption(parameters);
        if (encryption == null) {
            this.cipher = null;
            this.compressor = null;
        } else {
            if (this.compressionConfig != null) {
                throw new IllegalStateException("a hints file cannot be configured for both compression and encryption");
            }
            this.cipher = encryption.cipher;
            this.compressor = encryption.compressor;
            parameters = encryption.params;
        }
        this.parameters = parameters;
    }

    HintsDescriptor(UUID hostId, long timestamp, ImmutableMap<String, Object> parameters) {
        this(hostId, 1, timestamp, parameters);
    }

    HintsDescriptor(UUID hostId, long timestamp) {
        this(hostId, 1, timestamp, ImmutableMap.of());
    }

    static ParameterizedClass createCompressionConfig(Map<String, Object> params) {
        if (params.containsKey(COMPRESSION)) {
            Map compressorConfig = (Map)params.get(COMPRESSION);
            return new ParameterizedClass((String)compressorConfig.get("class_name"), (Map)compressorConfig.get("parameters"));
        }
        return null;
    }

    static EncryptionData createEncryption(ImmutableMap<String, Object> params) {
        if (params.containsKey(ENCRYPTION)) {
            Map encryptionConfig = (Map)params.get(ENCRYPTION);
            EncryptionContext encryptionContext = EncryptionContext.createFromMap(encryptionConfig, DatabaseDescriptor.getEncryptionContext());
            try {
                Cipher cipher;
                if (encryptionConfig.containsKey("encIV")) {
                    cipher = encryptionContext.getDecryptor();
                } else {
                    cipher = encryptionContext.getEncryptor();
                    ImmutableMap<String, String> encParams = ImmutableMap.builder().putAll(encryptionContext.toHeaderParameters()).put("encIV", Hex.bytesToHex(cipher.getIV())).build();
                    HashMap<String, Object> map = new HashMap<String, Object>(params);
                    map.put(ENCRYPTION, encParams);
                    params = ImmutableMap.builder().putAll(map).build();
                }
                return new EncryptionData(cipher, encryptionContext.getCompressor(), params);
            }
            catch (IOException ioe) {
                logger.warn("failed to create encyption context for hints file. ignoring encryption for hints.", ioe);
                return null;
            }
        }
        return null;
    }

    String fileName() {
        return String.format("%s-%s-%s.hints", this.hostId, this.timestamp, this.version);
    }

    String checksumFileName() {
        return String.format("%s-%s-%s.crc32", this.hostId, this.timestamp, this.version);
    }

    int messagingVersion() {
        return HintsDescriptor.messagingVersion(this.version);
    }

    static int messagingVersion(int hintsVersion) {
        switch (hintsVersion) {
            case 1: {
                return 10;
            }
        }
        throw new AssertionError();
    }

    static boolean isHintFileName(Path path) {
        return pattern.matcher(path.getFileName().toString()).matches();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    static HintsDescriptor readFromFile(Path path) {
        try (RandomAccessFile raf = new RandomAccessFile(path.toFile(), "r");){
            HintsDescriptor hintsDescriptor = HintsDescriptor.deserialize(raf);
            return hintsDescriptor;
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, path.toFile());
        }
    }

    public boolean isCompressed() {
        return this.compressionConfig != null;
    }

    public boolean isEncrypted() {
        return this.cipher != null;
    }

    public ICompressor createCompressor() {
        if (this.isCompressed()) {
            return CompressionParams.createCompressor(this.compressionConfig);
        }
        if (this.isEncrypted()) {
            return this.compressor;
        }
        return null;
    }

    public Cipher getCipher() {
        return this.isEncrypted() ? this.cipher : null;
    }

    public String toString() {
        return MoreObjects.toStringHelper(this).add("hostId", this.hostId).add("version", this.version).add("timestamp", this.timestamp).add("parameters", this.parameters).toString();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof HintsDescriptor)) {
            return false;
        }
        HintsDescriptor hd = (HintsDescriptor)o;
        return Objects.equal(this.hostId, hd.hostId) && Objects.equal(this.version, hd.version) && Objects.equal(this.timestamp, hd.timestamp) && Objects.equal(this.parameters, hd.parameters);
    }

    public int hashCode() {
        return Objects.hashCode(this.hostId, this.version, this.timestamp, this.parameters);
    }

    void serialize(DataOutputPlus out) throws IOException {
        CRC32 crc = new CRC32();
        out.writeInt(this.version);
        FBUtilities.updateChecksumInt(crc, this.version);
        out.writeLong(this.timestamp);
        HintsDescriptor.updateChecksumLong(crc, this.timestamp);
        out.writeLong(this.hostId.getMostSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getMostSignificantBits());
        out.writeLong(this.hostId.getLeastSignificantBits());
        HintsDescriptor.updateChecksumLong(crc, this.hostId.getLeastSignificantBits());
        byte[] paramsBytes = JSONValue.toJSONString(this.parameters).getBytes(StandardCharsets.UTF_8);
        out.writeInt(paramsBytes.length);
        FBUtilities.updateChecksumInt(crc, paramsBytes.length);
        out.writeInt((int)crc.getValue());
        out.write(paramsBytes);
        crc.update(paramsBytes, 0, paramsBytes.length);
        out.writeInt((int)crc.getValue());
    }

    int serializedSize() {
        int size = TypeSizes.sizeof(this.version);
        size += TypeSizes.sizeof(this.timestamp);
        size += TypeSizes.sizeof(this.hostId.getMostSignificantBits());
        size += TypeSizes.sizeof(this.hostId.getLeastSignificantBits());
        byte[] paramsBytes = JSONValue.toJSONString(this.parameters).getBytes(StandardCharsets.UTF_8);
        size += TypeSizes.sizeof(paramsBytes.length);
        size += 4;
        size += paramsBytes.length;
        return size += 4;
    }

    static HintsDescriptor deserialize(DataInput in) throws IOException {
        CRC32 crc = new CRC32();
        int version = in.readInt();
        FBUtilities.updateChecksumInt(crc, version);
        long timestamp = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, timestamp);
        long msb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, msb);
        long lsb = in.readLong();
        HintsDescriptor.updateChecksumLong(crc, lsb);
        UUID hostId = new UUID(msb, lsb);
        int paramsLength = in.readInt();
        FBUtilities.updateChecksumInt(crc, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        byte[] paramsBytes = new byte[paramsLength];
        in.readFully(paramsBytes, 0, paramsLength);
        crc.update(paramsBytes, 0, paramsLength);
        HintsDescriptor.validateCRC(in.readInt(), (int)crc.getValue());
        return new HintsDescriptor(hostId, version, timestamp, HintsDescriptor.decodeJSONBytes(paramsBytes));
    }

    private static ImmutableMap<String, Object> decodeJSONBytes(byte[] bytes) {
        return ImmutableMap.copyOf((Map)JSONValue.parse(new String(bytes, StandardCharsets.UTF_8)));
    }

    private static void updateChecksumLong(CRC32 crc, long value) {
        FBUtilities.updateChecksumInt(crc, (int)(value & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(value >>> 32));
    }

    private static void validateCRC(int expected, int actual) throws IOException {
        if (expected != actual) {
            throw new IOException("Hints Descriptor CRC Mismatch");
        }
    }

    private static final class EncryptionData {
        final Cipher cipher;
        final ICompressor compressor;
        final ImmutableMap<String, Object> params;

        private EncryptionData(Cipher cipher, ICompressor compressor, ImmutableMap<String, Object> params) {
            this.cipher = cipher;
            this.compressor = compressor;
            this.params = params;
        }
    }
}

