package org.hibernate.search.spi;

import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

import org.hibernate.search.engine.SearchFactoryImplementor;
import org.hibernate.search.store.DirectoryProvider;

/**
 * Build context that can be used by some services at initialization
 * 
 * @author Emmanuel Bernard
 */
public interface BuildContext {
	/**
	 * Returns the SessionFactoryImplementor instance. Do not use until after the initialize and/or start method is
	 * fully executed.
	 * Implementations should not cache values provided by the SessionFactoryImplementor but rather access them
	 * each time: when the configuration is dynamically updated, new changes are available through the
	 * SearchFactoryImplementor
	 * For example, prefer
	 * <code>
	 * void method() {
	 *   int size = sfi.getDirectoryProviders().size();
	 * }
	 * </code>
	 * to
	 * <code>
	 * void method() {
	 *   int size = directoryProviders().size();
	 * }
	 * </code>
	 * where directoryProviders is a class variable. 
	 */
	SearchFactoryImplementor getUninitializedSearchFactory();

	String getIndexingStrategy();

	Set<DirectoryProvider<?>> getDirectoryProviders();

	/**
	 * This method cannot be used in intialize methods. start methods can use it though.
	 * @param provider
	 * @return
	 */
	ReentrantLock getDirectoryProviderLock(DirectoryProvider<?> provider);
}
