/*
 * Copyright (c) 2008, 2009, 2011 Oracle, Inc. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.  The Eclipse Public License is available
 * at http://www.eclipse.org/legal/epl-v10.html and the Eclipse Distribution License
 * is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package javax.persistence;

import javax.persistence.metamodel.Attribute;
import java.util.List;

/**
 * This type represents a AttributeNode of an EntityGraph that corresponds to a Managed Type.  Using this class, an
 * entity subgraph can be embedded within an EntityGraph.
 *
 * @param <T> The type of the attribute.
 *
 * @since JPA 2.1
 */
public interface Subgraph<T> extends AttributeNode<T> {
	/**
	 * Add one or more attribute nodes to the entity graph.
	 * @param attributeName name of the attribute
	 * @throws IllegalArgumentException if the attribute is not an attribute of this managed type.
	 * @throws IllegalStateException if the EntityGraph has been statically defined
	 */
	public void addAttributeNodes(String ... attributeName);

	/**
	 * Add one or more attribute nodes to the entity graph.
	 * @param attribute attribute
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public void addAttributeNodes(Attribute<T, ?> ... attribute);

	/**
	 * Add a node to the graph that corresponds to a managed
	 * type. This allows for construction of multi-node entity graphs
	 * that include related managed types.
	 * @param attribute attribute
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if the EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addSubgraph(Attribute<T, X> attribute);

	/**
	 * Add a node to the graph that corresponds to a managed
	 * type with inheritance. This allows for multiple subclass
	 * subgraphs to be defined for this node of the entity
	 * graph. Subclass subgraphs will automatically include the
	 * specified attributes of superclass subgraphs
	 * @param attribute attribute
	 * @param type entity subclass
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<? extends X> addSubgraph(Attribute<T, X> attribute, Class<? extends X> type);

	/**
	 * Add a node to the graph that corresponds to a managed
	 * type. This allows for construction of multi-node entity graphs
	 * that include related managed types.
	 * @param attributeName name of the attribute
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute is not an attribute of this managed type.
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addSubgraph(String attributeName);

	/**
	 * Add a node to the graph that corresponds to a managed
	 * type with inheritance. This allows for multiple subclass
	 * subgraphs to be defined for this node of the entity
	 * graph. Subclass subgraphs will automatically include the
	 * specified attributes of superclass subgraphs
	 * @param attributeName name of the attribute
	 * @param type entity subclass
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute is not an attribute of this managed type.
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addSubgraph(String attributeName, Class<X> type);

	/**
	 * Add a node to the graph that corresponds to a map key
	 * that is a managed type. This allows for construction of
	 * multinode entity graphs that include related managed types.
	 * @param attribute attribute
	 * @return subgraph for the key attribute
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type entity
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addKeySubgraph(Attribute<T, X> attribute);

	/**
	 * Add a node to the graph that corresponds to a map key
	 * that is a managed type with inheritance. This allows for
	 * construction of multi-node entity graphs that include related
	 * managed types. Subclass subgraphs will automatically include
	 * the specified attributes of superclass subgraphs
	 * @param attribute attribute
	 * @param type entity subclass
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type entity
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<? extends X> addKeySubgraph(Attribute<T, X> attribute, Class<? extends X> type);

	/**
	 * Add a node to the graph that corresponds to a map key
	 * that is a managed type. This allows for construction of
	 * multi-node entity graphs that include related managed types.
	 * @param attributeName name of the attribute
	 * @return subgraph for the key attribute
	 * @throws IllegalArgumentException if the attribute is not an attribute of this entity.
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addKeySubgraph(String attributeName);

	/**
	 * Add a node to the graph that corresponds to a map key
	 * that is a managed type with inheritance. This allows for
	 * construction of multi-node entity graphs that include related
	 * managed types. Subclass subgraphs will include the specified
	 * attributes of superclass subgraphs
	 * @param attributeName name of the attribute
	 * @param type entity subclass
	 * @return subgraph for the attribute
	 * @throws IllegalArgumentException if the attribute is not an attribute of this entity.
	 * @throws IllegalArgumentException if the attribute's target type is not a managed type
	 * @throws IllegalStateException if this EntityGraph has been statically defined
	 */
	public <X> Subgraph<X> addKeySubgraph(String attributeName, Class<X> type);

	/**
	 * Return the attribute nodes corresponding to the attributes of
	 * this managed type that are included in the subgraph.
	 * @return list of attribute nodes included in the subgraph
	 */
	public List<AttributeNode<?>> getAttributeNodes();

	/**
	 * Return the type of for which this subgraph was defined.
	 * @return managed type referenced by the subgraph
	 */
	public Class<T> getClassType();

}
