/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.backend.elasticsearch.work.execution.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.hibernate.search.backend.elasticsearch.orchestration.impl.ElasticsearchSerialWorkOrchestrator;
import org.hibernate.search.backend.elasticsearch.work.builder.factory.impl.ElasticsearchWorkBuilderFactory;
import org.hibernate.search.backend.elasticsearch.work.impl.SingleDocumentIndexingWork;
import org.hibernate.search.engine.backend.common.spi.EntityReferenceFactory;
import org.hibernate.search.engine.backend.work.execution.DocumentRefreshStrategy;
import org.hibernate.search.engine.backend.work.execution.spi.IndexIndexingPlan;
import org.hibernate.search.engine.backend.work.execution.spi.DocumentContributor;
import org.hibernate.search.engine.backend.work.execution.spi.DocumentReferenceProvider;
import org.hibernate.search.engine.backend.session.spi.BackendSessionContext;
import org.hibernate.search.engine.backend.work.execution.spi.IndexIndexingPlanExecutionReport;

import com.google.gson.JsonObject;



public class ElasticsearchIndexIndexingPlan<R> implements IndexIndexingPlan<R> {

	private final ElasticsearchWorkBuilderFactory builderFactory;
	private final ElasticsearchSerialWorkOrchestrator orchestrator;
	private final WorkExecutionIndexManagerContext indexManagerContext;
	private final String tenantId;
	private final EntityReferenceFactory<R> entityReferenceFactory;
	private final DocumentRefreshStrategy refreshStrategy;

	private final List<SingleDocumentIndexingWork> works = new ArrayList<>();

	public ElasticsearchIndexIndexingPlan(ElasticsearchWorkBuilderFactory builderFactory,
			ElasticsearchSerialWorkOrchestrator orchestrator,
			WorkExecutionIndexManagerContext indexManagerContext,
			BackendSessionContext sessionContext,
			EntityReferenceFactory<R> entityReferenceFactory,
			DocumentRefreshStrategy refreshStrategy) {
		this.builderFactory = builderFactory;
		this.orchestrator = orchestrator;
		this.indexManagerContext = indexManagerContext;
		this.tenantId = sessionContext.tenantIdentifier();
		this.entityReferenceFactory = entityReferenceFactory;
		this.refreshStrategy = refreshStrategy;
	}

	@Override
	public void add(DocumentReferenceProvider referenceProvider,
			DocumentContributor documentContributor) {
		index( referenceProvider, documentContributor );
	}

	@Override
	public void update(DocumentReferenceProvider referenceProvider,
			DocumentContributor documentContributor) {
		index( referenceProvider, documentContributor );
	}

	@Override
	public void delete(DocumentReferenceProvider referenceProvider) {
		String elasticsearchId = indexManagerContext.toElasticsearchId( tenantId, referenceProvider.identifier() );
		String routingKey = referenceProvider.routingKey();

		collect(
				builderFactory.delete(
						indexManagerContext.getMappedTypeName(), referenceProvider.entityIdentifier(),
						indexManagerContext.getElasticsearchIndexWriteName(),
						elasticsearchId, routingKey
				)
						.refresh( refreshStrategy )
						.build()
		);
	}

	@Override
	public void process() {
		/*
		 * Nothing to do: we can't execute anything more
		 * without sending a request to the cluster.
		 */
	}

	@Override
	public CompletableFuture<IndexIndexingPlanExecutionReport<R>> executeAndReport() {
		try {
			ElasticsearchIndexIndexingPlanExecution<R> execution = new ElasticsearchIndexIndexingPlanExecution<>(
					orchestrator, entityReferenceFactory,
					new ArrayList<>( works ) // Copy the list, as we're going to clear it below
			);
			return execution.execute();
		}
		finally {
			works.clear();
		}
	}

	@Override
	public void discard() {
		works.clear();
	}

	private void index(DocumentReferenceProvider referenceProvider,
			DocumentContributor documentContributor) {
		String id = referenceProvider.identifier();
		String elasticsearchId = indexManagerContext.toElasticsearchId( tenantId, id );
		String routingKey = referenceProvider.routingKey();

		JsonObject document = indexManagerContext.createDocument( tenantId, id, documentContributor );

		collect(
				builderFactory.index(
						indexManagerContext.getMappedTypeName(), referenceProvider.entityIdentifier(),
						indexManagerContext.getElasticsearchIndexWriteName(),
						elasticsearchId, routingKey, document
				)
						.refresh( refreshStrategy )
						.build()
		);
	}

	private void collect(SingleDocumentIndexingWork work) {
		works.add( work );
	}

}
