/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.backend.elasticsearch.lowlevel.index.settings.impl;

import java.util.HashMap;
import java.util.Map;

import org.hibernate.search.backend.elasticsearch.gson.impl.SerializeExtraProperties;
import org.hibernate.search.backend.elasticsearch.lowlevel.index.analysis.impl.AnalyzerDefinition;
import org.hibernate.search.backend.elasticsearch.lowlevel.index.analysis.impl.CharFilterDefinition;
import org.hibernate.search.backend.elasticsearch.lowlevel.index.analysis.impl.NormalizerDefinition;
import org.hibernate.search.backend.elasticsearch.lowlevel.index.analysis.impl.TokenFilterDefinition;
import org.hibernate.search.backend.elasticsearch.lowlevel.index.analysis.impl.TokenizerDefinition;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.annotations.SerializedName;

@JsonAdapter(AnalysisJsonAdapterFactory.class)
public class Analysis {

	@SerializedName("analyzer")
	private Map<String, AnalyzerDefinition> analyzers;

	@SerializedName("normalizer")
	private Map<String, NormalizerDefinition> normalizers;

	@SerializedName("tokenizer")
	private Map<String, TokenizerDefinition> tokenizers;

	@SerializedName("filter")
	private Map<String, TokenFilterDefinition> tokenFilters;

	@SerializedName("char_filter")
	private Map<String, CharFilterDefinition> charFilters;

	@SerializeExtraProperties
	private Map<String, JsonElement> extraAttributes;

	public boolean isEmpty() {
		return ! hasContent( analyzers, normalizers, tokenizers, tokenFilters, charFilters );
	}

	private boolean hasContent(Map<?, ?> ... maps) {
		for ( Map<?, ?> map : maps ) {
			if ( map != null && !map.isEmpty() ) {
				return true;
			}
		}
		return false;
	}

	public Map<String, AnalyzerDefinition> getAnalyzers() {
		return analyzers;
	}

	public void setAnalyzers(Map<String, AnalyzerDefinition> analyzers) {
		this.analyzers = analyzers;
	}

	public Map<String, NormalizerDefinition> getNormalizers() {
		return normalizers;
	}

	public void setNormalizers(Map<String, NormalizerDefinition> normalizers) {
		this.normalizers = normalizers;
	}

	public Map<String, TokenizerDefinition> getTokenizers() {
		return tokenizers;
	}

	public void setTokenizers(Map<String, TokenizerDefinition> tokenizers) {
		this.tokenizers = tokenizers;
	}

	public Map<String, TokenFilterDefinition> getTokenFilters() {
		return tokenFilters;
	}

	public void setTokenFilters(Map<String, TokenFilterDefinition> tokenFilters) {
		this.tokenFilters = tokenFilters;
	}

	public Map<String, CharFilterDefinition> getCharFilters() {
		return charFilters;
	}

	public void setCharFilters(Map<String, CharFilterDefinition> charFilters) {
		this.charFilters = charFilters;
	}

	public Map<String, JsonElement> getExtraAttributes() {
		return extraAttributes;
	}

	@Override
	public String toString() {
		return new GsonBuilder().setPrettyPrinting().create().toJson( this );
	}

	/**
	 * Merge these (custom) analysis settings with analysis settings generated by Hibernate Search.
	 * <p>
	 * Any conflict of definition will be solved in favor of {@code this}.
	 *
	 * @param other The other analysis settings
	 */
	public void merge(Analysis other) {
		if ( other == null ) {
			// nothing to do
			return;
		}

		analyzers = merge( analyzers, other.analyzers );
		normalizers = merge( normalizers, other.normalizers );
		tokenizers = merge( tokenizers, other.tokenizers );
		tokenFilters = merge( tokenFilters, other.tokenFilters );
		charFilters = merge( charFilters, other.charFilters );

		if ( extraAttributes == null ) {
			extraAttributes = other.extraAttributes;
		}
	}

	private static <K> Map<String, K> merge(Map<String, K> originalImmutableMap, Map<String, K> otherMap) {
		if ( otherMap == null || otherMap.isEmpty() ) {
			return originalImmutableMap;
		}

		if ( originalImmutableMap == null || originalImmutableMap.isEmpty() ) {
			return otherMap;
		}

		// let's merge
		HashMap<String, K> result = new HashMap<>( otherMap );
		result.putAll( originalImmutableMap );
		return result;
	}
}
