/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.backend.elasticsearch.lowlevel.index.mapping.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.gson.annotations.JsonAdapter;
import com.google.gson.annotations.SerializedName;

/**
 * An object representing Elasticsearch type mappings.
 *
 * See https://www.elastic.co/guide/en/elasticsearch/reference/current/mapping.html#mapping-type
 */
/*
 * CAUTION: JSON serialization is controlled by a specific adapter, which must be
 * updated whenever fields of this class are added, renamed or removed.
 */
@JsonAdapter(RootTypeMappingJsonAdapterFactory.class)
public class RootTypeMapping extends AbstractTypeMapping {

	@SerializedName("_routing")
	private RoutingType routing;

	@SerializedName("dynamic_templates")
	private List<NamedDynamicTemplate> dynamicTemplates;

	public RoutingType getRouting() {
		return routing;
	}

	public void setRouting(RoutingType routing) {
		this.routing = routing;
	}

	public List<NamedDynamicTemplate> getDynamicTemplates() {
		return dynamicTemplates == null ? null : Collections.unmodifiableList( dynamicTemplates );
	}

	private List<NamedDynamicTemplate> getInitializedDynamicTemplates() {
		if ( dynamicTemplates == null ) {
			dynamicTemplates = new ArrayList<>();
		}
		return dynamicTemplates;
	}

	public void addDynamicTemplate(NamedDynamicTemplate template) {
		getInitializedDynamicTemplates().add( template );
	}

	/**
	 * Merge this mapping with another mapping generated by Hibernate Search:
	 *
	 * <ol>
	 * <li>Values for {@code dynamic}/{@code _routing}/{@code dynamic_templates} mapping parameters will be taken
	 * from the {@code other} unless {@code this} provides a corresponding non-null value.</li>
	 * <li>The mapping parameter {@code properties} will be merged, using properties defined in both {@code this} and {@code other}.</li>
	 * <li>Mapping parameters other than {@code dynamic} and {@code properties} will be those from {@code this};
	 * those from {@code other} will be ignored.</li>
	 * <li>If a property is defined on both sides, it will be merged recursively as described above.</li>
	 * </ol>
	 * @param other The other mapping.
	 */
	@Override
	public void merge(AbstractTypeMapping other) {
		if ( other == null ) {
			// nothing to do
			return;
		}
		// let's merge the properties as usual:
		super.merge( other );

		if ( other instanceof RootTypeMapping ) {
			RootTypeMapping otherRoot = (RootTypeMapping) other;

			// now let's see if we have any dynamic templates, if not get them from the other mapping:
			if ( ( this.dynamicTemplates == null || this.dynamicTemplates.isEmpty() )
					&& !( otherRoot.dynamicTemplates == null || otherRoot.dynamicTemplates.isEmpty() ) ) {
				this.getInitializedDynamicTemplates().addAll( otherRoot.dynamicTemplates );
			}

			// same for routing:
			if ( this.routing == null ) {
				this.routing = otherRoot.routing;
			}
		}
	}
}
