/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.backend.lucene.lowlevel.index.impl;

import org.hibernate.search.backend.lucene.cfg.LuceneIndexSettings;
import org.hibernate.search.backend.lucene.lowlevel.directory.spi.DirectoryHolder;
import org.hibernate.search.backend.lucene.lowlevel.directory.spi.DirectoryProvider;
import org.hibernate.search.backend.lucene.lowlevel.reader.impl.IndexReaderProvider;
import org.hibernate.search.backend.lucene.lowlevel.reader.impl.NearRealTimeIndexReaderProvider;
import org.hibernate.search.backend.lucene.lowlevel.writer.impl.IndexWriterProvider;
import org.hibernate.search.backend.lucene.search.timeout.spi.TimingSource;
import org.hibernate.search.engine.cfg.spi.ConfigurationProperty;
import org.hibernate.search.engine.cfg.spi.ConfigurationPropertySource;
import org.hibernate.search.engine.environment.thread.spi.ThreadPoolProvider;
import org.hibernate.search.engine.reporting.FailureHandler;
import org.hibernate.search.util.common.reporting.EventContext;

import org.apache.lucene.analysis.Analyzer;

public class NearRealTimeIOStrategy extends IOStrategy {

	private static final ConfigurationProperty<Integer> COMMIT_INTERVAL =
			ConfigurationProperty.forKey( LuceneIndexSettings.IO_COMMIT_INTERVAL )
					.asInteger()
					.withDefault( LuceneIndexSettings.Defaults.IO_COMMIT_INTERVAL )
					.build();

	private static final ConfigurationProperty<Integer> REFRESH_INTERVAL =
			ConfigurationProperty.forKey( LuceneIndexSettings.IO_REFRESH_INTERVAL )
					.asInteger()
					.withDefault( LuceneIndexSettings.Defaults.IO_REFRESH_INTERVAL )
					.build();

	public static NearRealTimeIOStrategy create(ConfigurationPropertySource propertySource,
			DirectoryProvider directoryProvider, TimingSource timingSource,
			ThreadPoolProvider threadPoolProvider, FailureHandler failureHandler) {
		int commitInterval = COMMIT_INTERVAL.get( propertySource );
		int refreshInterval = REFRESH_INTERVAL.get( propertySource );
		return new NearRealTimeIOStrategy(
				directoryProvider, timingSource, commitInterval, refreshInterval,
				threadPoolProvider, failureHandler
		);
	}

	private final TimingSource timingSource;
	private final int commitInterval;
	private final int refreshInterval;

	private NearRealTimeIOStrategy(DirectoryProvider directoryProvider,
			TimingSource timingSource, int commitInterval, int refreshInterval,
			ThreadPoolProvider threadPoolProvider,
			FailureHandler failureHandler) {
		super( directoryProvider, threadPoolProvider, failureHandler );
		this.timingSource = timingSource;
		this.commitInterval = commitInterval;
		this.refreshInterval = refreshInterval;
	}

	@Override
	IndexWriterProvider createIndexWriterProvider(String indexName, EventContext eventContext, Analyzer analyzer,
			DirectoryHolder directoryHolder) {
		if ( commitInterval != 0 ) {
			timingSource.ensureInitialized();
		}
		return new IndexWriterProvider(
				indexName, eventContext,
				directoryHolder, analyzer,
				timingSource, commitInterval, threadPoolProvider.getThreadProvider(),
				failureHandler
		);
	}

	@Override
	IndexReaderProvider createIndexReaderProvider(DirectoryHolder directoryHolder,
			IndexWriterProvider indexWriterProvider) {
		if ( refreshInterval != 0 ) {
			timingSource.ensureInitialized();
		}
		return new NearRealTimeIndexReaderProvider( indexWriterProvider, timingSource, refreshInterval );
	}

}
