/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.engine.backend.document.model.spi;

import org.hibernate.search.engine.backend.common.spi.FieldPaths;
import org.hibernate.search.engine.reporting.spi.EventContexts;
import org.hibernate.search.engine.search.common.spi.SearchIndexNodeTypeContext;
import org.hibernate.search.engine.search.common.spi.SearchIndexScope;
import org.hibernate.search.util.common.reporting.EventContext;

public abstract class AbstractIndexField<
				S extends AbstractIndexField<S, SC, ?, C>,
				SC extends SearchIndexScope<?>,
				NT extends SearchIndexNodeTypeContext<SC, ? super S>,
				C extends IndexCompositeNode<SC, ?, ?>
		>
		extends AbstractIndexNode<S, SC, NT>
		implements IndexField<SC, C> {
	protected final C parent;
	protected final String absolutePath;
	protected final String[] absolutePathComponents;
	protected final String relativeName;
	protected final IndexFieldInclusion inclusion;
	protected final boolean multiValued;
	protected final boolean multiValuedInRoot;

	public AbstractIndexField(C parent, String relativeFieldName, NT type, IndexFieldInclusion inclusion,
			boolean multiValued) {
		super( type );
		this.parent = parent;
		this.absolutePath = parent.absolutePath( relativeFieldName );
		this.absolutePathComponents = FieldPaths.split( absolutePath );
		this.relativeName = relativeFieldName;
		this.inclusion = inclusion;
		this.multiValued = multiValued;
		this.multiValuedInRoot = multiValued || parent.multiValuedInRoot();
	}

	@Override
	public String toString() {
		return getClass().getSimpleName() + "[absolutePath=" + absolutePath + ", type=" + type + "]";
	}

	@Override
	public final EventContext relativeEventContext() {
		return EventContexts.fromIndexFieldAbsolutePath( absolutePath );
	}

	@Override
	public final C parent() {
		return parent;
	}

	@Override
	public final String absolutePath() {
		return absolutePath;
	}

	@Override
	public final String[] absolutePathComponents() {
		return absolutePathComponents;
	}

	@Override
	public final String relativeName() {
		return relativeName;
	}

	@Override
	public final IndexFieldInclusion inclusion() {
		return inclusion;
	}

	@Override
	public final boolean multiValued() {
		return multiValued;
	}

	@Override
	public final boolean multiValuedInRoot() {
		return multiValuedInRoot;
	}

}
