/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.mapper.pojo.mapping.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.hibernate.search.mapper.pojo.model.spi.PojoRawTypeIdentifier;
import org.hibernate.search.mapper.pojo.model.spi.PojoRawTypeModel;
import org.hibernate.search.mapper.pojo.work.impl.PojoWorkContainedTypeContextProvider;
import org.hibernate.search.util.common.impl.Closer;

public class PojoContainedTypeManagerContainer
		implements PojoWorkContainedTypeContextProvider {

	public static Builder builder() {
		return new Builder();
	}

	private final Map<PojoRawTypeIdentifier<?>, PojoContainedTypeManager<?, ?>> byExactType;
	private final Map<String, PojoContainedTypeManager<?, ?>> byEntityName;
	private final Set<PojoContainedTypeManager<?, ?>> all;

	private PojoContainedTypeManagerContainer(Builder builder) {
		this.byExactType = new HashMap<>( builder.byExactType );
		this.byEntityName = new LinkedHashMap<>( builder.byExactType.size() );
		for ( PojoContainedTypeManager<?, ?> typeManager : builder.byExactType.values() ) {
			byEntityName.put( typeManager.entityName(), typeManager );
		}
		this.all = Collections.unmodifiableSet( new LinkedHashSet<>( byExactType.values() ) );
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> Optional<? extends PojoContainedTypeManager<?, E>> forExactType(
			PojoRawTypeIdentifier<E> typeIdentifier) {
		return Optional.ofNullable( (PojoContainedTypeManager<?, E>) byExactType.get( typeIdentifier ) );
	}

	@Override
	public Optional<? extends PojoContainedTypeManager<?, ?>> forEntityName(String entityName) {
		return Optional.ofNullable( (PojoContainedTypeManager<?, ?>) byEntityName.get( entityName ) );
	}

	Set<PojoContainedTypeManager<?, ?>> all() {
		return all;
	}

	public static class Builder {

		// Use a LinkedHashMap for deterministic iteration
		private final Map<PojoRawTypeIdentifier<?>, PojoContainedTypeManager<?, ?>> byExactType = new LinkedHashMap<>();

		private Builder() {
		}

		public <E> void add(PojoRawTypeModel<E> typeModel, PojoContainedTypeManager<?, E> typeManager) {
			byExactType.put( typeModel.typeIdentifier(), typeManager );
		}

		public void closeOnFailure() {
			try ( Closer<RuntimeException> closer = new Closer<>() ) {
				closer.pushAll( PojoContainedTypeManager::close, byExactType.values() );
			}
		}

		public PojoContainedTypeManagerContainer build() {
			return new PojoContainedTypeManagerContainer( this );
		}
	}

}
