/*
 * Decompiled with CFR 0.152.
 */
package org.hortonmachine.gears.io.las.databases;

import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.net.URL;
import java.nio.ByteBuffer;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import oms3.annotations.Author;
import oms3.annotations.Description;
import oms3.annotations.Execute;
import oms3.annotations.Finalize;
import oms3.annotations.In;
import oms3.annotations.Keywords;
import oms3.annotations.Label;
import oms3.annotations.License;
import oms3.annotations.Name;
import oms3.annotations.Status;
import oms3.annotations.UI;
import org.geotools.coverage.grid.GridCoordinates2D;
import org.geotools.coverage.grid.GridCoverage2D;
import org.geotools.coverage.grid.GridGeometry2D;
import org.geotools.coverage.grid.io.AbstractGridFormat;
import org.geotools.coverage.grid.io.GridFormatFinder;
import org.geotools.gce.imagemosaic.ImageMosaicReader;
import org.geotools.geometry.DirectPosition2D;
import org.geotools.geometry.Envelope2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.geometry.jts.ReferencedEnvelope3D;
import org.hortonmachine.dbs.compat.ASpatialDb;
import org.hortonmachine.dbs.compat.EDb;
import org.hortonmachine.gears.io.las.core.ALasReader;
import org.hortonmachine.gears.io.las.core.ILasHeader;
import org.hortonmachine.gears.io.las.core.LasRecord;
import org.hortonmachine.gears.io.las.databases.LasCell;
import org.hortonmachine.gears.io.las.databases.LasCellsTable;
import org.hortonmachine.gears.io.las.databases.LasLevel;
import org.hortonmachine.gears.io.las.databases.LasLevelsTable;
import org.hortonmachine.gears.io.las.databases.LasSource;
import org.hortonmachine.gears.io.las.databases.LasSourcesTable;
import org.hortonmachine.gears.libs.exceptions.ModelsIllegalargumentException;
import org.hortonmachine.gears.libs.modules.HMModel;
import org.hortonmachine.gears.modules.utils.fileiterator.OmsFileIterator;
import org.hortonmachine.gears.utils.CrsUtilities;
import org.hortonmachine.gears.utils.coverage.CoverageUtilities;
import org.hortonmachine.gears.utils.files.FileUtilities;
import org.hortonmachine.gears.utils.geometry.GeometryUtilities;
import org.hortonmachine.gears.utils.math.NumericsUtilities;
import org.hortonmachine.gears.utils.time.EggClock;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Polygon;
import org.opengis.coverage.PointOutsideCoverageException;
import org.opengis.geometry.DirectPosition;
import org.opengis.parameter.GeneralParameterValue;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

@Description(value="Inserts las files into a spatialite database.")
@Author(name="Andrea Antonello", contact="www.hydrologis.com")
@Keywords(value="las, lidar, spatialite")
@Label(value="Lesto/utilities")
@Name(value="spatialitelaswriter")
@Status(value=5)
@License(value="http://www.gnu.org/licenses/gpl-3.0.html")
public class DatabaseLasWriter
extends HMModel {
    @Description(value="The folder containing the las files to index.")
    @UI(value="infolder")
    @In
    public String inFolder;
    @Description(value="The output database path.")
    @UI(value="infile")
    @In
    public String inDatabasePath;
    @Description(value="The database to use")
    @UI(value="combo:SPATIALITE,H2GIS")
    @In
    public String pDbType = EDb.SPATIALITE.name();
    @Description(value="The optional image mosaic of ortophoto to take the color from (has to be 3-band).")
    @UI(value="infile_vector")
    @In
    public String inOrtophoto;
    @Description(value="The optional code defining the target coordinate reference system. This is needed only if the file has no prj file. If set, it will be used over the prj file.")
    @UI(value="crs")
    @In
    public String pCode;
    @Description(value="The size of the cells into which to split the las file for indexing (in units defined by the projection).")
    @In
    public double pCellsize = 3.0;
    @Description(value="Number of data summary levels to add.")
    @In
    public int pLevels = 2;
    @Description(value="The size multiplication factor to use for subsequent levels.")
    @In
    public int pFactor = 5;
    @Description(value="Flag to define if empty cells should also be written into the database.")
    @In
    public boolean doEmptyCells = false;
    @Description(value="Flag to define if the spatial index creation should be ignored (will not create levels).")
    @In
    public boolean doAvoidIndex = false;
    @Description(value="Optional las list names to process only those (inside las folder).")
    @In
    public List<String> inLasNames;
    private CoordinateReferenceSystem crs;
    private int srid = -9999;
    private double ortoXRes;
    private double ortoYRes;
    private ImageMosaicReader ortoReader;
    public boolean doVerbose = true;
    private static final String INTERRUPTED_BY_USER = "Interrupted by user.";

    @Execute
    public void process() throws Exception {
        File ortoFile;
        this.checkNull(this.inFolder, this.inDatabasePath);
        if (this.pCellsize <= 0.0) {
            throw new ModelsIllegalargumentException("The cell size parameter needs to be > 0.", this);
        }
        if (!new File(this.inFolder).exists()) {
            throw new ModelsIllegalargumentException("The inFolder parameter has to be valid.", this);
        }
        if (this.doAvoidIndex) {
            this.pLevels = 0;
        }
        if (this.inOrtophoto != null && (ortoFile = new File(this.inOrtophoto)).exists()) {
            URL imageMosaicUrl = ortoFile.toURI().toURL();
            AbstractGridFormat imageMosaicFormat = GridFormatFinder.findFormat((Object)imageMosaicUrl);
            this.ortoReader = (ImageMosaicReader)imageMosaicFormat.getReader((Object)imageMosaicUrl);
            File propertiesFile = FileUtilities.substituteExtention(ortoFile, "properties");
            LinkedHashMap<String, String> propertiesMap = FileUtilities.readFileToHashMap(propertiesFile.getAbsolutePath(), null, false);
            String xyREs = (String)((HashMap)propertiesMap).get("Levels");
            String[] split = xyREs.split(",");
            this.ortoXRes = Double.parseDouble(split[0]);
            this.ortoYRes = Double.parseDouble(split[1]);
        }
        EDb edb = EDb.SPATIALITE;
        if (this.pDbType.equals(EDb.H2GIS.name())) {
            edb = EDb.H2GIS;
        }
        try (ASpatialDb spatialiteDb = edb.getSpatialDb();){
            List<Object> filesList;
            boolean existed = spatialiteDb.open(this.inDatabasePath);
            if (!existed) {
                this.pm.beginTask("Create new spatialite database...", -1);
                spatialiteDb.initSpatialMetadata(null);
                this.pm.done();
            }
            try {
                if (this.pCode != null) {
                    this.crs = CrsUtilities.getCrsFromEpsg(this.pCode, null);
                } else {
                    File folderFile = new File(this.inFolder);
                    File[] prjFiles = folderFile.listFiles(new FilenameFilter(){

                        @Override
                        public boolean accept(File dir, String name) {
                            return name.toLowerCase().endsWith(".prj");
                        }
                    });
                    if (prjFiles != null && prjFiles.length > 0) {
                        this.crs = CrsUtilities.readProjectionFile(prjFiles[0].getAbsolutePath(), null);
                        this.pCode = CrsUtilities.getCodeFromCrs(this.crs);
                    }
                }
                if (this.pCode != null) {
                    this.pCode = this.pCode.replaceFirst("EPSG:", "");
                    this.srid = Integer.parseInt(this.pCode);
                }
            }
            catch (Exception e1) {
                throw new ModelsIllegalargumentException("An error occurred while reading the projection definition: " + e1.getLocalizedMessage(), this);
            }
            if (this.srid == -9999) {
                this.srid = 4326;
                this.pm.errorMessage("No crs has been defined. Setting it to 4326 by default.");
            }
            LasSourcesTable.createTable(spatialiteDb, this.srid, this.doAvoidIndex);
            LasCellsTable.createTable(spatialiteDb, this.srid, this.doAvoidIndex);
            this.pm.message("Las files to be added to the index:");
            if (this.inLasNames == null) {
                OmsFileIterator iter = new OmsFileIterator();
                iter.inFolder = this.inFolder;
                iter.fileFilter = new FileFilter(){

                    @Override
                    public boolean accept(File file) {
                        boolean isLas;
                        String name = file.getName();
                        boolean bl = isLas = name.toLowerCase().endsWith(".las") && !name.toLowerCase().endsWith("indexed.las");
                        if (isLas) {
                            DatabaseLasWriter.this.pm.message("   " + name);
                        }
                        return isLas;
                    }
                };
                iter.process();
                filesList = iter.filesList;
            } else {
                filesList = new ArrayList();
                for (String lasName : this.inLasNames) {
                    File lasFile = new File(this.inFolder + File.separator + lasName);
                    filesList.add(lasFile);
                }
            }
            List<LasSource> lasSources = LasSourcesTable.getLasSources(spatialiteDb);
            ArrayList<String> existingLasSourcesNames = new ArrayList<String>();
            for (LasSource lasSource : lasSources) {
                existingLasSourcesNames.add(lasSource.name);
            }
            for (File file : filesList) {
                ReferencedEnvelope3D envelope;
                if (this.pm.isCanceled()) {
                    return;
                }
                String lasName = FileUtilities.getNameWithoutExtention(file);
                if (existingLasSourcesNames.contains(lasName)) {
                    this.pm.errorMessage("Not inserting already existing file in database: " + lasName);
                    continue;
                }
                try (ALasReader reader = ALasReader.getReader(file, this.crs);){
                    reader.open();
                    ILasHeader header = reader.getHeader();
                    envelope = header.getDataEnvelope();
                }
                Polygon polygon = GeometryUtilities.createPolygonFromEnvelope((Envelope)envelope);
                GridCoverage2D ortoGC = null;
                if (this.ortoReader != null) {
                    double west = envelope.getMinX();
                    double east = envelope.getMaxX();
                    double south = envelope.getMinY();
                    double north = envelope.getMaxY();
                    GeneralParameterValue[] readGeneralParameterValues = CoverageUtilities.createGridGeometryGeneralParameter(this.ortoXRes, this.ortoYRes, north, south, east, west, this.crs);
                    ortoGC = this.ortoReader.read(readGeneralParameterValues);
                }
                long id = LasSourcesTable.insertLasSource(spatialiteDb, this.srid, this.pLevels, this.pCellsize, this.pFactor, polygon, lasName, envelope.getMinZ(), envelope.getMaxZ(), 0.0, 0.0);
                this.processFile(spatialiteDb, file, id, ortoGC);
            }
        }
    }

    private void processFile(final ASpatialDb spatialiteDb, File file, long sourceID, GridCoverage2D ortoGC) throws Exception {
        String name = file.getName();
        this.pm.message("Processing file: " + name);
        try (ALasReader reader = ALasReader.getReader(file, this.crs);){
            reader.open();
            ILasHeader header = reader.getHeader();
            long recordsCount = header.getRecordsCount();
            if (recordsCount == 0L) {
                this.pm.errorMessage("No points found in: " + name);
                return;
            }
            ReferencedEnvelope3D envelope = header.getDataEnvelope();
            ReferencedEnvelope env2d = new ReferencedEnvelope((ReferencedEnvelope)envelope);
            Envelope2D e = new Envelope2D((org.opengis.geometry.Envelope)env2d);
            double north = e.getMaxY();
            double south = e.getMinY();
            double east = e.getMaxX();
            double west = e.getMinX();
            double[] xRanges = NumericsUtilities.range2Bins(west, east, this.pCellsize, false);
            double[] yRanges = NumericsUtilities.range2Bins(south, north, this.pCellsize, false);
            int cols = xRanges.length - 1;
            int rows = yRanges.length - 1;
            int tilesCount = cols * rows;
            this.pm.message("Splitting " + name + " into " + tilesCount + " tiles.");
            GridGeometry2D gridGeometry = CoverageUtilities.gridGeometryFromRegionValues(north, south, east, west, cols, rows, reader.getHeader().getCrs());
            ArrayList[][] dotOnMatrixXY = new ArrayList[cols][rows];
            LasCell[][] lasCellsOnMatrixXY = new LasCell[cols][rows];
            if (this.doVerbose) {
                this.pm.beginTask("Sorting points for " + name, (int)recordsCount);
            }
            long readCount = 0L;
            int minIntens = Short.MAX_VALUE;
            int maxIntens = -32767;
            while (reader.hasNextPoint()) {
                int y;
                LasRecord dot = reader.getNextPoint();
                minIntens = (short)Math.min(minIntens, dot.intensity);
                maxIntens = (short)Math.max(maxIntens, dot.intensity);
                DirectPosition2D wPoint = new DirectPosition2D(dot.x, dot.y);
                GridCoordinates2D gridCoord = gridGeometry.worldToGrid((DirectPosition)wPoint);
                int x = gridCoord.x;
                if (x < 0) {
                    x = 0;
                }
                if (x > cols - 1) {
                    x = cols - 1;
                }
                if ((y = gridCoord.y) < 0) {
                    y = 0;
                }
                if (y > rows - 1) {
                    y = rows - 1;
                }
                if (dotOnMatrixXY[x][y] == null) {
                    dotOnMatrixXY[x][y] = new ArrayList();
                }
                dotOnMatrixXY[x][y].add(dot);
                if (this.doVerbose) {
                    this.pm.worked(1);
                }
                ++readCount;
            }
            if (this.doVerbose) {
                this.pm.done();
            }
            if (readCount != recordsCount) {
                throw new RuntimeException("Didn't read all the data...");
            }
            if (this.pm.isCanceled()) {
                throw new RuntimeException(INTERRUPTED_BY_USER);
            }
            LasSourcesTable.updateMinMaxIntensity(spatialiteDb, sourceID, minIntens, maxIntens);
            ExecutorService singleThreadExecutor = Executors.newSingleThreadExecutor();
            ArrayList<LasCell> cellsList = new ArrayList<LasCell>();
            Point2D.Double pos = new Point2D.Double();
            int[] ortoValues = new int[3];
            if (this.doVerbose) {
                this.pm.beginTask("Write las data...", cols * rows);
            } else {
                this.pm.message("Write las data...");
            }
            for (int r = 0; r < rows; ++r) {
                for (int c = 0; c < cols; ++c) {
                    ArrayList dotsList = dotOnMatrixXY[c][r];
                    Coordinate coord = CoverageUtilities.coordinateFromColRow(c, r, gridGeometry);
                    Envelope env = new Envelope(coord);
                    env.expandBy(this.pCellsize / 2.0, this.pCellsize / 2.0);
                    Polygon polygon = GeometryUtilities.createPolygonFromEnvelope(env);
                    if (dotsList == null || dotsList.size() == 0) {
                        if (!this.doEmptyCells) continue;
                        LasCell lasCell = new LasCell();
                        lasCell.polygon = polygon;
                        lasCell.sourceId = sourceID;
                        lasCell.pointsCount = 0;
                        lasCell.avgElev = -9999.0;
                        lasCell.minElev = -9999.0;
                        lasCell.maxElev = -9999.0;
                        lasCell.xyzs = new byte[0];
                        lasCell.avgIntensity = (short)-999;
                        lasCell.minIntensity = (short)-999;
                        lasCell.maxIntensity = (short)-999;
                        lasCell.intensitiesClassifications = new byte[0];
                        lasCell.returns = new byte[0];
                        lasCell.minGpsTime = -9999.0;
                        lasCell.maxGpsTime = -9999.0;
                        lasCell.gpsTimes = new byte[0];
                        lasCell.colors = new byte[0];
                        cellsList.add(lasCell);
                        continue;
                    }
                    int pointCount = dotsList.size();
                    double avgElev = 0.0;
                    double minElev = Double.POSITIVE_INFINITY;
                    double maxElev = Double.NEGATIVE_INFINITY;
                    byte[] position = new byte[24 * pointCount];
                    ByteBuffer positionBuffer = ByteBuffer.wrap(position);
                    double avgIntensity = 0.0;
                    int minIntensity = 30000;
                    int maxIntensity = -1;
                    byte[] intensClass = new byte[4 * pointCount];
                    ByteBuffer intensClassBuffer = ByteBuffer.wrap(intensClass);
                    byte[] returns = new byte[4 * pointCount];
                    ByteBuffer returnsBuffer = ByteBuffer.wrap(returns);
                    double minGpsTime = Double.POSITIVE_INFINITY;
                    double maxGpsTime = Double.NEGATIVE_INFINITY;
                    byte[] gpsTimes = new byte[8 * pointCount];
                    ByteBuffer gpsTimesBuffer = ByteBuffer.wrap(gpsTimes);
                    byte[] colors = new byte[6 * pointCount];
                    ByteBuffer colorsBuffer = ByteBuffer.wrap(colors);
                    int count = 0;
                    for (LasRecord dot : dotsList) {
                        avgElev += dot.z;
                        minElev = Math.min(dot.z, minElev);
                        maxElev = Math.max(dot.z, maxElev);
                        positionBuffer.putDouble(dot.x);
                        positionBuffer.putDouble(dot.y);
                        positionBuffer.putDouble(dot.z);
                        avgIntensity += (double)dot.intensity;
                        minIntensity = (short)Math.min(dot.intensity, minIntensity);
                        maxIntensity = (short)Math.max(dot.intensity, maxIntensity);
                        intensClassBuffer.putShort(dot.intensity);
                        intensClassBuffer.putShort(dot.classification);
                        returnsBuffer.putShort(dot.returnNumber);
                        returnsBuffer.putShort(dot.numberOfReturns);
                        minGpsTime = Math.min(dot.gpsTime, minGpsTime);
                        maxGpsTime = Math.max(dot.gpsTime, maxGpsTime);
                        gpsTimesBuffer.putDouble(dot.gpsTime);
                        if (ortoGC != null) {
                            pos.setLocation(dot.x, dot.y);
                            try {
                                ortoGC.evaluate((Point2D)pos, ortoValues);
                                colorsBuffer.putShort((short)ortoValues[0]);
                                colorsBuffer.putShort((short)ortoValues[1]);
                                colorsBuffer.putShort((short)ortoValues[2]);
                            }
                            catch (PointOutsideCoverageException poce) {
                                colorsBuffer.putShort((short)255);
                                colorsBuffer.putShort((short)255);
                                colorsBuffer.putShort((short)255);
                            }
                        } else if (dot.color != null) {
                            colorsBuffer.putShort(dot.color[0]);
                            colorsBuffer.putShort(dot.color[1]);
                            colorsBuffer.putShort(dot.color[2]);
                        }
                        ++count;
                    }
                    avgElev /= (double)count;
                    avgIntensity /= (double)count;
                    LasCell lasCell = new LasCell();
                    lasCell.polygon = polygon;
                    lasCell.sourceId = sourceID;
                    lasCell.pointsCount = pointCount;
                    lasCell.avgElev = avgElev;
                    lasCell.minElev = minElev;
                    lasCell.maxElev = maxElev;
                    lasCell.xyzs = position;
                    lasCell.avgIntensity = (short)Math.round(avgIntensity);
                    lasCell.minIntensity = (short)minIntensity;
                    lasCell.maxIntensity = (short)maxIntensity;
                    lasCell.intensitiesClassifications = intensClass;
                    lasCell.returns = returns;
                    lasCell.minGpsTime = minGpsTime;
                    lasCell.maxGpsTime = maxGpsTime;
                    lasCell.gpsTimes = gpsTimes;
                    lasCell.colors = colors;
                    cellsList.add(lasCell);
                    lasCellsOnMatrixXY[c][r] = lasCell;
                    if (cellsList.size() <= 100000) continue;
                    final ArrayList<LasCell> processCells = cellsList;
                    singleThreadExecutor.execute(new Runnable(){

                        @Override
                        public void run() {
                            try {
                                LasCellsTable.insertLasCells(spatialiteDb, DatabaseLasWriter.this.srid, processCells);
                                DatabaseLasWriter.this.pm.worked(processCells.size());
                            }
                            catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    });
                    cellsList = new ArrayList();
                }
                if (!this.pm.isCanceled()) continue;
                throw new RuntimeException(INTERRUPTED_BY_USER);
            }
            if (cellsList.size() > 0) {
                final ArrayList<LasCell> processCells = cellsList;
                singleThreadExecutor.execute(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            LasCellsTable.insertLasCells(spatialiteDb, DatabaseLasWriter.this.srid, processCells);
                            if (DatabaseLasWriter.this.doVerbose) {
                                DatabaseLasWriter.this.pm.worked(processCells.size());
                            }
                        }
                        catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                });
                cellsList = new ArrayList();
            }
            try {
                singleThreadExecutor.shutdown();
                singleThreadExecutor.awaitTermination(30L, TimeUnit.DAYS);
                singleThreadExecutor.shutdownNow();
            }
            catch (InterruptedException ex) {
                ex.printStackTrace();
            }
            if (this.doVerbose) {
                this.pm.done();
            } else {
                this.pm.message("Done.");
            }
            if (this.pLevels > 0) {
                for (int level = 1; level <= this.pLevels; ++level) {
                    if (this.pm.isCanceled()) {
                        throw new RuntimeException(INTERRUPTED_BY_USER);
                    }
                    LasLevelsTable.createTable(spatialiteDb, this.srid, level, this.doAvoidIndex);
                    if (level == 1) {
                        this.insertFirstLevel(spatialiteDb, sourceID, north, south, east, west, level);
                        continue;
                    }
                    this.insertLevel(spatialiteDb, sourceID, north, south, east, west, level);
                }
            }
        }
    }

    private void insertFirstLevel(ASpatialDb spatialiteDb, long sourceID, double north, double south, double east, double west, int level) throws Exception, SQLException {
        ArrayList<LasLevel> levelsList = new ArrayList<LasLevel>();
        double levelCellsize = this.pCellsize * (double)level * (double)this.pFactor;
        double[] xRangesLevel = NumericsUtilities.range2Bins(west, east, levelCellsize, false);
        double[] yRangesLevel = NumericsUtilities.range2Bins(south, north, levelCellsize, false);
        int size = (xRangesLevel.length - 1) * (yRangesLevel.length - 1);
        if (this.doVerbose) {
            this.pm.beginTask("Creating level " + level + " with " + size + " tiles...", xRangesLevel.length - 1);
        } else {
            this.pm.message("Creating level " + level + " with " + size + " tiles...");
        }
        for (int x = 0; x < xRangesLevel.length - 1; ++x) {
            double xmin = xRangesLevel[x];
            double xmax = xRangesLevel[x + 1];
            for (int y = 0; y < yRangesLevel.length - 1; ++y) {
                double ymin = yRangesLevel[y];
                double ymax = yRangesLevel[y + 1];
                Envelope levelEnv = new Envelope(xmin, xmax, ymin, ymax);
                Polygon polygon = GeometryUtilities.createPolygonFromEnvelope(levelEnv);
                List<LasCell> lasCells = LasCellsTable.getLasCells(spatialiteDb, (Geometry)polygon, true, true, false, false, false);
                double avgElev = 0.0;
                double minElev = Double.POSITIVE_INFINITY;
                double maxElev = Double.NEGATIVE_INFINITY;
                int avgIntensity = 0;
                int minIntensity = 30000;
                int maxIntensity = -1;
                int count = 0;
                for (LasCell cell : lasCells) {
                    avgElev += cell.avgElev;
                    minElev = Math.min(cell.minElev, minElev);
                    maxElev = Math.max(cell.maxElev, maxElev);
                    avgIntensity = (short)(avgIntensity + cell.avgIntensity);
                    minIntensity = (short)Math.min(cell.minIntensity, minIntensity);
                    maxIntensity = (short)Math.max(cell.maxIntensity, maxIntensity);
                    ++count;
                }
                if (count == 0) continue;
                avgIntensity = (short)(avgIntensity / count);
                LasLevel lasLevel = new LasLevel();
                lasLevel.polygon = polygon;
                lasLevel.level = level;
                lasLevel.avgElev = avgElev /= (double)count;
                lasLevel.minElev = minElev;
                lasLevel.maxElev = maxElev;
                lasLevel.avgIntensity = (short)avgIntensity;
                lasLevel.minIntensity = (short)minIntensity;
                lasLevel.maxIntensity = (short)maxIntensity;
                lasLevel.sourceId = sourceID;
                levelsList.add(lasLevel);
                if (levelsList.size() <= 10000) continue;
                LasLevelsTable.insertLasLevels(spatialiteDb, this.srid, levelsList);
                levelsList = new ArrayList();
            }
            if (!this.doVerbose) continue;
            this.pm.worked(1);
        }
        if (levelsList.size() > 0) {
            LasLevelsTable.insertLasLevels(spatialiteDb, this.srid, levelsList);
        }
        if (this.doVerbose) {
            this.pm.done();
        } else {
            this.pm.message("Done.");
        }
    }

    private void insertLevel(ASpatialDb spatialiteDb, long sourceID, double north, double south, double east, double west, int level) throws Exception, SQLException {
        int previousLevelNum = level - 1;
        ArrayList<LasLevel> levelsList = new ArrayList<LasLevel>();
        double levelCellsize = this.pCellsize * (double)level * (double)this.pFactor;
        double[] xRangesLevel = NumericsUtilities.range2Bins(west, east, levelCellsize, false);
        double[] yRangesLevel = NumericsUtilities.range2Bins(south, north, levelCellsize, false);
        int size = (xRangesLevel.length - 1) * (yRangesLevel.length - 1);
        if (this.doVerbose) {
            this.pm.beginTask("Creating level " + level + " with " + size + " tiles...", xRangesLevel.length - 1);
        } else {
            this.pm.message("Creating level " + level + " with " + size + " tiles...");
        }
        for (int x = 0; x < xRangesLevel.length - 1; ++x) {
            double xmin = xRangesLevel[x];
            double xmax = xRangesLevel[x + 1];
            for (int y = 0; y < yRangesLevel.length - 1; ++y) {
                double ymin = yRangesLevel[y];
                double ymax = yRangesLevel[y + 1];
                Envelope levelEnv = new Envelope(xmin, xmax, ymin, ymax);
                Polygon polygon = GeometryUtilities.createPolygonFromEnvelope(levelEnv);
                List<LasLevel> lasLevels = LasLevelsTable.getLasLevels(spatialiteDb, previousLevelNum, levelEnv);
                double avgElev = 0.0;
                double minElev = Double.POSITIVE_INFINITY;
                double maxElev = Double.NEGATIVE_INFINITY;
                int avgIntensity = 0;
                int minIntensity = 30000;
                int maxIntensity = -1;
                int count = 0;
                for (LasLevel lasLevel : lasLevels) {
                    avgElev += lasLevel.avgElev;
                    minElev = Math.min(lasLevel.minElev, minElev);
                    maxElev = Math.max(lasLevel.maxElev, maxElev);
                    avgIntensity = (short)(avgIntensity + lasLevel.avgIntensity);
                    minIntensity = (short)Math.min(lasLevel.minIntensity, minIntensity);
                    maxIntensity = (short)Math.max(lasLevel.maxIntensity, maxIntensity);
                    ++count;
                }
                if (count == 0) continue;
                avgIntensity = (short)(avgIntensity / count);
                LasLevel lasLevel = new LasLevel();
                lasLevel.polygon = polygon;
                lasLevel.level = level;
                lasLevel.avgElev = avgElev /= (double)count;
                lasLevel.minElev = minElev;
                lasLevel.maxElev = maxElev;
                lasLevel.avgIntensity = (short)avgIntensity;
                lasLevel.minIntensity = (short)minIntensity;
                lasLevel.maxIntensity = (short)maxIntensity;
                lasLevel.sourceId = sourceID;
                levelsList.add(lasLevel);
                if (levelsList.size() <= 10000) continue;
                LasLevelsTable.insertLasLevels(spatialiteDb, this.srid, levelsList);
                levelsList = new ArrayList();
            }
            if (!this.doVerbose) continue;
            this.pm.worked(1);
        }
        if (levelsList.size() > 0) {
            LasLevelsTable.insertLasLevels(spatialiteDb, this.srid, levelsList);
        }
        if (this.doVerbose) {
            this.pm.done();
        } else {
            this.pm.message("Done.");
        }
    }

    @Finalize
    public void close() throws Exception {
    }

    public static void main(String[] args) throws Exception {
        EggClock egg = new EggClock("DATABASE**************** ", "");
        egg.startAndPrint(System.out);
        DatabaseLasWriter w = new DatabaseLasWriter();
        w = new DatabaseLasWriter();
        w.pDbType = EDb.SPATIALITE.name();
        w.pLevels = 4;
        w.pCellsize = 1.0;
        w.inFolder = "/media/hydrologis/Samsung_T3/UNIBZ/monticolo_tls";
        w.inDatabasePath = "/media/hydrologis/Samsung_T3/UNIBZ/monticolo_tls/monticolo2018_point_cloud_02.sqlite";
        w.process();
        egg.printTimePassedInSeconds(System.out);
    }
}

