/*******************************************************************************
 * Copyright (c) 2011 HPCC Systems. All rights reserved. This program and the accompanying materials are made available
 * under the terms of the Eclipse Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors: HPCC Systems - initial API and implementation
 ******************************************************************************/
package org.hpccsystems.ws.client.utils;

import java.lang.reflect.Array;

public final class HashCodeUtil
{

    /**
     * An initial value for a <code>hashCode</code>, to which is added contributions from fields. Using a non-zero value
     * decreases collisons of <code>hashCode</code> values.
     */
    public static final int SEED = 23;

    /**
     * booleans.
     *
     * @param aSeed
     *            the a seed
     * @param aBoolean
     *            the a boolean
     * @return the int
     */
    public static int hash(int aSeed, boolean aBoolean)
    {
        return firstTerm(aSeed) + (aBoolean ? 1 : 0);
    }

    /**
     * chars.
     *
     * @param aSeed
     *            the a seed
     * @param aChar
     *            the a char
     * @return the int
     */
    public static int hash(int aSeed, char aChar)
    {
        return firstTerm(aSeed) + aChar;
    }

    /**
     * ints.
     *
     * @param aSeed
     *            the a seed
     * @param aInt
     *            the a int
     * @return the int
     */
    public static int hash(int aSeed, int aInt)
    {
        /*
         * Implementation Note Note that byte and short are handled by this method, through implicit conversion.
         */
        return firstTerm(aSeed) + aInt;
    }

    /**
     * longs.
     *
     * @param aSeed
     *            the a seed
     * @param aLong
     *            the a long
     * @return the int
     */
    public static int hash(int aSeed, long aLong)
    {
        return firstTerm(aSeed) + (int) (aLong ^ (aLong >>> 32));
    }

    /**
     * floats.
     *
     * @param aSeed
     *            the a seed
     * @param aFloat
     *            the a float
     * @return the int
     */
    public static int hash(int aSeed, float aFloat)
    {
        return hash(aSeed, Float.floatToIntBits(aFloat));
    }

    /**
     * doubles.
     *
     * @param aSeed
     *            the a seed
     * @param aDouble
     *            the a double
     * @return the int
     */
    public static int hash(int aSeed, double aDouble)
    {
        return hash(aSeed, Double.doubleToLongBits(aDouble));
    }

    /**
     * <code>aObject</code> is a possibly-null object field, and possibly an array.
     * 
     * If <code>aObject</code> is an array, then each element may be a primitive or a possibly-null object.
     *
     * @param aSeed
     *            the a seed
     * @param aObject
     *            the a object
     * @return the int
     */
    public static int hash(int aSeed, Object aObject)
    {
        int result = aSeed;
        if (aObject == null)
        {
            result = hash(result, 0);
        }
        else if (!isArray(aObject))
        {
            result = hash(result, aObject.hashCode());
        }
        else
        {
            int length = Array.getLength(aObject);
            for (int idx = 0; idx < length; ++idx)
            {
                Object item = Array.get(aObject, idx);
                // recursive call!
                result = hash(result, item);
            }
        }
        return result;
    }

    // / PRIVATE ///
    private static final int fODD_PRIME_NUMBER = 37;

    /**
     * First term.
     *
     * @param aSeed
     *            the a seed
     * @return the int
     */
    private static int firstTerm(int aSeed)
    {
        return fODD_PRIME_NUMBER * aSeed;
    }

    /**
     * Checks if is array.
     *
     * @param aObject
     *            the a object
     * @return true, if is array
     */
    private static boolean isArray(Object aObject)
    {
        return aObject.getClass().isArray();
    }
}
