/*
 * Copyright 2019 http4s.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.http4s
package laws

import cats.syntax.all._
import cats.effect._
import cats.effect.implicits._
import cats.laws._

trait EntityCodecLaws[F[_], A] extends EntityEncoderLaws[F, A] {
  implicit def F: Effect[F]
  implicit def encoder: EntityEncoder[F, A]
  implicit def decoder: EntityDecoder[F, A]

  def entityCodecRoundTrip(a: A): IsEq[IO[Either[DecodeFailure, A]]] =
    (for {
      entity <- F.delay(encoder.toEntity(a))
      message = Request(body = entity.body, headers = encoder.headers)
      a0 <- decoder.decode(message, strict = true).value
    } yield a0).toIO <-> IO.pure(Right(a))
}

object EntityCodecLaws {
  def apply[F[_], A](implicit
      F0: Effect[F],
      entityEncoderFA: EntityEncoder[F, A],
      entityDecoderFA: EntityDecoder[F, A]): EntityCodecLaws[F, A] =
    new EntityCodecLaws[F, A] {
      val F = F0
      val encoder = entityEncoderFA
      val decoder = entityDecoderFA
    }
}
