/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.fabric.client;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import org.hyperledger.fabric.client.ChaincodeEvent;
import org.hyperledger.fabric.client.Checkpointer;

public final class FileCheckpointer
implements Checkpointer,
AutoCloseable {
    private static final String CONFIG_KEY_BLOCK = "blockNumber";
    private static final String CONFIG_KEY_TRANSACTIONID = "transactionId";
    private static final Set<OpenOption> OPEN_OPTIONS = Collections.unmodifiableSet(EnumSet.of(StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE));
    private OptionalLong blockNumber = OptionalLong.empty();
    private String transactionId;
    private final Path path;
    private final Reader fileReader;
    private final Writer fileWriter;
    private final FileChannel fileChannel;
    private final Gson gson = new Gson();

    public FileCheckpointer(Path path) throws IOException {
        this.path = path;
        this.fileChannel = FileChannel.open(path, OPEN_OPTIONS, new FileAttribute[0]);
        this.lockFile();
        CharsetEncoder utf8Encoder = StandardCharsets.UTF_8.newEncoder();
        this.fileWriter = Channels.newWriter(this.fileChannel, utf8Encoder, -1);
        CharsetDecoder utf8Decoder = StandardCharsets.UTF_8.newDecoder();
        this.fileReader = Channels.newReader(this.fileChannel, utf8Decoder, -1);
        if (this.fileChannel.size() > 0L) {
            this.load();
        }
        this.save();
    }

    private void lockFile() throws IOException {
        FileLock fileLock;
        try {
            fileLock = this.fileChannel.tryLock();
        }
        catch (OverlappingFileLockException e) {
            throw new IOException("File is already locked: " + this.path, e);
        }
        if (fileLock == null) {
            throw new IOException("Another process holds an overlapping lock for file: " + this.path);
        }
    }

    @Override
    public void checkpointBlock(long blockNumber) throws IOException {
        this.checkpointTransaction(blockNumber + 1L, null);
    }

    @Override
    public void checkpointTransaction(long blockNumber, String transactionID) throws IOException {
        this.blockNumber = OptionalLong.of(blockNumber);
        this.transactionId = transactionID;
        this.save();
    }

    @Override
    public void checkpointChaincodeEvent(ChaincodeEvent event) throws IOException {
        this.checkpointTransaction(event.getBlockNumber(), event.getTransactionId());
    }

    @Override
    public OptionalLong getBlockNumber() {
        return this.blockNumber;
    }

    @Override
    public Optional<String> getTransactionId() {
        return Optional.ofNullable(this.transactionId);
    }

    private void load() throws IOException {
        JsonObject data = this.readFile();
        if (data != null) {
            this.parseJson(data);
        }
    }

    private JsonObject readFile() throws IOException {
        this.fileChannel.position(0L);
        JsonReader jsonReader = new JsonReader(this.fileReader);
        try {
            return (JsonObject)this.gson.fromJson(jsonReader, JsonObject.class);
        }
        catch (RuntimeException e) {
            throw new IOException("Failed to parse checkpoint data from file: " + this.path, e);
        }
    }

    private void parseJson(JsonObject json) throws IOException {
        try {
            this.blockNumber = json.has(CONFIG_KEY_BLOCK) ? OptionalLong.of(json.get(CONFIG_KEY_BLOCK).getAsLong()) : OptionalLong.empty();
            this.transactionId = json.has(CONFIG_KEY_TRANSACTIONID) ? json.get(CONFIG_KEY_TRANSACTIONID).getAsString() : null;
        }
        catch (RuntimeException e) {
            throw new IOException("Bad format of checkpoint data from file: " + this.path, e);
        }
    }

    private void save() throws IOException {
        JsonObject jsonData = this.buildJson();
        this.fileChannel.position(0L);
        this.saveJson(jsonData);
        this.fileChannel.truncate(this.fileChannel.position());
    }

    private void saveJson(JsonObject json) throws IOException {
        JsonWriter jsonWriter = new JsonWriter(this.fileWriter);
        try {
            this.gson.toJson((JsonElement)json, jsonWriter);
        }
        catch (RuntimeException e) {
            throw new IOException("Failed to write checkpoint data to file: " + this.path, e);
        }
        this.fileWriter.flush();
    }

    private JsonObject buildJson() {
        JsonObject object = new JsonObject();
        this.blockNumber.ifPresent(block -> object.addProperty(CONFIG_KEY_BLOCK, (Number)block));
        if (this.transactionId != null) {
            object.addProperty(CONFIG_KEY_TRANSACTIONID, this.transactionId);
        }
        return object;
    }

    @Override
    public void close() throws IOException {
        this.fileChannel.close();
    }

    public void sync() throws IOException {
        this.fileChannel.force(false);
    }
}

