/*
 * Decompiled with CFR 0.152.
 */
package nl.nn.testtool.storage.xml;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.regex.Pattern;
import nl.nn.testtool.Report;
import nl.nn.testtool.storage.StorageException;
import nl.nn.testtool.storage.xml.Metadata;
import nl.nn.testtool.storage.xml.XmlStorage;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetadataHandler {
    private final String DEFAULT_PATH = "metadata.xml";
    private XmlStorage storage;
    private HashMap<Integer, Metadata> metadataMap;
    protected File metadataFile;
    private int lastStorageId = 1;
    private final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

    public MetadataHandler(String filePath, XmlStorage storage, boolean forceDiscover) throws IOException {
        this.storage = storage;
        if (StringUtils.isEmpty((String)filePath)) {
            this.log.warn("No filepath was given for Ladybug MetadataHandler. Continuing with default [metadata.xml]");
            filePath = "metadata.xml";
        }
        this.metadataMap = new HashMap();
        this.metadataFile = new File(filePath);
        if (this.metadataFile.exists() && !forceDiscover) {
            this.log.debug("Metadata for ladybug already exists. Reading from file [" + this.metadataFile.getName() + "] ...");
            this.readFromFile();
        } else {
            this.buildFromDirectory(storage.getReportsFolder(), true);
        }
    }

    private void buildFromDirectory(File dir, boolean searchSubDirs) throws IOException {
        if (dir == null || !dir.isDirectory()) {
            return;
        }
        this.log.debug("Building from directory " + dir.getPath());
        HashMap<Integer, HashMap> reports = new HashMap<Integer, HashMap>();
        for (File file : dir.listFiles()) {
            if (searchSubDirs && file.isDirectory()) {
                this.buildFromDirectory(file, true);
            }
            if (!file.isFile() || !file.getName().endsWith(".report.xml")) continue;
            try {
                Report report = this.storage.readReportFromFile(file);
                HashMap reportsForStorageId = reports.computeIfAbsent(report.getStorageId(), k -> new HashMap());
                reportsForStorageId.put(file, report);
            }
            catch (StorageException exception) {
                this.log.warn("Exception while reading report [" + file.getPath() + "] during build from directory.");
            }
        }
        for (Integer storageId : reports.keySet()) {
            HashMap reportsForStorageId = (HashMap)reports.get(storageId);
            boolean originalStorageIdTaken = false;
            for (File file : reportsForStorageId.keySet()) {
                Report report = (Report)reportsForStorageId.get(file);
                int targetStorageId = storageId;
                if (originalStorageIdTaken) {
                    targetStorageId = this.getNextStorageId();
                    while (reports.containsKey(targetStorageId) || this.metadataMap.containsKey(targetStorageId)) {
                        targetStorageId = this.getNextStorageId();
                    }
                    this.log.warn("Storage Id conflict on update! Changing storage id from [" + storageId + "] to [" + targetStorageId + "] for report with correlation id [" + report.getCorrelationId() + "]");
                }
                report.setStorageId(targetStorageId);
                try {
                    if (originalStorageIdTaken) {
                        this.storage.store(report, file);
                    }
                    Metadata metadata = Metadata.fromReport(report, file, this.storage.getReportsFolder());
                    this.add(metadata, false);
                }
                catch (IOException | StorageException e) {
                    this.log.error("Error while updating metadata from file [" + file.getPath() + "]", (Throwable)e);
                }
                originalStorageIdTaken = true;
            }
        }
        this.save();
    }

    private void readFromFile() throws IOException {
        if (!this.metadataFile.exists()) {
            return;
        }
        this.log.debug("Reading from file " + this.metadataFile.getPath());
        Scanner scanner = new Scanner(this.metadataFile);
        StringBuilder stringBuilder = new StringBuilder();
        while (scanner.hasNextLine()) {
            String line = scanner.nextLine();
            if (line.contains("<Metadata>")) {
                stringBuilder = new StringBuilder();
                stringBuilder.append(line);
            }
            while (stringBuilder.length() > 0 && scanner.hasNextLine()) {
                line = scanner.nextLine();
                stringBuilder.append(line);
                if (!line.contains("</Metadata>")) continue;
                Metadata m = Metadata.fromXml(stringBuilder.toString());
                this.add(m, false);
                if (m.getStorageId() > this.lastStorageId) {
                    this.lastStorageId = m.getStorageId();
                }
                stringBuilder = new StringBuilder();
            }
        }
        scanner.close();
    }

    public int getNextStorageId() {
        return ++this.lastStorageId;
    }

    public Metadata getMetadata(String correlationId) {
        if (StringUtils.isEmpty((String)correlationId)) {
            return null;
        }
        for (Integer i : this.metadataMap.keySet()) {
            Metadata m = this.metadataMap.get(i);
            if (m == null || !m.correlationId.equals(correlationId)) continue;
            return m;
        }
        return null;
    }

    public Metadata getMetadata(Integer storageId) {
        if (storageId == null) {
            return null;
        }
        return this.metadataMap.get(storageId);
    }

    public boolean contains(Integer storageId) {
        if (storageId == null) {
            return false;
        }
        return this.metadataMap.containsKey(storageId);
    }

    public void add(Metadata m) throws IOException {
        this.add(m, true);
    }

    private void add(Metadata m, boolean saveNow) throws IOException {
        if (StringUtils.isEmpty((String)m.path)) {
            m.path = "/";
        }
        this.metadataMap.put(m.storageId, m);
        if (saveNow) {
            this.save();
        }
    }

    public List<List<Object>> getAsListofObjects(int maxNumberOfRecords, List<String> metadataNames, List<String> searchValues, int metadataValueType) {
        if (metadataNames == null || metadataNames.size() == 0) {
            return new ArrayList<List<Object>>();
        }
        if (maxNumberOfRecords < 0) {
            maxNumberOfRecords = Integer.MAX_VALUE;
        }
        ArrayList<Pattern> patterns = new ArrayList<Pattern>(metadataNames.size());
        for (int i = 0; i < metadataNames.size(); ++i) {
            if (searchValues != null && StringUtils.isNotEmpty((String)searchValues.get(i))) {
                String searchValue = searchValues.get(i);
                if ("path".equalsIgnoreCase(metadataNames.get(i))) {
                    searchValue = "^" + searchValue.substring(1, searchValue.length() - 1).replaceAll("/", "\\/").replaceAll("\\*", ".*");
                }
                patterns.add(Pattern.compile(searchValue, 2));
                continue;
            }
            patterns.add(null);
        }
        ArrayList<List<Object>> result = new ArrayList<List<Object>>(this.metadataMap.size());
        Iterator<Integer> iterator = this.metadataMap.keySet().iterator();
        while (iterator.hasNext() && result.size() < maxNumberOfRecords) {
            Integer storageId = iterator.next();
            Metadata m = this.metadataMap.get(storageId);
            boolean filterPassed = true;
            for (int i = 0; i < metadataNames.size() && filterPassed; ++i) {
                filterPassed = m.fieldEquals(metadataNames.get(i), (Pattern)patterns.get(i));
            }
            if (!filterPassed) continue;
            result.add(m.toObjectList(metadataNames, metadataValueType));
        }
        return result;
    }

    public List<Integer> getStorageIds() {
        return new ArrayList<Integer>(this.metadataMap.keySet());
    }

    public int getSize() {
        return this.metadataMap.size();
    }

    private void save() throws IOException {
        if (this.metadataMap == null || this.metadataMap.size() == 0) {
            if (this.metadataFile.exists()) {
                this.metadataFile.delete();
            }
            return;
        }
        if (!this.metadataFile.exists()) {
            this.log.debug("Creating metadata file at location [" + this.metadataFile.getPath() + "]");
            this.metadataFile.getParentFile().mkdirs();
            this.metadataFile.createNewFile();
        }
        this.log.debug("Saving the metadata to file [" + this.metadataFile.getName() + "]...");
        FileWriter writer = new FileWriter(this.metadataFile, false);
        writer.append("<MetadataList>\n");
        for (Integer storageId : this.metadataMap.keySet()) {
            writer.append(this.metadataMap.get(storageId).toXml());
        }
        writer.append("</MetadataList>\n");
        writer.close();
    }

    public void delete(Report report) throws IOException {
        this.metadataMap.remove(report.getStorageId());
        this.save();
    }

    public void updateMetadata() throws IOException {
        Metadata m;
        HashMap<String, Metadata> pathMap = new HashMap<String, Metadata>(this.metadataMap.size());
        HashSet<Integer> duplicates = new HashSet<Integer>();
        for (Integer c : this.metadataMap.keySet()) {
            m = this.metadataMap.get(c);
            String path = m.path + m.name + ".report.xml";
            if (pathMap.containsKey(path)) {
                duplicates.add(m.storageId);
                duplicates.add(pathMap.get((Object)path).storageId);
            }
            pathMap.put(path, m);
        }
        for (int storageId : duplicates) {
            m = this.metadataMap.remove(storageId);
            pathMap.remove(m.path + m.name + ".report.xml");
        }
        Set<Integer> updatedIds = this.updateMetadata(this.storage.getReportsFolder(), pathMap, null);
        for (String p : pathMap.keySet()) {
            Metadata m2 = pathMap.get(p);
            if (updatedIds.contains(m2.storageId)) continue;
            this.log.debug("Deleting metadata with storage id [" + m2.storageId + "] correlation id [" + m2.correlationId + "] and path [" + m2.path + m2.name + "]");
            this.metadataMap.remove(m2.storageId);
        }
        this.save();
    }

    private Set<Integer> updateMetadata(File dir, HashMap<String, Metadata> map, Set<Integer> updatedIds) {
        if (updatedIds == null) {
            updatedIds = new HashSet<Integer>();
        }
        if (dir == null || !dir.isDirectory()) {
            return updatedIds;
        }
        for (File file : dir.listFiles()) {
            String path;
            Metadata m;
            if (file.isDirectory()) {
                this.updateMetadata(file, map, updatedIds);
                continue;
            }
            if (!file.getName().endsWith(".report.xml") || (m = map.remove(path = "/" + this.storage.getReportsFolder().toPath().relativize(file.toPath()).toString().replaceAll("\\\\", "/"))) != null && m.lastModified >= file.lastModified()) continue;
            try {
                Report oldReport;
                String oldPath;
                Report report = this.storage.readReportFromFile(file);
                if (report == null) {
                    map.put(path, m);
                    continue;
                }
                int storageId = report.getStorageId();
                Metadata oldMetadata = this.metadataMap.get(storageId);
                if (oldMetadata != null && !oldMetadata.equals(m) && StringUtils.isNotEmpty((String)(oldPath = this.storage.resolvePath(storageId))) && !oldPath.equalsIgnoreCase(file.getPath()) && (oldReport = this.storage.readReportFromFile(new File(oldPath))) != null && report.getStorageId().equals(oldReport.getStorageId())) {
                    while (this.metadataMap.containsKey(storageId)) {
                        storageId = this.getNextStorageId();
                    }
                    if (storageId >= this.lastStorageId) {
                        this.lastStorageId = storageId + 1;
                    }
                    report.setStorageId(storageId);
                    this.storage.store(report, file);
                }
                Metadata metadata = Metadata.fromReport(report, file, this.storage.getReportsFolder());
                this.add(metadata, false);
                updatedIds.add(report.getStorageId());
            }
            catch (IOException | StorageException e) {
                this.log.error("Error while updating metadata from file [" + file.getPath() + "]", (Throwable)e);
            }
        }
        return updatedIds;
    }
}

