package org.icroco.tablemodel;

import javax.swing.JTable;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;

import org.icroco.tablemodel.blinking.IBlinkTableModel;
import org.icroco.tablemodel.impl.DefaultBeanTableModelFactory;
import org.icroco.tablemodel.impl.blinking.DefaultBeanBlinkTableModelFactory;

/**
 * {@link TableModelFactory} is an helper to build table model. In fact, it's just a wrapper to another implementation because we have to
 * use some package protected method.
 * 
 * @author ouaibsky
 * 
 */
public final class TableModelFactory
{
    
    private TableModelFactory()
    {
        super();
        
    }
    
    /**
     * This is the better way to get a {@link BeanTableModel}. We make some stuff like JTable optimization for flashing, install renderer
     * wrapper. Warning set up your own renderer on your JTable before calling this method because we make some stuff concerning blinking or
     * renderer annotation.
     * 
     * After this call, {@link TableModel} is setup on the given {@link JTable}.
     * 
     * @param <B>
     * @param aTable
     * @param aClazz
     * @return
     * @throws BeanTableModelException
     */
    public final static <B> IBlinkTableModel<B> createBlinkTableModel(final JTable aTable, final Class<B> aClazz)
        throws BeanTableModelException
    {
        return DefaultBeanBlinkTableModelFactory.createBlinkTableModel(aTable, aClazz);
    }
    
    /**
     * same as {@link #createBlinkTableModel(JTable, Class)} but add a {@link TableRowSorter} and return it.
     * 
     * <br/>You can get underlying model like this:
     * <code><pre>
     * TableRowSorter<IBeanTableModel<MyClass>> rowSorter = TableModelFactory.createSortedBlinkTableModel(aJtable, MyClass.class);
     * IBeanTableModel<MyClass> model = rowSorter.getModel();
     * </pre></code>
     * 
     * @param <B>
     * @param aTable
     * @param aClazz
     * @return
     * @throws BeanTableModelException
     */
    public final static <B> TableRowSorter<IBlinkTableModel<B>> createSortedBlinkTableModel(final JTable aTable, final Class<B> aClazz)
        throws BeanTableModelException
    {
        final IBlinkTableModel<B> model = DefaultBeanBlinkTableModelFactory.createBlinkTableModel(aTable, aClazz);
        final TableRowSorter<IBlinkTableModel<B>> sorter = new TableRowSorter<IBlinkTableModel<B>>(model);
        aTable.setRowSorter(sorter);
        return sorter;
    }
    
    /**
     * create a simple TableModel based on POJO with annotation {@link ABeanTableModelProperty} and {@link ABeanTableModel}.
     * 
     * @param <B>
     * @param aClazz
     * @return
     * @throws BeanTableModelException
     */
    public final static <B> IBeanTableModel<B> createTableModel(final JTable aTable, final Class<B> aClazz) throws BeanTableModelException
    {
        return DefaultBeanTableModelFactory.createTableModel(aTable, aClazz);
    }
    
    /**
     * Same as {@link #createTableModel(Class)} but we set the model on the given {@link JTable} and we add a {@link TableRowSorter}. It
     * just a facilities. <br/>You can get underlying model like this:
     * <code><pre>
     * TableRowSorter<IBeanTableModel<MyClass>> rowSorter = TableModelFactory.createSortedTableModel(aJtable, MyClass.class);
     * IBeanTableModel<MyClass> model = rowSorter.getModel();
     * </pre></code>
     * 
     * @param <B> represents the bean class for generic.
     * @param aClazz represents same class but as Class<?>.
     * @return
     * @throws BeanTableModelException
     */
    public final static <B> TableRowSorter<IBeanTableModel<B>> createSortedTableModel(final JTable aTable, final Class<B> aClazz)
        throws BeanTableModelException
    {
        final IBeanTableModel<B> model = DefaultBeanTableModelFactory.createTableModel(aTable, aClazz);
        aTable.setModel(model);
        final TableRowSorter<IBeanTableModel<B>> sorter = new TableRowSorter<IBeanTableModel<B>>(model);
        aTable.setRowSorter(sorter);
        return sorter;
    }
    
}
